﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/os/os_Config.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_MutexTypes.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/edid.h>
#include <nn/settings/system/settings_Tv.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <cec_ManagerImpl.private.h>
#include "cec_DetailLog.h"
#include "cec_Error.h"
#include "cec_Manager.h"
#include "cec_FiniteStateMachine.h"
#include "cec_TransmitterFiniteStateMachine.h"
#include "cec_LogicalAddressFiniteStateMachine.h"
#include "cec_DebounceFiniteStateMachine.h"
#include "cec_ReceiveBuffer.h"
#include "cec_LowLevelInterface.h"
#include "cec_Utilities.h"
#include "cec_Receiver.h"
#include "cec_TopLevelFiniteStateMachine.h"
#include "cec_Opcodes.h"

namespace nn { namespace cec { namespace detail {

namespace {

    const int TopLevelParamMaximumQueuedCommandCount = 8;

    typedef struct {
        uint8_t         command[TopLevelParamMaximumQueuedCommandCount];
        uint8_t         readIndex;
        uint8_t         writeIndex;
        uint8_t         elementCount;
        os::MutexType   mutex;
    } FiniteStateMachineInputQueueType;

    const size_t ConsoleStyleMaximumStringLength = 32;
    const char ConsoleStyleStringName[] = "cec";
    const char ConsoleStyleStringKey[] = "console_style";
    const char ConsoleStyleStringConsole[] = "console";
    const char ConsoleStyleStringHandheld[] = "handheld";

    const int TopLevelParamGetPhysicalAddressMaxRetries = 20;
    const int TopLevelParamGetPhysicalAddressRetryPeriodInMilliSeconds = 50;
    const int TopLevelParamOnHtpHighHoldoffTimeInMilliseconds = 400;

    const uint64_t  TopLevelParamForceNewStartInMilliseconds = 15000;
    const uint64_t  TopLevelParamUnconditionalNewStartInMilliseconds = 30000;

    const int   TopLevelParamEdidVariationSize = 32;

    typedef struct {
        uint8_t     edidVariation[TopLevelParamEdidVariationSize];
    } EdidVariationDetailType;

    const int   TopLevelParamEdidVariationOperationIndex = 0;
    const int   TopLevelParamEdidVariationDataIndex = 1;

    const int   EdidVariationNoFeatureAbort = 0;
    const int   EdidVariationSlowOtpSequence = 1;
    const int   EdidVariationNoOsdReply = 2;

    const EdidVariationDetailType   edidVariationTable[] =
    {
        {{EdidVariationNoFeatureAbort, 'P', 'I', 'O'}},     // Pioneer TVs: don't send Feature Abort for unsupported reqeuests
        {{EdidVariationSlowOtpSequence, 'V', 'I', 'Z'}},    // Vizio TVs: add extra delay between Image View On and Active Source
        {{EdidVariationNoOsdReply, 'X', 'X', 'X'}},         // Insignia TVs: don't reply to Give OSD Name
        {{EdidVariationNoOsdReply, 'H', 'S', 'E'}}          // Hisense TVs: don't reply to Give OSD Name
    };
    const int   EdidVariationManufacturerCodeSize = 3;
    const int   EdidVariationDetailCount  = sizeof(edidVariationTable) / sizeof(EdidVariationDetailType);

    typedef struct {
        os::MutexType                       mutex;
        os::MutexType                       finiteStateMachineMutex;
        uint32_t                            flags;
        OperatingStateType                  operatingState;
        EdidVariationType                   edidVariation;
        FiniteStateMachineInputQueueType    topLevelFiniteStateMachineInputQueue;
        bool                                booleanArg;
        bool                                enablePollingRead;
        int32_t                             lastError;
        int64_t                             hpdLowTick;
        int64_t                             otpCompletedTick;
        uint8_t                             currentState;
    } TopLevelStateType;

    const uint32_t TopLevelStateFlagLockBit     = (1 << 0);
    const uint32_t TopLevelStateFlagEnabledBit  = (1 << 1);

    const uint8_t TopLevelStateInit = 0;
    const uint8_t TopLevelStateStarted = 1;
    const uint8_t TopLevelStateBestGuessStarted = 2;
    const uint8_t TopLevelStateOtpTriggered = 3;
    const uint8_t TopLevelStateOtpWithBestGuessTriggered = 4;
    const uint8_t TopLevelStateOtpTriggeredNeedsResend = 5;
    const uint8_t TopLevelStateOtpWithBestGuessTriggeredNeedsResend = 6;

    const uint8_t TopLevelInputStart = 0;
    const uint8_t TopLevelInputReset = 1;
    const uint8_t TopLevelInputHpdLow = 2;
    const uint8_t TopLevelInputHpdHigh = 3;
    const uint8_t TopLevelInputOtp = 4;
    const uint8_t TopLevelInputStandby = 5;
    const uint8_t TopLevelInputInactive = 6;
    const uint8_t TopLevelInputReceivedStandby = 7;
    const uint8_t TopLevelInputOnActiveSource = 8;
    const uint8_t TopLevelInputOnInactiveSource = 9;
    const uint8_t TopLevelInputOnTxFailure = 10;

    void    CecTopLevelGetManufacturerCode(uint8_t* pName, void* pRawData) NN_NOEXCEPT
    {
        uint16_t    manufacturerCode;
        uint8_t*    pData;

        pData = static_cast<uint8_t*>(pRawData);
        manufacturerCode = static_cast<uint16_t>(pData[9]);
        manufacturerCode += static_cast<uint16_t>(pData[8]) << 8;
        pName[3] = '\0';
        pName[2] = 0x40 + (manufacturerCode & 0x1F);
        manufacturerCode >>= 5;
        pName[1] = 0x40 + (manufacturerCode & 0x1F);
        manufacturerCode >>= 5;
        pName[0] = 0x40 + (manufacturerCode & 0x1F);
        NN_CEC_INFO("Manufacturer code %s\t%02X:%02X\n", pName, pData[8], pData[9]);
    }

    int32_t CecTopLevelApplyEdidVariation(TopLevelStateType* pState) NN_NOEXCEPT
    {
        int32_t                 rval = CecTopLevelNoPhysAddrError;
        settings::system::Edid  edidData;
        edid::Edid              edidHandle;
        uint8_t                 manufacturerName[4];

        memset(&edidData, 0, sizeof(edidData));
        NN_STATIC_ASSERT(sizeof(settings::system::Edid) == 256);
        settings::system::GetEdid(&edidData);
        if(edid::OpenEdid(&edidHandle, &edidData, sizeof(settings::system::Edid)) == edid::Error_None)
        {
            CecTopLevelGetManufacturerCode(manufacturerName, &edidData);
            if(memcmp(manufacturerName, pState->edidVariation.edidManufacturerName, 3))
            {
                int     index;

                os::LockMutex(&pState->edidVariation.mutex);
                memcpy(pState->edidVariation.edidManufacturerName, manufacturerName, 3);
                pState->edidVariation.edidManufacturerName[3] = '\0';
                pState->edidVariation.flags = 0;
                pState->edidVariation.featureAbortHandler = CecReceiverSendFeatureAbort;
                for(index = 0; index < EdidVariationDetailCount; index++)
                {
                    const EdidVariationDetailType*  pInstance;
                    bool                            nameMatches;
                    uint8_t                         opcode;

                    pInstance = &edidVariationTable[index];
                    nameMatches = !memcmp(manufacturerName,
                                          &pInstance->edidVariation[TopLevelParamEdidVariationDataIndex],
                                          EdidVariationManufacturerCodeSize);
                    opcode = pInstance->edidVariation[TopLevelParamEdidVariationOperationIndex];
                    if(nameMatches)
                    {
                        switch(opcode)
                        {
                            case EdidVariationNoFeatureAbort:
                                pState->edidVariation.featureAbortHandler = CecReceiverSendFeatureAbortNop;
                                break;
                            case EdidVariationSlowOtpSequence:
                                pState->edidVariation.flags |= EdidVariationFlagsSlowOtpSequenceBit;
                                break;
                            case EdidVariationNoOsdReply:
                                pState->edidVariation.flags |= EdidVariationFlagsNoOsdReply;
                                break;
                            default:
                                break;
                        }
                    }
                }
                NN_CEC_INFO("%s\tManufacturer %s flags %X\n", __func__, pState->edidVariation.edidManufacturerName,
                           pState->edidVariation.flags);
                os::UnlockMutex(&pState->edidVariation.mutex);
            }
            edid::CloseEdid(&edidHandle);
            rval = CecNoError;
        }
        return rval;
    }

    int32_t CecTopLevelGetCachedPhysicalAddress(uint16_t* pOutPhysicalAddress) NN_NOEXCEPT
    {
        int32_t                 rval = CecTopLevelNoPhysAddrError;
        settings::system::Edid  edidData;
        edid::Edid              edidHandle;

        memset(&edidData, 0, sizeof(edidData));
        NN_STATIC_ASSERT(sizeof(settings::system::Edid) == 256);
        settings::system::GetEdid(&edidData);
        if(edid::OpenEdid(&edidHandle, &edidData, sizeof(settings::system::Edid)) == edid::Error_None)
        {
            if(edid::GetSourcePhysicalAddress(pOutPhysicalAddress, &edidHandle))
            {
                rval = CecNoError;
            }
            edid::CloseEdid(&edidHandle);
        }
        return rval;
    }

    int32_t CecTopLevelGetPhysicalAddress(uint16_t* pOutPhysicalAddress) NN_NOEXCEPT
    {
        int32_t     rval;
        uint32_t    hpdStatus;
        uint32_t    sinkStatus;

        rval = CecLowLevelInterfaceGetConnectionStatus(&hpdStatus, &sinkStatus);
        if(rval == CecNoError)
        {
            if(hpdStatus != 0)
            {
                int retries = 0;
                uint8_t physicalAddress[2];

                rval = CecLowLevelInterfaceGetPhysicalAddress(physicalAddress);
                while((retries < TopLevelParamGetPhysicalAddressMaxRetries) &&
                      (rval != CecNoError) && (rval != CecDisplayPortInterfaceInvalidInitError))
                {
                    if(CecCancelSleepThreadMilliseconds(TopLevelParamGetPhysicalAddressRetryPeriodInMilliSeconds))
                    {
                        rval = CecManagerImplCancelledError;
                        break;
                    }
                    retries++;
                    rval = CecLowLevelInterfaceGetPhysicalAddress(physicalAddress);
                }
                if(rval == CecNoError)
                {
                    *pOutPhysicalAddress = (static_cast<uint16_t>(physicalAddress[1]) << 0) +
                                           (static_cast<uint16_t>(physicalAddress[0]) << 8);
                }
            }
            else
            {
                rval = CecTopLevelNoPhysAddrError;
            }
        }
        if(rval != CecNoError)
        {
            rval = CecTopLevelGetCachedPhysicalAddress(pOutPhysicalAddress);
            if(rval != CecNoError)
            {
                *pOutPhysicalAddress = TopLevelMadeUpPhysicalAddress;
                rval = CecTopLevelMadeUpPhysicalAddressError;
            }
            else
            {
                if((*pOutPhysicalAddress == TopLevelInvalidPhysicalAddresses[InvalidPhysicalAddressTvIndex]) ||
                   (*pOutPhysicalAddress == TopLevelInvalidPhysicalAddresses[InvalidPhysicalAddressNotAnAddressIndex]))
                {
                    *pOutPhysicalAddress = TopLevelMadeUpPhysicalAddress;
                    rval = CecTopLevelMadeUpPhysicalAddressError;
                }
                else
                {
                    rval = CecTopLevelCachedPhysicalAddressError;
                }
            }
        }
        NN_CEC_INFO("%s: physical address: 0x%04X --> %X\n", NN_CURRENT_FUNCTION_NAME, *pOutPhysicalAddress, rval);
        NN_CEC_INFO("%s: HPD %d sink %d\n", NN_CURRENT_FUNCTION_NAME, hpdStatus, sinkStatus);
        return rval;
    }

    int32_t CecTopLevelGetLogicalAddress(TopLevelStateType* pState, bool needsReset) NN_NOEXCEPT
    {
        int32_t rval;
        uint8_t logicalAddress;

        if(needsReset)
        {
            CecLogicalAddressReset();
        }
        rval = CecLogicalAddressGetLogicalAddress(&logicalAddress);
        os::LockMutex(&pState->operatingState.mutex);
        if((rval == CecNoError) && (logicalAddress != CecConstantNotALogicalAddress))
        {
            pState->operatingState.logicalAddress = logicalAddress;
        }
        else
        {
            pState->operatingState.logicalAddress = CecConstantNotALogicalAddress;
            rval = CecTopLevelNoLogAddrError;
        }
        os::UnlockMutex(&pState->operatingState.mutex);
        return rval;
    }

    void CecTopLevelClearLogicalAddress(TopLevelStateType* pState) NN_NOEXCEPT
    {
        os::LockMutex(&pState->operatingState.mutex);
        pState->operatingState.logicalAddress = CecConstantNotALogicalAddress;
        CecLowLevelInterfaceSetLogicalAddress(pState->operatingState.logicalAddress);
        os::UnlockMutex(&pState->operatingState.mutex);
        CecLogicalAddressReset();
    }

    void CecTopLevelSetPhysicalAddressValue(TopLevelStateType* pState, uint16_t physicalAddressValue, bool isValid) NN_NOEXCEPT
    {
        os::LockMutex(&pState->operatingState.mutex);
        pState->operatingState.physAddr[1] = static_cast<uint8_t>((physicalAddressValue & 0x00FF) >> 0);
        pState->operatingState.physAddr[0] = static_cast<uint8_t>((physicalAddressValue & 0xFF00) >> 8);
        pState->operatingState.opStatus &= ~CecOpStatusFlagPhysicalAddress;
        if(isValid)
        {
            pState->operatingState.opStatus |= CecOpStatusFlagPhysicalAddress;
        }
        os::UnlockMutex(&pState->operatingState.mutex);
    }

    bool CecTopLevelComparePhysicalAddress(TopLevelStateType* pState, uint16_t physicalAddressValue) NN_NOEXCEPT
    {
        bool rval;

        os::LockMutex(&pState->operatingState.mutex);
        rval = ((pState->operatingState.physAddr[1] == static_cast<uint8_t>((physicalAddressValue & 0x00FF) >> 0)) &&
                (pState->operatingState.physAddr[0] == static_cast<uint8_t>((physicalAddressValue & 0xFF00) >> 8)));
        os::UnlockMutex(&pState->operatingState.mutex);
        return(!rval);
    }

    void CecTopLevelSetLogicalAddress(TopLevelStateType* pState) NN_NOEXCEPT
    {
        int32_t rval;
        uint8_t logicalAddressValue;

        os::LockMutex(&pState->operatingState.mutex);
        logicalAddressValue = pState->operatingState.logicalAddress;
        rval = CecLowLevelInterfaceSetLogicalAddress(pState->operatingState.logicalAddress);
        os::UnlockMutex(&pState->operatingState.mutex);
        NN_CEC_INFO("%s: %d --> %X\n", NN_CURRENT_FUNCTION_NAME, logicalAddressValue, rval);
    }

    int32_t CecTopLevelBroadcastPhysicalAddress(TopLevelStateType* pState) NN_NOEXCEPT
    {
        uint8_t txBuffer[CecReceiveParamMaxMessageLength];

        os::LockMutex(&pState->operatingState.mutex);
        txBuffer[CecBufferLogicalAddressIndex] = (pState->operatingState.logicalAddress << 4) |
                                                    CecReceiveParamBroadcastLogicalAddress;
        txBuffer[CecBufferOpcodeIndex] = CecOpcodeReportPhysicalAddress;
        txBuffer[CecBufferPayloadIndex + 0] = pState->operatingState.physAddr[0];
        txBuffer[CecBufferPayloadIndex + 1] = pState->operatingState.physAddr[1];
        txBuffer[CecBufferPayloadIndex + 2] = CecDeviceTypePlaybackDevice;
        os::UnlockMutex(&pState->operatingState.mutex);
        return(CecTransmitBestEffort(txBuffer, CecBufferElementCountIsFive, false));
    }

    int32_t CecTopLevelMandatoryBroadcast(TopLevelStateType* pState, bool addDelay) NN_NOEXCEPT
    {
        bool    isSent;
        int32_t rval;

        rval = CecTopLevelBroadcastPhysicalAddress(pState);
        if(rval == CecNoError)
        {
            if(addDelay)
            {
                CecCancelSleepThreadMilliseconds(TopLevelParamOnHtpHighHoldoffTimeInMilliseconds);
            }
            rval = CecReceiverSendVendorId(&isSent);
        }
        return rval;
    }

    void CecTopLevelStopCecProtocol(TopLevelStateType* pState) NN_NOEXCEPT
    {
        if(pState->enablePollingRead)
        {
            CecLowLevelInterfaceFinalizeCecPolling();
        }
        CecReceiverStop();
        CecTopLevelClearLogicalAddress(pState);
        os::LockMutex(&pState->operatingState.mutex);
        CecLowLevelInterfaceEnable(false);
        pState->operatingState.opStatus &= ~CecOpStatusFlagCecEnabled;
        os::UnlockMutex(&pState->operatingState.mutex);
        pState->lastError = CecNoError;
    }

    int32_t CecTopLevelEnableCecInterface(TopLevelStateType* pState) NN_NOEXCEPT
    {
        int32_t rval;

        os::LockMutex(&pState->operatingState.mutex);
        rval = CecLowLevelInterfaceEnable(true);
        pState->operatingState.opStatus |= CecOpStatusFlagCecEnabled;
        os::UnlockMutex(&pState->operatingState.mutex);
        return rval;
    }

    int32_t CecTopLevelDisableCecInterface(TopLevelStateType* pState) NN_NOEXCEPT
    {
        int32_t rval;

        os::LockMutex(&pState->operatingState.mutex);
        rval = CecLowLevelInterfaceEnable(false);
        pState->operatingState.opStatus &= ~CecOpStatusFlagCecEnabled;
        os::UnlockMutex(&pState->operatingState.mutex);
        return rval;
    }

    void CecTopLevelInitializeFiniteStateMachine(TopLevelStateType* pState) NN_NOEXCEPT
    {
        pState->topLevelFiniteStateMachineInputQueue.readIndex = 0;
        pState->topLevelFiniteStateMachineInputQueue.writeIndex = 0;
        pState->topLevelFiniteStateMachineInputQueue.elementCount = 0;
        os::InitializeMutex(&pState->topLevelFiniteStateMachineInputQueue.mutex, false, 0);
        os::InitializeMutex(&pState->finiteStateMachineMutex, false, 0);
        pState->currentState = TopLevelStateInit;
    }

    void CecTopLevelFinalizeFiniteStateMachine(TopLevelStateType* pState) NN_NOEXCEPT
    {
        os::LockMutex(&pState->finiteStateMachineMutex);
        os::UnlockMutex(&pState->finiteStateMachineMutex);
        os::FinalizeMutex(&pState->finiteStateMachineMutex);
        pState->currentState = TopLevelStateInit;
        os::LockMutex(&pState->topLevelFiniteStateMachineInputQueue.mutex);
        os::UnlockMutex(&pState->topLevelFiniteStateMachineInputQueue.mutex);
        os::FinalizeMutex(&pState->topLevelFiniteStateMachineInputQueue.mutex);
    }

    void CecTopLevelActiveSourceProcessing(TopLevelStateType* pState,
                                           uint8_t nextStateValue, uint8_t altNextStateValue) NN_NOEXCEPT
    {
        pState->lastError = CecReceiverSendActiveSource();
        if((pState->lastError == CecNoError) || (pState->lastError == CecReceiverHpdChangedError))
        {
            os::LockMutex(&pState->operatingState.mutex);
            pState->otpCompletedTick = os::GetSystemTick().GetInt64Value();
            pState->operatingState.powerState = CecPowerStatePowerOn;
            pState->operatingState.opStatus |= CecOpStatusFlagActiveSource;
            os::UnlockMutex(&pState->operatingState.mutex);
            if(pState->lastError == CecNoError)
            {
                pState->currentState = nextStateValue;
            }
            else
            {
                pState->currentState = altNextStateValue;
            }
            pState->lastError = CecNoError;
        }
    }

    void CecTopLevelInitStateHandler(TopLevelStateType* pState, uint8_t inputValue) NN_NOEXCEPT
    {
        switch(inputValue)
        {
            case TopLevelInputHpdLow:
                {
                    uint32_t    hpdStatus;
                    uint32_t    sinkStatus;

                    pState->lastError = CecLowLevelInterfaceGetConnectionStatus(&hpdStatus, &sinkStatus);
                    pState->hpdLowTick = os::GetSystemTick().GetInt64Value();
                    if((pState->lastError != CecNoError) ||
                       ((pState->lastError == CecNoError) && (hpdStatus == 0) && (sinkStatus == 0)))
                    {
                        break;
                    }
                }
            case TopLevelInputStart:
            case TopLevelInputHpdHigh:
                {
                    uint16_t    physicalAddressValue;
                    int32_t     rval;

                    pState->lastError = CecTopLevelEnableCecInterface(pState);
                    if(pState->lastError != CecNoError)
                    {
                        break;
                    }
                    CecTopLevelClearLogicalAddress(pState);
                    pState->lastError = CecTopLevelGetLogicalAddress(pState, false);
                    if(pState->lastError != CecNoError)
                    {
                        CecTopLevelDisableCecInterface(pState);
                        break;
                    }
                    rval = CecTopLevelGetPhysicalAddress(&physicalAddressValue);
                    CecTopLevelApplyEdidVariation(pState);
                    CecTopLevelSetPhysicalAddressValue(pState, physicalAddressValue, (rval == CecNoError));
                    CecTopLevelSetLogicalAddress(pState);   /* can fail if HPD is low.  Necessary for Baldr. */
                    CecReceiveBufferPurgeBuffer(false);
                    if(pState->enablePollingRead)
                    {
                        CecLowLevelInterfaceInitializeCecPolling();
                    }
                    CecReceiverStart(&pState->operatingState, &pState->edidVariation);
                    pState->lastError = CecTopLevelMandatoryBroadcast(pState, false);
                    if(pState->lastError != CecNoError)
                    {
                        CecTopLevelStopCecProtocol(pState);
                        break;
                    }
                    if(rval == CecNoError)
                    {
                        pState->currentState = TopLevelStateStarted;
                    }
                    else
                    {
                        pState->currentState = TopLevelStateBestGuessStarted;
                    }
                    server::CecManagerImplOnEvent(server::CecEventStarted, 0, nullptr);
                    pState->hpdLowTick = os::GetSystemTick().GetInt64Value();
                }
                break;
            case TopLevelInputOtp:
            case TopLevelInputStandby:
            case TopLevelInputInactive:
            case TopLevelInputReceivedStandby:
                pState->lastError = CecTopLevelInvalidStateError;
                break;
            case TopLevelInputReset:
            case TopLevelInputOnActiveSource:
            case TopLevelInputOnInactiveSource:
                pState->lastError = CecNoError;
            default:
                break;
        }
    }

    void CecTopLevelStartedStateHandler(TopLevelStateType* pState, uint8_t inputValue) NN_NOEXCEPT
    {
        switch(inputValue)
        {
            case TopLevelInputReset:
                server::CecManagerImplOnEvent(server::CecEventSuspending, 0, nullptr);
                CecTopLevelStopCecProtocol(pState);
                pState->currentState = TopLevelStateInit;
                break;
            case TopLevelInputHpdLow:
                pState->hpdLowTick = os::GetSystemTick().GetInt64Value();
                break;
            case TopLevelInputHpdHigh:
                {
                    int64_t     currentTick;
                    uint64_t    deltaInMilliseconds;
                    uint16_t    physicalAddressValue;
                    int32_t     rval;

                    currentTick = os::GetSystemTick().GetInt64Value();
                    deltaInMilliseconds = os::ConvertToTimeSpan(os::Tick(currentTick - pState->hpdLowTick)).GetMilliSeconds();
                    pState->hpdLowTick = currentTick;
                    CecTopLevelEnableCecInterface(pState);
                    rval = CecTopLevelGetPhysicalAddress(&physicalAddressValue);
                    CecTopLevelApplyEdidVariation(pState);
                    if(rval != CecNoError)
                    {
                        break;
                    }
                    if(!(CecTopLevelComparePhysicalAddress(pState, physicalAddressValue)) &&
                       (deltaInMilliseconds <= TopLevelParamForceNewStartInMilliseconds))
                    {
                        break;
                    }
                    pState->lastError = CecTopLevelGetLogicalAddress(pState, true);
                    if(pState->lastError != CecNoError)
                    {
                        break;
                    }
                    CecTopLevelSetPhysicalAddressValue(pState, physicalAddressValue, true);
                    CecTopLevelSetLogicalAddress(pState);
                    CecTopLevelMandatoryBroadcast(pState, true);
                    server::CecManagerImplOnEvent(server::CecEventStarted, 0, nullptr);
                }
                break;
            case TopLevelInputOtp:
                CecTopLevelActiveSourceProcessing(pState,
                                                  TopLevelStateOtpTriggered, TopLevelStateOtpTriggeredNeedsResend);
                break;
            case TopLevelInputStandby:
                pState->lastError = CecReceiverSendGoStandby(pState->booleanArg);
                if(pState->lastError == CecNoError)
                {
                    os::LockMutex(&pState->operatingState.mutex);
                    pState->operatingState.opStatus &= ~CecOpStatusFlagActiveSource;
                    os::UnlockMutex(&pState->operatingState.mutex);
                }
                break;
            case TopLevelInputInactive:
                pState->lastError = CecReceiverSendInactiveSource();
                break;
            case TopLevelInputOnActiveSource:
                pState->currentState = TopLevelStateOtpTriggered;
            case TopLevelInputStart:
            case TopLevelInputOnInactiveSource:
                pState->lastError = CecNoError;
            case TopLevelInputReceivedStandby:
            default:
                break;
        }
    }

    void CecTopLevelBestGuessStartedStateHandler(TopLevelStateType* pState, uint8_t inputValue) NN_NOEXCEPT
    {
        switch(inputValue)
        {
            case TopLevelInputReset:
                server::CecManagerImplOnEvent(server::CecEventSuspending, 0, nullptr);
                CecTopLevelStopCecProtocol(pState);
                pState->currentState = TopLevelStateInit;
                break;
            case TopLevelInputOnActiveSource:
                pState->currentState = TopLevelStateOtpWithBestGuessTriggered;
            case TopLevelInputStart:
                pState->lastError = CecNoError;
                break;
            case TopLevelInputHpdLow:
                pState->hpdLowTick = os::GetSystemTick().GetInt64Value();
                break;
            case TopLevelInputHpdHigh:
                {
                    int64_t     currentTick;
                    uint64_t    deltaInMilliseconds;
                    uint16_t    physicalAddressValue;
                    bool        needsRestart = false;

                    currentTick = os::GetSystemTick().GetInt64Value();
                    deltaInMilliseconds = os::ConvertToTimeSpan(os::Tick(currentTick - pState->hpdLowTick)).GetMilliSeconds();
                    pState->hpdLowTick = currentTick;
                    CecTopLevelEnableCecInterface(pState);
                    CecTopLevelSetLogicalAddress(pState);
                    pState->lastError = CecTopLevelGetPhysicalAddress(&physicalAddressValue);
                    CecTopLevelApplyEdidVariation(pState);
                    if(pState->lastError == CecNoError)
                    {
                        if(CecTopLevelComparePhysicalAddress(pState, physicalAddressValue))
                        {
                            CecTopLevelSetPhysicalAddressValue(pState, physicalAddressValue, true);
                            if(deltaInMilliseconds <= TopLevelParamForceNewStartInMilliseconds)
                            {
                                os::SleepThread(TimeSpan::FromMilliSeconds(TopLevelParamOnHtpHighHoldoffTimeInMilliseconds));
                                CecTopLevelBroadcastPhysicalAddress(pState);
                            }
                            else
                            {
                                needsRestart = true;
                            }
                        }
                        else if(deltaInMilliseconds > TopLevelParamForceNewStartInMilliseconds)
                        {
                            needsRestart = true;
                        }
                        pState->currentState = TopLevelStateStarted;
                    }
                    if(needsRestart)
                    {
                        pState->lastError = CecTopLevelGetLogicalAddress(pState, true);
                        if(pState->lastError == CecNoError)
                        {
                            CecTopLevelSetLogicalAddress(pState);
                            CecTopLevelMandatoryBroadcast(pState, true);
                            server::CecManagerImplOnEvent(server::CecEventStarted, 0, nullptr);
                        }
                    }
                }
                break;
            case TopLevelInputOtp:
                CecTopLevelActiveSourceProcessing(pState,
                                                  TopLevelStateOtpWithBestGuessTriggered,
                                                  TopLevelStateOtpWithBestGuessTriggeredNeedsResend);
                break;
            case TopLevelInputStandby:
                pState->lastError = CecReceiverSendGoStandby(pState->booleanArg);
                if(pState->lastError == CecNoError)
                {
                    os::LockMutex(&pState->operatingState.mutex);
                    pState->operatingState.opStatus &= ~CecOpStatusFlagActiveSource;
                    os::UnlockMutex(&pState->operatingState.mutex);
                }
                break;
            case TopLevelInputInactive:
                pState->lastError = CecReceiverSendInactiveSource();
                break;
            case TopLevelInputReceivedStandby:
            case TopLevelInputOnInactiveSource:
                pState->lastError = CecNoError;
            default:
                break;
        }
    }

    void CecTopLevelOtpTriggeredStateHandler(TopLevelStateType* pState, uint8_t inputValue) NN_NOEXCEPT
    {
        switch(inputValue)
        {
            case TopLevelInputReset:
                server::CecManagerImplOnEvent(server::CecEventSuspending, 0, nullptr);
                CecTopLevelStopCecProtocol(pState);
                pState->currentState = TopLevelStateInit;
                break;
            case TopLevelInputOnTxFailure:
                pState->currentState = TopLevelStateOtpTriggeredNeedsResend;
                break;
            case TopLevelInputHpdLow:
                pState->hpdLowTick = os::GetSystemTick().GetInt64Value();
                break;
            case TopLevelInputHpdHigh:
                {
                    int64_t     currentTick;
                    uint64_t    deltaInMilliseconds;
                    uint16_t    physicalAddressValue;
                    bool        needsActiveSource = false;

                    currentTick = os::GetSystemTick().GetInt64Value();
                    deltaInMilliseconds = os::ConvertToTimeSpan(os::Tick(currentTick - pState->hpdLowTick)).GetMilliSeconds();
                    pState->hpdLowTick = currentTick;
                    CecTopLevelEnableCecInterface(pState);
                    pState->lastError = CecTopLevelGetPhysicalAddress(&physicalAddressValue);
                    CecTopLevelApplyEdidVariation(pState);
                    if(pState->lastError != CecNoError)
                    {
                        break;
                    }
                    if(pState->currentState == TopLevelStateOtpTriggered)
                    {
                        if(CecTopLevelComparePhysicalAddress(pState, physicalAddressValue))
                        {
                            needsActiveSource = true;
                        }
                        else if(deltaInMilliseconds <= TopLevelParamUnconditionalNewStartInMilliseconds)
                        {
                            break;
                        }
                        pState->lastError = CecTopLevelGetLogicalAddress(pState, true);
                        if(pState->lastError == CecNoError)
                        {
                            CecTopLevelSetPhysicalAddressValue(pState, physicalAddressValue, true);
                            CecTopLevelSetLogicalAddress(pState);
                            CecTopLevelMandatoryBroadcast(pState, true);
                            server::CecManagerImplOnEvent(server::CecEventStarted, 0, nullptr);
                        }
                    }
                    else
                    {
                        needsActiveSource = true;
                        if((CecTopLevelComparePhysicalAddress(pState, physicalAddressValue)) ||
                           (deltaInMilliseconds > TopLevelParamUnconditionalNewStartInMilliseconds))
                        {
                            pState->lastError = CecTopLevelGetLogicalAddress(pState, true);
                            if(pState->lastError == CecNoError)
                            {
                                CecTopLevelSetPhysicalAddressValue(pState, physicalAddressValue, true);
                                CecTopLevelSetLogicalAddress(pState);
                                CecTopLevelMandatoryBroadcast(pState, true);
                                server::CecManagerImplOnEvent(server::CecEventStarted, 0, nullptr);
                            }
                        }
                    }
                    if(needsActiveSource)
                    {
                        os::SleepThread(TimeSpan::FromMilliSeconds(TopLevelParamOnHtpHighHoldoffTimeInMilliseconds));
                        pState->lastError = CecReceiverSendActiveSourceCommand(true);
                        if(pState->lastError != CecNoError)
                        {
                            pState->currentState = TopLevelStateStarted;
                        }
                        else
                        {
                            pState->currentState = TopLevelStateOtpTriggered;
                        }
                    }
                }
                break;
            case TopLevelInputStandby:
                pState->lastError = CecReceiverSendGoStandby(pState->booleanArg);
                if(pState->lastError == CecNoError)
                {
                    os::LockMutex(&pState->operatingState.mutex);
                    pState->operatingState.opStatus &= ~CecOpStatusFlagActiveSource;
                    os::UnlockMutex(&pState->operatingState.mutex);
                    pState->currentState = TopLevelStateStarted;
                }
                break;
            case TopLevelInputOtp:
                break;
            case TopLevelInputInactive:
                pState->lastError = CecReceiverSendInactiveSource();
                if(pState->lastError == CecNoError)
                {
                    pState->currentState = TopLevelStateStarted;
                }
                break;
            case TopLevelInputReceivedStandby:
            case TopLevelInputOnInactiveSource:
                pState->currentState = TopLevelStateStarted;
            case TopLevelInputStart:
            case TopLevelInputOnActiveSource:
                pState->lastError = CecNoError;
            default:
                break;
        }
    } // NOLINT(impl/function_size)

    void CecTopLevelOtpWithBestGuessTriggeredStateHandler(TopLevelStateType* pState, uint8_t inputValue) NN_NOEXCEPT
    {
        switch(inputValue)
        {
            case TopLevelInputReset:
                server::CecManagerImplOnEvent(server::CecEventSuspending, 0, nullptr);
                CecTopLevelStopCecProtocol(pState);
                pState->currentState = TopLevelStateInit;
                break;
            case TopLevelInputOnTxFailure:
                pState->currentState = TopLevelStateOtpWithBestGuessTriggeredNeedsResend;
                break;
            case TopLevelInputHpdLow:
                pState->hpdLowTick = os::GetSystemTick().GetInt64Value();
                break;
            case TopLevelInputHpdHigh:
                {
                    uint16_t    physicalAddressValue;

                    pState->hpdLowTick = os::GetSystemTick().GetInt64Value();
                    CecTopLevelEnableCecInterface(pState);
                    CecTopLevelSetLogicalAddress(pState);
                    pState->lastError = CecTopLevelGetPhysicalAddress(&physicalAddressValue);
                    CecTopLevelApplyEdidVariation(pState);
                    if(pState->lastError == CecNoError)
                    {
                        if(CecTopLevelComparePhysicalAddress(pState, physicalAddressValue))
                        {
                            CecTopLevelSetPhysicalAddressValue(pState, physicalAddressValue, true);
                            os::SleepThread(TimeSpan::FromMilliSeconds(TopLevelParamOnHtpHighHoldoffTimeInMilliseconds));
                            CecTopLevelBroadcastPhysicalAddress(pState);
                            os::SleepThread(TimeSpan::FromMilliSeconds(TopLevelParamOnHtpHighHoldoffTimeInMilliseconds));
                        }
                        else if(pState->currentState == TopLevelStateOtpWithBestGuessTriggered)
                        {
                            pState->currentState = TopLevelStateOtpTriggered;
                        }
                        if(pState->currentState != TopLevelStateOtpTriggered)
                        {
                            pState->lastError = CecReceiverSendActiveSourceCommand(true);
                            pState->currentState = (pState->lastError == CecNoError) ?
                                                   TopLevelStateOtpTriggered : TopLevelStateStarted;
                        }
                    }
                }
                break;
            case TopLevelInputStandby:
                pState->lastError = CecReceiverSendGoStandby(pState->booleanArg);
                if(pState->lastError == CecNoError)
                {
                    os::LockMutex(&pState->operatingState.mutex);
                    pState->operatingState.opStatus &= ~CecOpStatusFlagActiveSource;
                    os::UnlockMutex(&pState->operatingState.mutex);
                    pState->currentState = TopLevelStateBestGuessStarted;
                }
                break;
            case TopLevelInputOtp:
                break;
            case TopLevelInputInactive:
                pState->lastError = CecReceiverSendInactiveSource();
                if(pState->lastError == CecNoError)
                {
                    pState->currentState = TopLevelStateBestGuessStarted;
                }
                break;
            case TopLevelInputReceivedStandby:
            case TopLevelInputOnInactiveSource:
                pState->currentState = TopLevelStateBestGuessStarted;
            case TopLevelInputStart:
            case TopLevelInputOnActiveSource:
                pState->lastError = CecNoError;
            default:
                break;
        }
    }

    void CecTopLevelStepFiniteStateMachineLocked(TopLevelStateType* pState, uint8_t inputValue) NN_NOEXCEPT
    {
        NN_CEC_INFO("%s On Entry: state %d input value %d\n", NN_CURRENT_FUNCTION_NAME, pState->currentState, inputValue);
        pState->lastError = CecNoError;
        switch(pState->currentState)
        {
            case TopLevelStateInit:
                CecTopLevelInitStateHandler(pState, inputValue);
                break;
            case TopLevelStateStarted:
                CecTopLevelStartedStateHandler(pState, inputValue);
                break;
            case TopLevelStateBestGuessStarted:
                CecTopLevelBestGuessStartedStateHandler(pState, inputValue);
                break;
            case TopLevelStateOtpTriggered:
                CecTopLevelOtpTriggeredStateHandler(pState, inputValue);
                break;
            case TopLevelStateOtpWithBestGuessTriggered:
                CecTopLevelOtpWithBestGuessTriggeredStateHandler(pState, inputValue);
                break;
            case TopLevelStateOtpTriggeredNeedsResend:
                CecTopLevelOtpTriggeredStateHandler(pState, inputValue);
                break;
            case TopLevelStateOtpWithBestGuessTriggeredNeedsResend:
                CecTopLevelOtpWithBestGuessTriggeredStateHandler(pState, inputValue);
                break;
            default:
                break;
        }
        NN_CEC_INFO("%s On Exit: state %d last error %X\n", NN_CURRENT_FUNCTION_NAME, pState->currentState, pState->lastError);
    }

    void CecTopLevelStepFiniteStateMachine(TopLevelStateType* pState, uint8_t inputValue) NN_NOEXCEPT
    {
        FiniteStateMachineInputQueueType*   pInputQueue;

        pInputQueue = &pState->topLevelFiniteStateMachineInputQueue;
        os::LockMutex(&pInputQueue->mutex);
        if(pInputQueue->elementCount < TopLevelParamMaximumQueuedCommandCount)
        {
            pInputQueue->command[pInputQueue->writeIndex] = inputValue;
            pInputQueue->writeIndex += 1;
            if(pInputQueue->writeIndex >= TopLevelParamMaximumQueuedCommandCount)
            {
                pInputQueue->writeIndex -= TopLevelParamMaximumQueuedCommandCount;
            }
            pInputQueue->elementCount += 1;
        }
        os::UnlockMutex(&pInputQueue->mutex);
        if(os::TryLockMutex(&pState->finiteStateMachineMutex))
        {
            while(true)
            {
                uint8_t fsmInputValue;

                os::LockMutex(&pInputQueue->mutex);
                if(pInputQueue->elementCount == 0)
                {
                    os::UnlockMutex(&pState->finiteStateMachineMutex);
                    os::UnlockMutex(&pInputQueue->mutex);
                    break;
                }
                fsmInputValue = pInputQueue->command[pInputQueue->readIndex];
                pInputQueue->readIndex += 1;
                if(pInputQueue->readIndex >= TopLevelParamMaximumQueuedCommandCount)
                {
                    pInputQueue->readIndex -= TopLevelParamMaximumQueuedCommandCount;
                }
                pInputQueue->elementCount -= 1;
                os::UnlockMutex(&pInputQueue->mutex);
                CecTopLevelStepFiniteStateMachineLocked(pState, fsmInputValue);
            }
        }
    }

    bool CecTopLevelIsInInitialState(TopLevelStateType* pState)
    {
        bool    rval;

        os::LockMutex(&pState->finiteStateMachineMutex);
        rval = (pState->currentState == TopLevelStateInit);
        os::UnlockMutex(&pState->finiteStateMachineMutex);
        return rval;
    }

    TopLevelStateType   s_TopLevelState;

    bool CecTopLevelIsInitializationOk() NN_NOEXCEPT
    {
        return(!CecTopLevelIsInInitialState(&s_TopLevelState));
    }

    void    CecTopLevelOnApiEntry() NN_NOEXCEPT
    {
        os::LockMutex(&s_TopLevelState.mutex);
        s_TopLevelState.flags |= TopLevelStateFlagLockBit;
    }
    int32_t CecTopLevelOnApiExit(int32_t returnValue) NN_NOEXCEPT
    {
        s_TopLevelState.flags &= ~TopLevelStateFlagLockBit;
        os::UnlockMutex(&s_TopLevelState.mutex);
        return((CecCancelClearCancelState()) ? CecManagerImplCancelledError : returnValue);
    }

    int32_t CecTopLevelProcessSystemSettings() NN_NOEXCEPT
    {
        int32_t rval = CecNoError;
        char consoleStyleStringValue[ConsoleStyleMaximumStringLength];
        size_t consoleStyleStringLength;

        s_TopLevelState.enablePollingRead = true;
        consoleStyleStringLength = settings::fwdbg::GetSettingsItemValue(consoleStyleStringValue,
                                                                         ConsoleStyleMaximumStringLength,
                                                                         ConsoleStyleStringName,
                                                                         ConsoleStyleStringKey);
        if((consoleStyleStringLength > 0) && (consoleStyleStringLength <= ConsoleStyleMaximumStringLength))
        {
            if(::std::strncmp(consoleStyleStringValue, ConsoleStyleStringConsole, sizeof(ConsoleStyleStringConsole)) == 0)
            {
                s_TopLevelState.enablePollingRead = false;
            }
            else if(::std::strncmp(consoleStyleStringValue, ConsoleStyleStringHandheld, sizeof(ConsoleStyleStringHandheld)) == 0)
            {
                rval = CecTopLevelNotSupportedError;
            }
            NN_CEC_INFO("%s: Console Style %s\tPolling Read %d\n",
                        NN_CURRENT_FUNCTION_NAME, consoleStyleStringValue, s_TopLevelState.enablePollingRead);
        }
        return rval;
    }
}

int32_t CecTopLevelInit() NN_NOEXCEPT
{
    int32_t rval;

    NN_SDK_ASSERT(!(s_TopLevelState.flags & TopLevelStateFlagEnabledBit));
    rval = CecTopLevelProcessSystemSettings();
    if(rval == CecNoError)
    {
        rval = CecReceiverInit();
        if(rval == CecNoError)
        {
            CecTopLevelInitializeFiniteStateMachine(&s_TopLevelState);
            s_TopLevelState.hpdLowTick = os::GetSystemTick().GetInt64Value();
            os::InitializeMutex(&s_TopLevelState.mutex, false, 0);
            os::InitializeMutex(&s_TopLevelState.operatingState.mutex, false, 0);
            CecTopLevelSetPhysicalAddressValue(&s_TopLevelState, TopLevelInvalidPhysicalAddresses[InvalidPhysicalAddressNotAnAddressIndex], false);
            s_TopLevelState.operatingState.powerState = CecPowerStatePowerOn;
            s_TopLevelState.operatingState.logicalAddress = CecConstantNotALogicalAddress;
            s_TopLevelState.operatingState.opStatus = CecOpStatusFlagMenuActiveState;
            os::InitializeMutex(&s_TopLevelState.edidVariation.mutex, false, 0);
            s_TopLevelState.edidVariation.featureAbortHandler = CecReceiverSendFeatureAbort;
            s_TopLevelState.edidVariation.flags = 0;
            memset(s_TopLevelState.edidVariation.edidManufacturerName, 0, 4);
            s_TopLevelState.lastError = CecNoError;
            s_TopLevelState.flags = 0;
            s_TopLevelState.flags |= TopLevelStateFlagEnabledBit;
        }
    }
    return rval;
}

void    CecTopLevelShutdown() NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_TopLevelState.flags & TopLevelStateFlagEnabledBit);
    os::LockMutex(&s_TopLevelState.mutex);
    os::UnlockMutex(&s_TopLevelState.mutex);
    os::FinalizeMutex(&s_TopLevelState.mutex);
    os::LockMutex(&s_TopLevelState.operatingState.mutex);
    os::UnlockMutex(&s_TopLevelState.operatingState.mutex);
    os::FinalizeMutex(&s_TopLevelState.operatingState.mutex);
    os::LockMutex(&s_TopLevelState.edidVariation.mutex);
    os::UnlockMutex(&s_TopLevelState.edidVariation.mutex);
    os::FinalizeMutex(&s_TopLevelState.edidVariation.mutex);
    s_TopLevelState.flags &= ~TopLevelStateFlagEnabledBit;
    CecReceiverShutdown();
    CecTopLevelFinalizeFiniteStateMachine(&s_TopLevelState);
}

int32_t CecTopLevelPingTv(bool* pOutIsPingSuccessful) NN_NOEXCEPT
{
    int32_t                 rval;
    TransmitReturnValueType transmitReturnValue;
    uint8_t                 pollValue;
    uint8_t                 sourceAddress;

    if(CecDebounceIsInDebouncePeriod())
    {
        rval = CecTopLevelBusyError;
        *pOutIsPingSuccessful = false;
    }
    else
    {
        NN_SDK_ASSERT(pOutIsPingSuccessful != nullptr);
        os::LockMutex(&s_TopLevelState.operatingState.mutex);
        sourceAddress = s_TopLevelState.operatingState.logicalAddress;
        os::UnlockMutex(&s_TopLevelState.operatingState.mutex);
        if(sourceAddress == CecReceiveParamUnassignedLogicalAddress)
        {
            sourceAddress = CecReceiveParamFreeUseLogicalAddress;
        }
        pollValue = (sourceAddress << 4) | CecReceiveParamTvLogicalAddress;
        CecTopLevelOnApiEntry();
        os::LockMutex(&s_TopLevelState.operatingState.mutex);
        if(!(s_TopLevelState.operatingState.opStatus & CecOpStatusFlagCecEnabled))
        {
            CecLowLevelInterfaceEnable(true);
        }
        CecTransmitPollingRequest(&transmitReturnValue, pollValue);
        if(!(s_TopLevelState.operatingState.opStatus & CecOpStatusFlagCecEnabled))
        {
            CecLowLevelInterfaceEnable(false);
        }
        os::UnlockMutex(&s_TopLevelState.operatingState.mutex);
        if(transmitReturnValue.error == CecNoError)
        {
            if(transmitReturnValue.ackNak == TransmitAckNakValueAck)
            {
                *pOutIsPingSuccessful = true;
            }
            else
            {
                *pOutIsPingSuccessful = false;
            }
            rval = CecNoError;
        }
        else
        {
            *pOutIsPingSuccessful = false;
            rval = transmitReturnValue.error;
        }
        rval = CecTopLevelOnApiExit(rval);
    }
    return rval;
}

uint8_t CecTopLevelGetPowerStatus() NN_NOEXCEPT
{
    uint8_t rval;

    NN_SDK_ASSERT(s_TopLevelState.flags & TopLevelStateFlagEnabledBit);
    os::LockMutex(&s_TopLevelState.operatingState.mutex);
    rval = s_TopLevelState.operatingState.powerState;
    os::UnlockMutex(&s_TopLevelState.operatingState.mutex);
    return rval;
}

void CecTopLevelUpdatePowerStatus(uint8_t curPowerState) NN_NOEXCEPT
{
    NN_SDK_ASSERT(s_TopLevelState.flags & TopLevelStateFlagEnabledBit);
    os::LockMutex(&s_TopLevelState.operatingState.mutex);
    s_TopLevelState.operatingState.powerState = curPowerState;
    os::UnlockMutex(&s_TopLevelState.operatingState.mutex);
    if((curPowerState == CecPowerStateStandby) || (curPowerState == CecPowerStateToStandby))
    {
        CecTopLevelStepFiniteStateMachine(&s_TopLevelState, TopLevelInputOnInactiveSource);
    }
}

int32_t CecTopLevelStart() NN_NOEXCEPT
{
    int32_t rval;

    rval = CecNoError;
    if(s_TopLevelState.flags & TopLevelStateFlagEnabledBit)
    {
        CecTopLevelOnApiEntry();

        rval = CecLowLevelInterfaceStart();
        NN_SDK_ASSERT(rval == CecNoError);

        CecTopLevelStepFiniteStateMachine(&s_TopLevelState, TopLevelInputStart);
        rval = s_TopLevelState.lastError;
        if(rval == CecNoError)
        {
            os::LockMutex(&s_TopLevelState.operatingState.mutex);
            s_TopLevelState.operatingState.powerState = CecPowerStatePowerOn;
            os::UnlockMutex(&s_TopLevelState.operatingState.mutex);
        }

        rval = CecTopLevelOnApiExit(rval);
        if(rval != CecNoError)
        {
            CecTopLevelOnApiEntry();
            CecLowLevelInterfacePrepareForStop();
            CecTopLevelStepFiniteStateMachine(&s_TopLevelState, TopLevelInputReset);
            os::LockMutex(&s_TopLevelState.finiteStateMachineMutex);
            os::UnlockMutex(&s_TopLevelState.finiteStateMachineMutex);
            CecLowLevelInterfaceStop();
            CecDebounceOnReset();
            CecTopLevelOnApiExit(CecNoError);
        }
    }
    return rval;
}

void CecTopLevelOnHpd(bool isHigh) NN_NOEXCEPT
{
    uint8_t inputValue;

    if(s_TopLevelState.flags & TopLevelStateFlagEnabledBit)
    {
        inputValue = (isHigh) ? TopLevelInputHpdHigh : TopLevelInputHpdLow;
        CecTopLevelStepFiniteStateMachine(&s_TopLevelState, inputValue);
    }
}

int32_t CecTopLevelPerformActiveSource(bool setActiveSource) NN_NOEXCEPT
{
    int32_t rval;

    rval = CecNoError;
    if(s_TopLevelState.flags & TopLevelStateFlagEnabledBit)
    {
        CecTopLevelOnApiEntry();

        if(setActiveSource)
        {
            if(CecDebounceIsInDebouncePeriod())
            {
                rval = CecTopLevelBusyError;
            }
            else
            {
                CecTopLevelStepFiniteStateMachine(&s_TopLevelState, TopLevelInputOtp);
                if(s_TopLevelState.lastError == CecNoError)
                {
                    CecDebounceOnOtp();
                }
            }
        }
        else
        {
            CecTopLevelStepFiniteStateMachine(&s_TopLevelState, TopLevelInputInactive);
        }
        rval = s_TopLevelState.lastError;

        rval = CecTopLevelOnApiExit(rval);
    }
    NN_CEC_INFO("%s: -> 0x%X\n", NN_CURRENT_FUNCTION_NAME, rval);
    return rval;
}

int32_t CecTopLevelSendDeviceMenuStateCommand(bool activate) NN_NOEXCEPT
{
    int32_t rval;

    CecTopLevelOnApiEntry();
    if(CecTopLevelIsInitializationOk())
    {
        rval = CecReceiverSendDeviceMenuActivationCommand(activate);
    }
    else
    {
        rval = CecTopLevelNotInitError;
    }
    rval = CecTopLevelOnApiExit(rval);
    return rval;
}

int32_t CecTopLevelSendRemoteControlCommand(bool start, uint8_t argument) NN_NOEXCEPT
{
    int32_t rval;

    CecTopLevelOnApiEntry();
    if(CecTopLevelIsInitializationOk())
    {
        rval = CecReceiverSendRemoteControlCommand(start, argument);
    }
    else
    {
        rval = CecTopLevelNotInitError;
    }
    rval = CecTopLevelOnApiExit(rval);
    return rval;
}

int32_t CecTopLevelSetOnScreenDisplay(uint8_t* string) NN_NOEXCEPT
{
    int32_t rval;

    CecTopLevelOnApiEntry();
    if(CecTopLevelIsInitializationOk())
    {
        rval = CecReceiverSendOnScreenDisplay(string);
    }
    else
    {
        rval = CecTopLevelNotInitError;
    }
    rval = CecTopLevelOnApiExit(rval);
    return rval;
}

bool    CecTopLevelSetCancelState() NN_NOEXCEPT
{
    NN_CEC_INFO("%s Entry\n", NN_CURRENT_FUNCTION_NAME);
    return(CecCancelSetCancelState(&s_TopLevelState.mutex));
}

int32_t CecTopLevelGoStandby(bool tvOnly) NN_NOEXCEPT
{
    int32_t rval;

    CecTopLevelOnApiEntry();
    s_TopLevelState.booleanArg = tvOnly;
    CecTopLevelStepFiniteStateMachine(&s_TopLevelState, TopLevelInputStandby);
    rval = CecTopLevelOnApiExit(s_TopLevelState.lastError);
    return rval;
}

void    CecTopLevelHandleReceivedStandby() NN_NOEXCEPT
{
    CecTopLevelStepFiniteStateMachine(&s_TopLevelState, TopLevelInputReceivedStandby);
}

void    CecTopLevelOnTvPowerOffChange() NN_NOEXCEPT
{
    CecTopLevelStepFiniteStateMachine(&s_TopLevelState, TopLevelInputOnInactiveSource);
}

void    CecTopLevelOnTvPowerGoingOnChange() NN_NOEXCEPT
{
    CecTopLevelStepFiniteStateMachine(&s_TopLevelState, TopLevelInputOnTxFailure);
}

void    CecTopLevelOnActiveSourceStateChange() NN_NOEXCEPT
{
    bool    isActiveSource;
    uint8_t fsmInput;

    os::LockMutex(&s_TopLevelState.operatingState.mutex);
    isActiveSource = (s_TopLevelState.operatingState.opStatus & CecOpStatusFlagActiveSource);
    fsmInput = (isActiveSource) ? TopLevelInputOnActiveSource : TopLevelInputOnInactiveSource;
    os::UnlockMutex(&s_TopLevelState.operatingState.mutex);
    CecTopLevelStepFiniteStateMachine(&s_TopLevelState, fsmInput);
}

bool    CecTopLevelIsActiveSource() NN_NOEXCEPT
{
    bool isActiveSource;

    os::LockMutex(&s_TopLevelState.operatingState.mutex);
    isActiveSource = (s_TopLevelState.operatingState.opStatus & CecOpStatusFlagActiveSource);
    os::UnlockMutex(&s_TopLevelState.operatingState.mutex);
    return isActiveSource;
}

void    CecTopLevelSetActiveSourceState(bool activeSource) NN_NOEXCEPT
{
    os::LockMutex(&s_TopLevelState.operatingState.mutex);
    s_TopLevelState.operatingState.opStatus &= ~CecOpStatusFlagActiveSource;
    if(activeSource)
    {
        s_TopLevelState.operatingState.opStatus |= CecOpStatusFlagActiveSource;
    }
    os::UnlockMutex(&s_TopLevelState.operatingState.mutex);
}

int32_t CecTopLevelIsStarted(bool* pIsStarted) NN_NOEXCEPT
{
    int32_t rval;

    NN_SDK_ASSERT(s_TopLevelState.flags & TopLevelStateFlagEnabledBit);
    *pIsStarted = CecTopLevelIsInitializationOk();
    rval = CecNoError;
    return rval;
}

int32_t CecTopLevelGetTvPowerStatus(uint8_t* pOutTvPowerState) NN_NOEXCEPT
{
    int32_t rval = CecManagerImplSubsystemSuspended;
    uint8_t logicalAddress;

    NN_SDK_ASSERT(s_TopLevelState.flags & TopLevelStateFlagEnabledBit);
    NN_SDK_ASSERT(pOutTvPowerState != nullptr);
    CecTopLevelOnApiEntry();
    if(CecTopLevelIsInitializationOk())
    {
        os::LockMutex(&s_TopLevelState.operatingState.mutex);
        logicalAddress = s_TopLevelState.operatingState.logicalAddress;
        os::UnlockMutex(&s_TopLevelState.operatingState.mutex);
        if(logicalAddress != CecConstantNotALogicalAddress)
        {
            rval = CecManagerGetTvPowerStatus(pOutTvPowerState, logicalAddress);
            if(rval == CecManagerImplCommandExecutionFailed)
            {
                CecTopLevelStepFiniteStateMachine(&s_TopLevelState, TopLevelInputOnTxFailure);
            }
        }
    }
    rval = CecTopLevelOnApiExit(rval);
    return rval;
}

int32_t CecTopLevelStop() NN_NOEXCEPT
{
    int32_t rval;

    NN_SDK_ASSERT(s_TopLevelState.flags & TopLevelStateFlagEnabledBit);
    CecTopLevelOnApiEntry();

    CecLowLevelInterfacePrepareForStop();
    os::LockMutex(&s_TopLevelState.operatingState.mutex);
    s_TopLevelState.operatingState.powerState = CecPowerStateStandby;
    os::UnlockMutex(&s_TopLevelState.operatingState.mutex);
    CecTopLevelStepFiniteStateMachine(&s_TopLevelState, TopLevelInputReset);
    CecDebounceOnReset();

    rval = CecTopLevelOnApiExit(s_TopLevelState.lastError);

    os::LockMutex(&s_TopLevelState.finiteStateMachineMutex);
    os::UnlockMutex(&s_TopLevelState.finiteStateMachineMutex);
    CecLowLevelInterfaceStop();
    return rval;
}

}
}   // namespace cec
}   // namespace nn
