﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/os/os_MutexTypes.h>
#include <nn/nn_SdkAssert.h>
#include <nn/settings/system/settings_Tv.h>

#include "cec_DetailLog.h"
#include "cec_Error.h"
#include "cec_FiniteStateMachine.h"
#include "cec_Utilities.h"
#include "cec_LowLevelInterface.h"
#include "cec_TransmitterFiniteStateMachine.h"
#include "cec_LogicalAddressFiniteStateMachine.h"
#include "cec_TopLevelFiniteStateMachine.h"
#include "cec_DebounceFiniteStateMachine.h"
#include "cec_Receiver.h"
#include "cec_Opcodes.h"
#include "cec_Manager.h"
#include "cec_ManagerImpl.private.h"

namespace nn { namespace cec { namespace detail {

namespace {

    uint32_t    s_CecManagerFlags = 0;
    const uint32_t  CecManagerFlagIsInit = (1 << 0);

    enum CommandSequencerState
    {
        CommandSequencerState_pending,
        CommandSequencerState_responded,
        CommandSequencerState_featureAbort,
        CommandSequencerState_cancelled
    };

    struct CecManagerCommandSequencerState
    {
        struct CecManagerCommandSequencerState* pLinkToNextStruct;
        nn::os::MutexType                       commandSequenceMutex;
        nn::os::TimerEventType                  responseTimeoutEvent;
        enum CommandSequencerState              currentState;
        bool                                    isDoneUponCompletion;
        uint8_t                                 responseBuffer[CecReceiveParamMaxMessageLength];
        uint8_t                                 responseBufferByteCount;
        uint8_t                                 commandOpcode;
        uint8_t                                 commandCompletionOpcode;
        uint8_t                                 commandCompletionByteCount;
    };

    struct CecManagerCommandSequenceCompletionState
    {
        nn::os::MutexType                       completionAccessMutex;
        struct CecManagerCommandSequencerState* pHead;
    };

    const uint32_t CecManagerSequenceCompletionTimeoutMilliseconds = 1000;
    struct CecManagerCommandSequenceCompletionState s_CommandCompletionState;

    void CecManagerAddCommandSequencerStateToTail(struct CecManagerCommandSequencerState* pCommandSequencerState) NN_NOEXCEPT
    {
        struct CecManagerCommandSequencerState**    ppCurrent;

        nn::os::LockMutex(&s_CommandCompletionState.completionAccessMutex);
        ppCurrent = &s_CommandCompletionState.pHead;
        pCommandSequencerState->pLinkToNextStruct = nullptr;
        while(*ppCurrent != nullptr)
        {
            ppCurrent = &(*ppCurrent)->pLinkToNextStruct;
        }
        *ppCurrent = pCommandSequencerState;
        nn::os::UnlockMutex(&s_CommandCompletionState.completionAccessMutex);
    }

    void CecManagerDeleteCommandSequencerState(struct CecManagerCommandSequencerState* pCommandSequencerState) NN_NOEXCEPT
    {
        struct CecManagerCommandSequencerState**    ppCurrent;

        nn::os::LockMutex(&s_CommandCompletionState.completionAccessMutex);
        ppCurrent = &s_CommandCompletionState.pHead;
        while(*ppCurrent != nullptr)
        {
            if(*ppCurrent == pCommandSequencerState)
            {
                *ppCurrent = pCommandSequencerState->pLinkToNextStruct;
                pCommandSequencerState->pLinkToNextStruct = nullptr;
                break;
            }
            ppCurrent = &(*ppCurrent)->pLinkToNextStruct;
        }
        nn::os::UnlockMutex(&s_CommandCompletionState.completionAccessMutex);
    }

    int32_t CecManagerSequenceCompletion(struct CecManagerCommandSequencerState* pCommandSequencerState,
                                         uint8_t* pTxBuffer, uint8_t txCount, bool isDirectAddressed) NN_NOEXCEPT
    {
        int32_t rval;

        NN_SDK_ASSERT(txCount >= CecBufferElementCountIsTwo);
        pCommandSequencerState->currentState = CommandSequencerState_pending;
        pCommandSequencerState->commandOpcode = pTxBuffer[CecBufferOpcodeIndex];
        nn::os::InitializeMutex(&pCommandSequencerState->commandSequenceMutex, false, 0);
        nn::os::InitializeTimerEvent(&pCommandSequencerState->responseTimeoutEvent, nn::os::EventClearMode_AutoClear);
        CecManagerAddCommandSequencerStateToTail(pCommandSequencerState);
        rval = CecTransmitBestEffort(pTxBuffer, txCount, isDirectAddressed);
        nn::os::StartOneShotTimerEvent(&pCommandSequencerState->responseTimeoutEvent,
                                       nn::TimeSpan::FromMilliSeconds(CecManagerSequenceCompletionTimeoutMilliseconds));
        if(rval == CecNoError)
        {
            nn::os::WaitTimerEvent(&pCommandSequencerState->responseTimeoutEvent);
            nn::os::LockMutex(&pCommandSequencerState->commandSequenceMutex);
            switch(pCommandSequencerState->currentState)
            {
                case CommandSequencerState_pending:
                    pCommandSequencerState->responseBufferByteCount = 0;
                    rval = CecManagerImplTimeoutError;
                    break;
                case CommandSequencerState_featureAbort:
                    pCommandSequencerState->responseBufferByteCount = 0;
                    rval = CecManagerImplFeatureAbortError;
                    break;
                case CommandSequencerState_cancelled:
                    pCommandSequencerState->responseBufferByteCount = 0;
                    rval = CecManagerImplCancelledError;
                    break;
                default:
                    break;
            }
            nn::os::UnlockMutex(&pCommandSequencerState->commandSequenceMutex);
        }
        else
        {
            nn::os::ClearTimerEvent(&pCommandSequencerState->responseTimeoutEvent);
            rval = CecManagerImplCommandExecutionFailed;
        }
        CecManagerDeleteCommandSequencerState(pCommandSequencerState);
        nn::os::FinalizeTimerEvent(&pCommandSequencerState->responseTimeoutEvent);
        nn::os::FinalizeMutex(&pCommandSequencerState->commandSequenceMutex);
        return rval;
    }

}

int32_t CecManagerGetTvPowerStatus(uint8_t* pOutPowerState, uint8_t logicalAddress) NN_NOEXCEPT
{
    int32_t rval;
    struct CecManagerCommandSequencerState  commandSequencerState;
    uint8_t logicalAddressFieldValue;
    uint8_t txBuffer[CecReceiveParamMaxMessageLength];

    logicalAddressFieldValue = (logicalAddress << 4) | CecReceiveParamTvLogicalAddress;
    txBuffer[CecBufferLogicalAddressIndex] = logicalAddressFieldValue;
    txBuffer[CecBufferOpcodeIndex] = CecOpcodeGiveDevicePowerStatus;
    commandSequencerState.commandCompletionOpcode = CecOpcodeReportPowerStatus;
    commandSequencerState.commandCompletionByteCount = CecBufferElementCountIsOne;
    commandSequencerState.isDoneUponCompletion = false;
    rval = CecManagerSequenceCompletion(&commandSequencerState, txBuffer, CecBufferElementCountIsTwo, true);
    if(rval == CecNoError)
    {
        *pOutPowerState = commandSequencerState.responseBuffer[0];
    }
    return rval;
}

bool CecManagerSequenceCompletionHandler(uint8_t* cecBuffer, uint8_t cecBufferByteCount) NN_NOEXCEPT
{
    bool    rval = false;
    bool    signalEvent = false;
    struct CecManagerCommandSequencerState**    ppCurrent;
    struct CecManagerCommandSequencerState*     pCurrentCommandSequencerState;

    if(cecBufferByteCount >= CecBufferElementCountIsTwo)
    {
        nn::os::LockMutex(&s_CommandCompletionState.completionAccessMutex);
        ppCurrent = &s_CommandCompletionState.pHead;
        while(*ppCurrent != nullptr)
        {
            pCurrentCommandSequencerState = *ppCurrent;
            nn::os::LockMutex(&pCurrentCommandSequencerState->commandSequenceMutex);
            if(pCurrentCommandSequencerState->currentState == CommandSequencerState_pending)
            {
                if(pCurrentCommandSequencerState->commandCompletionOpcode == cecBuffer[CecBufferOpcodeIndex])
                {
                    uint8_t index;

                    for(index = 0; index < pCurrentCommandSequencerState->commandCompletionByteCount; index++)
                    {
                        pCurrentCommandSequencerState->responseBuffer[index] = cecBuffer[CecBufferPayloadIndex + index];
                    }
                    pCurrentCommandSequencerState->responseBufferByteCount = index;
                    pCurrentCommandSequencerState->currentState = CommandSequencerState_responded;
                    signalEvent = true;
                    rval = pCurrentCommandSequencerState->isDoneUponCompletion;
                }
                else if((cecBuffer[CecBufferOpcodeIndex] == CecOpcodeFeatureAbort) &&
                        (cecBufferByteCount == CecBufferElementCountIsFour) &&
                        (pCurrentCommandSequencerState->commandOpcode == cecBuffer[CecBufferFeatureAbortOpcodeIndex]))
                {
                    pCurrentCommandSequencerState->currentState = CommandSequencerState_featureAbort;
                    signalEvent = true;
                    rval = true;
                }
            }
            nn::os::UnlockMutex(&pCurrentCommandSequencerState->commandSequenceMutex);
            if(signalEvent == true)
            {
                break;
            }
            ppCurrent = &(*ppCurrent)->pLinkToNextStruct;
        }
        nn::os::UnlockMutex(&s_CommandCompletionState.completionAccessMutex);
        if(signalEvent == true)
        {
            nn::os::SignalTimerEvent(&pCurrentCommandSequencerState->responseTimeoutEvent);
        }
    }
    return rval;
}

void CecManagerSequenceCancellationHandler() NN_NOEXCEPT
{
    struct CecManagerCommandSequencerState**    ppCurrent;
    struct CecManagerCommandSequencerState*     pCurrentCommandSequencerState;

    if(s_CecManagerFlags & CecManagerFlagIsInit)
    {
        nn::os::LockMutex(&s_CommandCompletionState.completionAccessMutex);
        ppCurrent = &s_CommandCompletionState.pHead;
        while(*ppCurrent != nullptr)
        {
            pCurrentCommandSequencerState = *ppCurrent;
            nn::os::LockMutex(&pCurrentCommandSequencerState->commandSequenceMutex);
            if(pCurrentCommandSequencerState->currentState == CommandSequencerState_pending)
            {
                nn::os::SignalTimerEvent(&pCurrentCommandSequencerState->responseTimeoutEvent);
                pCurrentCommandSequencerState->currentState = CommandSequencerState_cancelled;
            }
            nn::os::UnlockMutex(&pCurrentCommandSequencerState->commandSequenceMutex);
            ppCurrent = &(*ppCurrent)->pLinkToNextStruct;
        }
        nn::os::UnlockMutex(&s_CommandCompletionState.completionAccessMutex);
    }
}

bool CecManagerInit(bool startCec) NN_NOEXCEPT
{
    int32_t functionReturnValue;
    int32_t rval = true;

    if(!(s_CecManagerFlags & CecManagerFlagIsInit))
    {
        NN_CEC_INFO("%s: begin\n", NN_CURRENT_FUNCTION_NAME);
        CecCancelInitialize();
        nn::os::InitializeMutex(&s_CommandCompletionState.completionAccessMutex, false, 0);
        s_CommandCompletionState.pHead = nullptr;
        CecDebounceInit();
        functionReturnValue = CecTopLevelInit();
        if(functionReturnValue == CecTopLevelNotSupportedError)
        {
            rval = false;
        }
        else
        {
            NN_SDK_ASSERT(functionReturnValue == CecNoError);
            CecLogicalAddressInit();
            CecTransmitInit();
            functionReturnValue = CecLowLevelInterfaceInit(LowLevelInterfaceSelectDisplayPort);
            NN_SDK_ASSERT(functionReturnValue == CecNoError);
            if(startCec)
            {
                CecTopLevelStart();
            }
            s_CecManagerFlags |= CecManagerFlagIsInit;
            NN_CEC_INFO("%s: end\n", NN_CURRENT_FUNCTION_NAME);
        }
    }
    return rval;
}

void    CecManagerShutdown() NN_NOEXCEPT
{
    if(s_CecManagerFlags & CecManagerFlagIsInit)
    {
        CecTopLevelShutdown();
        CecLogicalAddressShutdown();
        CecTransmitShutdown();
        CecLowLevelInterfaceStop();
        CecLowLevelInterfaceShutdown();
        CecDebounceFinalize();
        CecCancelFinalize();
        s_CecManagerFlags = 0;
    }
}

void    CecManagerHandleActiveSource(uint8_t sourceLogicalAddress, bool isSame, bool isRoutingChange) NN_NOEXCEPT
{
    uint8_t data[16];

    if(isRoutingChange)
    {
        CecDebounceOnRoutingChange(isSame);
    }
    else if(sourceLogicalAddress == CecReceiveParamTvLogicalAddress)
    {
        CecDebounceOnRoutingChange(false);
    }
    CecTopLevelOnActiveSourceStateChange();
    data[0] = sourceLogicalAddress;
    data[1] = isSame;
    data[2] = isRoutingChange;
    server::CecManagerImplOnEvent(server::CecEventActiveSource, 3, data);
    NN_CEC_INFO("%s: %02X, is same %d routing change %s\n", NN_CURRENT_FUNCTION_NAME, sourceLogicalAddress, isSame,
                (isRoutingChange == true) ? "true" : "false");
}
void    CecManagerHandleGoStandby() NN_NOEXCEPT
{
    uint8_t data[16];

    CecDebounceOnStandby();
    CecTopLevelHandleReceivedStandby();
    server::CecManagerImplOnEvent(server::CecEventGoStandby, 0, data);
    NN_CEC_INFO("%s\n", NN_CURRENT_FUNCTION_NAME);
}
/*
 * this group of functions is used to create a map of present devices and their respective attributes
 */
void    CecManagerHandleCecVersion(uint8_t sourceLogicalAddress, uint8_t cecVersion) NN_NOEXCEPT
{
    NN_CEC_INFO("%s: %02X\t%02X\n", NN_CURRENT_FUNCTION_NAME, sourceLogicalAddress, cecVersion);
}
void    CecManagerHandleDeviceVendorId(uint8_t sourceLogicalAddress, uint8_t* pVendorId) NN_NOEXCEPT
{
    NN_CEC_INFO("%s: %02X\t%02X:%02X:%02X\n", NN_CURRENT_FUNCTION_NAME, sourceLogicalAddress, pVendorId[0], pVendorId[1], pVendorId[2]);
}
/*
 * only relevant in context of transmitting commands
 */
void    CecManagerHandleFeatureAbort(uint8_t sourceLogicalAddress, uint8_t opcode, uint8_t reason) NN_NOEXCEPT
{
    uint8_t data[16];

    if(sourceLogicalAddress == CecReceiveParamTvLogicalAddress)
    {
        data[0] = reason;
        switch(opcode)
        {
            case CecOpcodeImageViewOn:
            case CecOpcodeActiveSource:
                server::CecManagerImplOnEvent(server::CecEventOtpFeatureAbort, 1, data);
                break;
            case CecOpcodeStandby:
                server::CecManagerImplOnEvent(server::CecEventStandbyFeatureAbort, 1, data);
                break;
            case CecOpcodeSetOsdString:
                server::CecManagerImplOnEvent(server::CecEventSetOsdStringFeatureAbort, 1, data);
                break;
            case CecOpcodeMenuRequest:
            case CecOpcodeMenuStatus:
            case CecOpcodeUserControlPressed:
            case CecOpcodeUserControlReleased:
                server::CecManagerImplOnEvent(server::CecEventDeviceMenuControlFeatureAbort, 1, data);
                break;
            default:
                break;
        }
    }
    NN_CEC_INFO("%s: %02X\t%02X\n", NN_CURRENT_FUNCTION_NAME, sourceLogicalAddress, opcode);
}

void    CecManagerHandleReportPowerStatus(uint8_t sourceLogicalAddress, uint8_t powerStatus) NN_NOEXCEPT
{
    if(sourceLogicalAddress == CecReceiveParamTvLogicalAddress)
    {
        if((powerStatus == CecPowerStatusStandby) || (powerStatus == CecPowerStatusTransitionToStandby))
        {
            CecTopLevelOnTvPowerOffChange();
        }
        else if(powerStatus == CecPowerStatusTransitionToOn)
        {
            CecTopLevelOnTvPowerGoingOnChange();
        }
    }
    NN_CEC_INFO("%s: %02X\t%02X\n", NN_CURRENT_FUNCTION_NAME, sourceLogicalAddress, powerStatus);
}
/*
 * vendor command top level functions as invoked by receiver
 */
bool CecManagerHandleVendorCommand(uint8_t sourceLogicalAddress, uint8_t count, uint8_t* pData) NN_NOEXCEPT
{
    bool isHandled = true;

    NN_CEC_INFO("%s: %02X\n", NN_CURRENT_FUNCTION_NAME, sourceLogicalAddress);
    return isHandled;
}
bool CecManagerHandleVendorCommandWithId(uint8_t sourceLogicalAddress, uint8_t* pVendorId, uint8_t count, uint8_t* pData) NN_NOEXCEPT
{
    bool isHandled;

    isHandled = ((pVendorId[0] == CecDeviceVendorIdLow) &&
                 (pVendorId[1] == CecDeviceVendorIdMid) &&
                 (pVendorId[2] == CecDeviceVendorIdHigh));

    NN_CEC_INFO("%s: %02X, %02X:%02X:%02X\t%d\n", NN_CURRENT_FUNCTION_NAME, sourceLogicalAddress,
                pVendorId[0], pVendorId[1], pVendorId[2], isHandled);
    return isHandled;
}
bool CecManagerHandleVendorRemoteControlButtonDown(uint8_t sourceLogicalAddress, uint8_t count, uint8_t* pRemoteControlButtonData) NN_NOEXCEPT
{
    bool isHandled = true;

    NN_CEC_INFO("%s: %02X\n", NN_CURRENT_FUNCTION_NAME, sourceLogicalAddress);
    return isHandled;
}
bool CecManagerHandleVendorRemoteControlButtonUp(uint8_t sourceLogicalAddress) NN_NOEXCEPT
{
    bool isHandled = true;

    NN_CEC_INFO("%s: %02X\n", NN_CURRENT_FUNCTION_NAME, sourceLogicalAddress);
    return isHandled;
}

}
}   // namespace cec
}   // namespace nn
