﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_FormatString.h>

#include "cdmsc_Log.h"
#include "cdmsc_Utils.h"

namespace nn {
namespace cdmsc {
namespace detail {

void DumpBuffer(void *buffer, int size) NN_NOEXCEPT
{
    char *p = (char*)buffer;

    int remain = size;
    int line   = 0;

    // NN_CDMSC_INFO won't expand for release build
    NN_UNUSED(line);

    while (remain)
    {
        int i;
        int offset = size - remain;

        // offset marker
        NN_CDMSC_INFO("%04x: ", line++);

        // hex dump
        for (i = 0; i < remain && i < 16; i++)
        {
            NN_CDMSC_INFO(" %02x", p[offset + i]);

            if (i % 16 == 7)
            {
                NN_CDMSC_INFO(" ");
            }
        }

        // hex padding
        for (; i < 16; i++)
        {
            NN_CDMSC_INFO("   ");

            if (i % 16 == 7)
            {
                NN_CDMSC_INFO(" ");
            }
        }

        // delimiter between hex and char
        NN_CDMSC_INFO(" | ");

        // char dump
        for (i = 0; i < remain && i < 16; i++)
        {
            char c = p[offset + i];
            if (c >= ' ' && c <= '~')
            {
                NN_CDMSC_INFO("%c", c);
            }
            else
            {
                NN_CDMSC_INFO(".");
            }

            if (i % 16 == 7)
            {
                NN_CDMSC_INFO(" ");
            }
        }

        // char padding
        for (; i < 16; i++)
        {
            NN_CDMSC_INFO(" ");

            if (i % 16 == 7)
            {
                NN_CDMSC_INFO(" ");
            }
        }

        NN_CDMSC_INFO("\n");

        remain = (remain >= 16) ? (remain - 16) : 0;
    }
}

static AllocateFunction   g_AllocateFunction;
static DeallocateFunction g_DeallocateFunction;

static nn::os::MutexType  g_Mutex = NN_OS_MUTEX_INITIALIZER(false);

void SetAllocator(AllocateFunction alloc, DeallocateFunction dealloc) NN_NOEXCEPT
{
    nn::os::LockMutex(&g_Mutex);

    NN_CDMSC_ABORT_IF_NULL(alloc);
    NN_CDMSC_ABORT_IF_NULL(dealloc);

    g_AllocateFunction   = alloc;
    g_DeallocateFunction = dealloc;

    nn::os::UnlockMutex(&g_Mutex);
}

void* Allocate(size_t alignment, size_t size) NN_NOEXCEPT
{
    nn::os::LockMutex(&g_Mutex);

    NN_CDMSC_ABORT_IF_NULL(g_AllocateFunction);

    void* p = g_AllocateFunction(alignment, size);

    nn::os::UnlockMutex(&g_Mutex);

    return p;
}

void Deallocate(void* p, size_t size) NN_NOEXCEPT
{
    nn::os::LockMutex(&g_Mutex);

    NN_CDMSC_ABORT_IF_NULL(g_DeallocateFunction);

    if( p != nullptr )
    {
        g_DeallocateFunction(p, size);
    }

    nn::os::UnlockMutex(&g_Mutex);
}

} // end of namespace detail
} // end of namespace cdmsc
} // end of namespace nn
