﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cdhidWin32Mouse_Private.h"
#include <nn/nn_SystemThreadDefinition.h>

namespace nn {
namespace cdhid {
namespace win32 {

//////////////////////////////////////////////////////////////////////////////
//  public functions
//////////////////////////////////////////////////////////////////////////////
Result Win32MouseInterface::Initialize() NN_NOEXCEPT
{
    m_Hdr.Initialize();
    m_CodeBook.Initialize();
    m_Buffer.Initialize(&m_MouseData[0][0], MOUSE_DATA_ENTRIES, MOUSE_DATA_SIZE);

    AttachDevice();

    // start pollng thread for reads
    nn::os::CreateThread(
        &m_ReadThread,
        ThreadFunction,
        this,
        m_ReadThreadStack,
        MOUSE_THREAD_STACK_SIZE,
        NN_SYSTEM_THREAD_PRIORITY(cdhid,Win32MouseInterface)
        );

    nn::os::SetThreadNamePointer(&m_ReadThread, NN_SYSTEM_THREAD_NAME(cdhid, Win32MouseInterface));
    nn::os::StartThread(&m_ReadThread);

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Win32MouseInterface::Finalize() NN_NOEXCEPT
{
    if (m_Hwnd)
    {
        SendMessage(m_Hwnd, WM_DESTROY, 0, 0);  // hack :0

        nn::os::WaitThread(&m_ReadThread);
        nn::os::DestroyThread(&m_ReadThread);
    }

    DetachDevice();

    m_Buffer.Finalize();
    m_CodeBook.Finalize();
    m_Hdr.Finalize();

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Win32MouseInterface::AcquireDevice() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if (m_IsAcquired == false)
    {
        m_IsAcquired    = true;
        m_CtrlSession   = nullptr;
    }
    else
    {
        result = nn::ahid::ResultAcquire();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Win32MouseInterface::ReleaseDevice() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    if (m_IsAcquired == true)
    {
        m_CtrlSession   = nullptr;
        m_IsAcquired    = false;
    }
    else
    {
        result = nn::ahid::ResultAcquire();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Win32MouseInterface::SetCtrlSession(nn::sf::SharedPointer<nn::ahid::ICtrlSession> pCtrlSession) NN_NOEXCEPT
{
    m_CtrlSession = pCtrlSession;

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Win32MouseInterface::AhidGetCodeBook(uint8_t* pBuffer, uint32_t bufferSize) NN_NOEXCEPT
{
    return m_CodeBook.Get(pBuffer, bufferSize);
}


//////////////////////////////////////////////////////////////////////////////
Result Win32MouseInterface::AhidRead(uint32_t* pBytesTransferred, uint8_t* pBuffer, uint32_t bytes, nn::TimeSpan timeout) NN_NOEXCEPT
{
    NN_UNUSED(bytes);
    NN_UNUSED(timeout);

    // TODO check buffer size

    return m_Buffer.Get(pBuffer, pBytesTransferred);
}


//////////////////////////////////////////////////////////////////////////////
//  private functions
//////////////////////////////////////////////////////////////////////////////
Result Win32MouseInterface::AttachDevice() NN_NOEXCEPT
{
    m_IsAcquired = false;

    nn::ahid::hdr::DeviceParameters hdrDeviceParameters;

    memset(&hdrDeviceParameters, 0, sizeof(nn::ahid::hdr::DeviceParameters));

    sprintf(reinterpret_cast<char*>(hdrDeviceParameters.servicePath), "mouse");
    hdrDeviceParameters.deviceHandle    = 1;
    hdrDeviceParameters.busId           = nn::ahid::hdr::AhidBusIdWin32;
    hdrDeviceParameters.usagePage       = 0x01;     // Generic Desktop Controls
    hdrDeviceParameters.usageId         = 0x02;     // Mouse

    char manufacturerString[16] = {16 , nn::ahid::hdr::AhidDescriptorTypeString, 'H', 0, 'o', 0, 'r', 0, 'i', 0, 'z', 0, 'o', 0, 'n', 0};
    char productStringMouse[22] = {22 , nn::ahid::hdr::AhidDescriptorTypeString, 'A', 0, 'H', 0, 'I', 0, 'D', 0, ' ', 0, 'M', 0, 'o', 0, 'u', 0, 's', 0, 'e', 0};

    memcpy(hdrDeviceParameters.manufacturer, manufacturerString, sizeof(manufacturerString));
    memcpy(hdrDeviceParameters.product, productStringMouse, sizeof(productStringMouse));

    return m_Hdr.AttachDevice(&m_HdrDeviceHandle, &hdrDeviceParameters);
}


//////////////////////////////////////////////////////////////////////////////
Result Win32MouseInterface::DetachDevice() NN_NOEXCEPT
{
    if (m_CtrlSession)
    {
        m_CtrlSession->SignalStateChangeEvent();
    }

    m_IsAcquired = false;

    return m_Hdr.DetachDevice(m_HdrDeviceHandle);
}


//////////////////////////////////////////////////////////////////////////////
static Win32MouseBuffer*    g_pWin32MouseBuffer;
static Win32MouseInterface* g_pWin32MouseInterface;

static LRESULT CALLBACK WndProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
    switch (uMsg)
    {
    case WM_DESTROY:
        {
            PostQuitMessage(0);
        }
        return 0;

    case WM_INPUT:
        {
            RAWINPUT    rawInput;
            UINT        bufferSize  = sizeof(RAWINPUT);
            UINT        dataSize;

            dataSize = GetRawInputData((HRAWINPUT)lParam, RID_INPUT, &rawInput, &bufferSize, sizeof(RAWINPUTHEADER));

            if (rawInput.header.dwType == RIM_TYPEMOUSE)
            {
                uint8_t data[4];

                // buttons
                data[0] = 0;

                if (rawInput.data.mouse.usButtonFlags & 0x01)   { g_pWin32MouseInterface->m_LeftButton    = true;     }
                if (rawInput.data.mouse.usButtonFlags & 0x02)   { g_pWin32MouseInterface->m_LeftButton    = false;    }
                if (rawInput.data.mouse.usButtonFlags & 0x04)   { g_pWin32MouseInterface->m_RightButton   = true;     }
                if (rawInput.data.mouse.usButtonFlags & 0x08)   { g_pWin32MouseInterface->m_RightButton   = false;    }
                if (rawInput.data.mouse.usButtonFlags & 0x10)   { g_pWin32MouseInterface->m_CenterButton  = true;     }
                if (rawInput.data.mouse.usButtonFlags & 0x20)   { g_pWin32MouseInterface->m_CenterButton  = false;    }
                if (rawInput.data.mouse.usButtonFlags & 0x40)   { g_pWin32MouseInterface->m_4thButton     = true;     }
                if (rawInput.data.mouse.usButtonFlags & 0x80)   { g_pWin32MouseInterface->m_4thButton     = false;    }
                if (rawInput.data.mouse.usButtonFlags & 0x100)  { g_pWin32MouseInterface->m_5thButton     = true;     }
                if (rawInput.data.mouse.usButtonFlags & 0x200)  { g_pWin32MouseInterface->m_5thButton     = false;    }

                if (g_pWin32MouseInterface->m_LeftButton)       { data[0]  |= 0x01; }
                if (g_pWin32MouseInterface->m_RightButton)      { data[0]  |= 0x02; }
                if (g_pWin32MouseInterface->m_CenterButton)     { data[0]  |= 0x04; }
                if (g_pWin32MouseInterface->m_4thButton)        { data[0]  |= 0x08; }
                if (g_pWin32MouseInterface->m_5thButton)        { data[0]  |= 0x10; }

                // x movement
                data[1] = static_cast<uint8_t>(rawInput.data.mouse.lLastX);

                // y movement
                data[2] = static_cast<uint8_t>(rawInput.data.mouse.lLastY);

                // scroll wheel
                data[3] = 0;

                if ((rawInput.data.mouse.usButtonFlags == 0x0400) && (rawInput.data.mouse.usButtonData == 0x0078))
                {
                    data[3]  = 0x01;
                }

                if ((rawInput.data.mouse.usButtonFlags == 0x0400) && (rawInput.data.mouse.usButtonData == 0xff88))
                {
                    data[3]  = 0xff;
                }

                g_pWin32MouseBuffer->Put(data);
            }
        }

        return 0;

    default:

        break;
    }

    return DefWindowProc(hwnd, uMsg, wParam, lParam);
}


//////////////////////////////////////////////////////////////////////////////
void Win32MouseInterface::ReadFromMouse()
{
    g_pWin32MouseInterface = this;
    g_pWin32MouseBuffer = &this->m_Buffer;

    const wchar_t CLASS_NAME[]  = L"AHID Mouse";

    WNDCLASS wndclass = { };

    wndclass.lpfnWndProc    = WndProc;
    wndclass.hInstance      = 0;
    wndclass.lpszClassName  = CLASS_NAME;

    RegisterClass(&wndclass);

    m_Hwnd = CreateWindow(
                            CLASS_NAME,
                            CLASS_NAME,
                            0,
                            0,
                            0,
                            0,
                            0,
                            HWND_MESSAGE,
                            NULL,
                            0,
                            this
                            );

    RAWINPUTDEVICE rid;

    rid.usUsagePage = 0x01;
    rid.usUsage     = 0x02; // mouse
    rid.dwFlags     = RIDEV_NOLEGACY | RIDEV_INPUTSINK;
    rid.hwndTarget  = m_Hwnd;

    RegisterRawInputDevices(&rid, 1, sizeof(rid));

    ShowWindow(m_Hwnd, SW_MAXIMIZE);

    MSG msg = { };

    while (GetMessage(&msg, m_Hwnd, 0, 0))
    {
        TranslateMessage(&msg);
        DispatchMessage(&msg);

        if (msg.message == 0)   // hack :0
        {
            break;
        }
    }
}


} // end of namespace win32
} // end of namespace cdhid
} // end of namespace nn
