﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cdhidWin32Hid_Private.h"

namespace nn {
namespace cdhid {
namespace win32 {

//////////////////////////////////////////////////////////////////////////////
//  public functions
//////////////////////////////////////////////////////////////////////////////
Result Win32HidInterface::Initialize(uint16_t index) NN_NOEXCEPT
{
    m_HandleIndex   = index;
    m_IsOccupied    = false;
    m_IsAcquired    = false;
    m_IsAttached    = false;
    m_DevInst       = 0;

    ChangeHandle();

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Win32HidInterface::Finalize() NN_NOEXCEPT
{
    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Win32HidInterface::IncHandleSemaphore(uint32_t handle) NN_NOEXCEPT
{
    if (((handle >> 16) == m_HandleIndex) && ((handle & 0xffff) == m_HandleInstance))
    {
        // TODO inc transaction semaphore

        return ResultSuccess();
    }

    return nn::ahid::ResultInvalidHandle();
}


//////////////////////////////////////////////////////////////////////////////
Result Win32HidInterface::DecHandleSemaphore() NN_NOEXCEPT
{
    // TODO dec transaction semaphore

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Win32HidInterface::ClientAcquireDevice(uint32_t handle) NN_NOEXCEPT
{
    Result result = IncHandleSemaphore(handle);

    if (result.IsSuccess())
    {
        if (m_IsAcquired == false)
        {
            m_IsAcquired    = true;
            m_CtrlSession   = nullptr;
        }
        else
        {
            result = nn::ahid::ResultAcquire();
        }

        DecHandleSemaphore();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Win32HidInterface::ClientReleaseDevice(uint32_t handle) NN_NOEXCEPT
{
    Result result = IncHandleSemaphore(handle);

    if (result.IsSuccess())
    {
        if (m_IsAcquired == true)
        {
            m_CtrlSession   = nullptr;
            m_IsAcquired    = false;
        }
        else
        {
            result = nn::ahid::ResultAcquire();
        }

        DecHandleSemaphore();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result Win32HidInterface::SetCtrlSession(nn::sf::SharedPointer<nn::ahid::ICtrlSession> pCtrlSession) NN_NOEXCEPT
{
    m_CtrlSession = pCtrlSession;

    return ResultSuccess();
}


//////////////////////////////////////////////////////////////////////////////
Result Win32HidInterface::EnumAttachDevice(nn::ahid::hdr::Hdr* pHdr, nn::ahid::hdr::DeviceParameters* pHdrDeviceParameters) NN_NOEXCEPT
{
    if (m_IsOccupied == false)
    {
        m_IsAcquired    = false;
        m_IsAttached    = true;
        m_IsOccupied    = true;

        pHdrDeviceParameters->deviceHandle = (m_HandleIndex << 16) | m_HandleInstance;

        return pHdr->AttachDevice(&m_HdrDeviceHandle, pHdrDeviceParameters);
    }

    return nn::ahid::ResultInvalidHandle();
}


//////////////////////////////////////////////////////////////////////////////
Result Win32HidInterface::EnumDetachDevice(nn::ahid::hdr::Hdr* pHdr) NN_NOEXCEPT
{
    if (m_IsOccupied)
    {
        if (m_CtrlSession)
        {
            m_CtrlSession->SignalStateChangeEvent();
        }

        ChangeHandle();
        m_IsAttached    = false;
        m_IsAcquired    = false;
        m_IsOccupied    = false;

        return pHdr->DetachDevice(m_HdrDeviceHandle);
    }

    return nn::ahid::ResultInvalidHandle();
}


//////////////////////////////////////////////////////////////////////////////
void Win32HidInterface::SetDevInst(DWORD devInst) NN_NOEXCEPT
{
    m_DevInst = devInst;
}

//////////////////////////////////////////////////////////////////////////////
DWORD Win32HidInterface::GetDevInst() NN_NOEXCEPT
{
    return m_DevInst;
}


//////////////////////////////////////////////////////////////////////////////
void Win32HidInterface::SetIsAttached(bool isAttached) NN_NOEXCEPT
{
    m_IsAttached = isAttached;
}


//////////////////////////////////////////////////////////////////////////////
bool Win32HidInterface::GetIsAttached() NN_NOEXCEPT
{
    return m_IsAttached;
}


//////////////////////////////////////////////////////////////////////////////
void Win32HidInterface::SetWin32Handle(HANDLE handle) NN_NOEXCEPT
{
    m_Win32Handle = handle;
}


//////////////////////////////////////////////////////////////////////////////
HANDLE Win32HidInterface::GetWin32Handle() NN_NOEXCEPT
{
    return m_Win32Handle;
}


//////////////////////////////////////////////////////////////////////////////
//  private functions
//////////////////////////////////////////////////////////////////////////////
void Win32HidInterface::ChangeHandle() NN_NOEXCEPT
{
    m_HandleInstance++;

    if (m_HandleInstance == 0)
    {
        m_HandleInstance = 1;
    }
}


} // end of namespace win32
} // end of namespace cdhid
} // end of namespace nn
