﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cdhidUsb_Private.h"

namespace nn {
namespace cdhid {
namespace usb {

//////////////////////////////////////////////////////////////////////////////
ServerSession::ServerSession(UsbHidEnum *pUsbHidEnum, nn::sf::ExpHeapAllocator *pAllocator) NN_NOEXCEPT
{
    m_DeviceHandle  = 0;
    m_pUsbHidEnum   = pUsbHidEnum;
    m_pAllocator    = pAllocator;
}


//////////////////////////////////////////////////////////////////////////////
ServerSession::~ServerSession() NN_NOEXCEPT
{
    if (m_DeviceHandle)
    {
        m_pUsbHidEnum->ReleaseDevice(m_DeviceHandle);
    }
}


//////////////////////////////////////////////////////////////////////////////
Result ServerSession::AcquireDevice(uint32_t deviceHandle) NN_NOEXCEPT
{
    Result result = m_pUsbHidEnum->AcquireDevice(deviceHandle, &m_pUsbHidInterface);

    if (result.IsSuccess())
    {
        m_DeviceHandle = deviceHandle;
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result ServerSession::ReleaseDevice(uint32_t deviceHandle) NN_NOEXCEPT
{
    Result result = m_pUsbHidEnum->ReleaseDevice(deviceHandle);

    if (result.IsSuccess())
    {
        m_DeviceHandle = 0;
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result ServerSession::GetCtrlSession(uint32_t deviceHandle, nn::sf::Out<nn::sf::SharedPointer<nn::ahid::ICtrlSession>> outSession) NN_NOEXCEPT
{
    Result result;

    nn::sf::SharedPointer<nn::ahid::ICtrlSession> session;

    session = Factory::CreateSharedEmplaced<nn::ahid::ICtrlSession, CtrlSession>(m_pAllocator, m_pUsbHidInterface, deviceHandle);

    if (session)
    {
        // set ctrl session to interface for state change signal
        m_pUsbHidInterface->SetCtrlSession(session);

        *outSession = std::move(session);
        result = ResultSuccess();
    }
    else
    {
        result = nn::ahid::ResultMemAllocFailure();
    }

    return result;
}


//////////////////////////////////////////////////////////////////////////////
Result ServerSession::GetReadSession(uint32_t deviceHandle, nn::sf::Out<nn::sf::SharedPointer<nn::ahid::IReadSession>> outSession) NN_NOEXCEPT
{
    Result result;

    nn::sf::SharedPointer<nn::ahid::IReadSession> session;

    session = Factory::CreateSharedEmplaced<nn::ahid::IReadSession, ReadSession>(m_pAllocator, m_pUsbHidInterface, deviceHandle);

    if (session)
    {
        *outSession = std::move(session);
        result = ResultSuccess();
    }
    else
    {
        result = nn::ahid::ResultMemAllocFailure();
    }

    return result;
}


} // end of namespace usb
} // end of namespace cdhid
} // end of namespace nn
