﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/ahid/hdr/hdr.h>

namespace nn {
namespace cdhid {
namespace usb {

struct UsbHidDescriptorParam
{

    uint8_t bDescriptorType;
    uint16_t wDescriptorLength;

} __attribute__((__packed__));


struct UsbHidDescriptorHeader
{

    uint8_t bLength;
    uint8_t bDescriptorType;
    uint16_t bcdHid;
    uint8_t bCountryCode;
    uint8_t bNumDescriptors;

    UsbHidDescriptorParam descriptorParam[1];

} __attribute__((__packed__));

class UsbHidInterface
{
    NN_DISALLOW_COPY(UsbHidInterface);
    NN_DISALLOW_MOVE(UsbHidInterface);

public:

    UsbHidInterface() NN_NOEXCEPT{};
    ~UsbHidInterface() NN_NOEXCEPT{};

    Result Initialize(int index) NN_NOEXCEPT;
    Result Finalize() NN_NOEXCEPT;

    Result IncHandleSemaphore(uint32_t handle) NN_NOEXCEPT;
    Result DecHandleSemaphore() NN_NOEXCEPT;

    nn::usb::InterfaceHandle GetInterfaceHandle() NN_NOEXCEPT;

    Result ClientAcquireDevice(uint32_t handle) NN_NOEXCEPT;
    Result ClientReleaseDevice(uint32_t handle) NN_NOEXCEPT;
    Result HostAttachDevice(nn::usb::Host* pClient, nn::usb::InterfaceQueryOutput* pProfile, nn::ahid::hdr::Hdr* pHdr, int interfaceIndex) NN_NOEXCEPT;
    Result HostDetachDevice(nn::ahid::hdr::Hdr* pHdr) NN_NOEXCEPT;
    Result SetCtrlSession(nn::sf::SharedPointer<nn::ahid::ICtrlSession> pCtrlSession) NN_NOEXCEPT;

    Result AhidGetCodeBook(uint8_t* pBuffer, uint32_t bufferSize, int32_t deviceHandle) NN_NOEXCEPT;
    Result AhidGetString(uint8_t* pBuffer, uint32_t bufferSize, uint8_t stringIndex, int32_t deviceHandle) NN_NOEXCEPT;
    Result AhidGetReport(uint32_t reportType, uint32_t reportId, uint8_t* pBuffer, uint32_t bufferSize, int32_t deviceHandle) NN_NOEXCEPT;
    Result AhidSetReport(uint32_t reportType, uint32_t reportId, uint8_t* pBuffer, uint32_t bufferSize, int32_t deviceHandle) NN_NOEXCEPT;
    Result AhidGetIdle(uint8_t* pIdle, uint8_t reportId, int32_t deviceHandle) NN_NOEXCEPT;
    Result AhidSetIdle(uint8_t idle, uint8_t reportId, int32_t deviceHandle) NN_NOEXCEPT;
    Result AhidGetProtocol(uint8_t* pProtocol, int32_t deviceHandle) NN_NOEXCEPT;
    Result AhidSetProtocol(uint8_t protocol, int32_t deviceHandle) NN_NOEXCEPT;
    Result AhidGetDescriptor(uint8_t* pBuffer, uint32_t bufferSize, uint8_t bmRequestType, uint16_t wValue, uint16_t wIndex, int32_t deviceHandle) NN_NOEXCEPT;
    Result AhidSetDescriptor(uint8_t* pBuffer, uint32_t bufferSize, uint8_t bmRequestType, uint16_t wValue, uint16_t wIndex, int32_t deviceHandle) NN_NOEXCEPT;
    Result AhidRead(uint32_t* pBytesTransferred, uint8_t* pBuffer, uint32_t bytes, int32_t deviceHandle, nn::TimeSpan timeout) NN_NOEXCEPT;
    Result AhidWrite(uint32_t* pBytesTransferred, uint8_t* pBuffer, uint32_t bytes, int32_t deviceHandle, nn::TimeSpan timeout) NN_NOEXCEPT;


private:

    uint16_t                        m_HandleIndex;
    uint16_t                        m_HandleInstance;
    bool                            m_IsOccupied;
    bool                            m_IsAcquired;
    nn::ahid::hdr::DeviceHandle     m_HdrDeviceHandle;
    nn::usb::InterfaceQueryOutput   m_HsProfile;
    nn::usb::Host*                  m_pHsClient;
    nn::usb::HostInterface          m_HsSession;
    nn::usb::HostEndpoint           m_HsSessionRead;
    nn::usb::HostEndpoint           m_HsSessionWrite;
    UsbHidCodeBook                  m_CodeBook;
    uint16_t                        m_LanguageId;
    nn::sf::SharedPointer<nn::ahid::ICtrlSession>  m_CtrlSession;

    NN_ALIGNAS(4096) uint8_t        m_CtrlBuffer[4096];
    NN_ALIGNAS(4096) uint8_t        m_ReadBuffer[4096];
    NN_ALIGNAS(4096) uint8_t        m_WriteBuffer[4096];

    void ChangeHandle() NN_NOEXCEPT;

    Result StartHsSession(nn::usb::Host* pClient, nn::usb::InterfaceQueryOutput* pProfile) NN_NOEXCEPT;
    Result StopHsSession() NN_NOEXCEPT;
    Result AttachToHdr(nn::ahid::hdr::Hdr* pHdr, int i) NN_NOEXCEPT;
    Result DetachFromHdr(nn::ahid::hdr::Hdr* pHdr) NN_NOEXCEPT;

    Result GetDeviceString(uint8_t* pBuffer, uint32_t bufferSize, uint8_t stringIndex) NN_NOEXCEPT;
    Result GetHidDescriptor() NN_NOEXCEPT;
    Result Ctrl(size_t* pBytesTransferred, uint8_t* pBuffer, uint8_t bmRequestType, uint8_t bRequest, uint16_t wValue, uint16_t wIndex, uint16_t wLength) NN_NOEXCEPT;
    Result Read(size_t* pBytesTransferred, uint8_t* pBuffer, uint32_t bytes, nn::TimeSpan timeout) NN_NOEXCEPT;
    Result Write(size_t* pBytesTransferred, uint8_t* pBuffer, uint32_t bytes, nn::TimeSpan timeout) NN_NOEXCEPT;
};

} // end of namespace usb
} // end of namespace cdhid
} // end of namespace nn
