﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "cdacm_PrivateIncludes.h"

namespace nn {
namespace cdacm {
namespace driver {

template <class ManagedObjectType, uint32_t maxHandleCount>
class SafeHandleManager
{
public:
    typedef uint32_t          HandleType;
    typedef ManagedObjectType ObjectType;

public:
    SafeHandleManager()
        : m_Mutex(true)
        , m_Sequence(0)
    {
        for (auto& entry : m_HandleTable)
        {
            entry.handle       = InvalidHandleValue;
            entry.pObject      = nullptr;
            entry.refCount     = 0;
            entry.isDisabled   = false;
            entry.pDetachEvent = nullptr;
        }
    }

    ~SafeHandleManager()
    {
    }

    Result Register(ObjectType* pObject, UnitHandle *pOutHandle)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        for (uint32_t i = 0; i < MaxHandleCount; i++)
        {
            if (m_HandleTable[i].pObject == nullptr)
            {
                m_HandleTable[i].handle       = MakeHandle(i);
                m_HandleTable[i].pObject      = pObject;
                m_HandleTable[i].refCount     = 0;
                m_HandleTable[i].isDisabled   = false;
                m_HandleTable[i].pDetachEvent = nullptr;

                *pOutHandle = m_HandleTable[i].handle;

                return ResultSuccess();
            }
        }

        return ResultHandleAllocError();
    }

    void Unregister(UnitHandle handle)
    {
        // Make sure the handle cannot be acquired anymore
        Disable(handle);

        // Wait until all ref counts are released
        while (!IsIdle(handle))
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }

        // Now remove the handle
        {
            std::lock_guard<nn::os::Mutex> lock(m_Mutex);

            uint32_t index = ParseHandle(handle);

            if ((index < MaxHandleCount) && (m_HandleTable[index].handle == handle))
            {
                NN_CDACM_ABORT_UNLESS(m_HandleTable[index].refCount == 0);

                m_HandleTable[index].handle       = InvalidHandleValue;
                m_HandleTable[index].pObject      = nullptr;
                m_HandleTable[index].refCount     = 0;
                m_HandleTable[index].isDisabled   = false;
                m_HandleTable[index].pDetachEvent = nullptr;
            }
            else
            {
                NN_CDACM_ABORT_UNLESS_SUCCESS(ResultInvalidUnitHandle());
            }
        }
    }

    Result Discover(nn::os::SystemEventType *pDetachEvent, UnitHandle *pOutHandle)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        NN_CDACM_ABORT_IF_NULL(pDetachEvent);

        for (auto& entry : m_HandleTable)
        {
            if (entry.handle       != InvalidHandleValue &&
                entry.pDetachEvent == nullptr             )
            {
                *pOutHandle = entry.handle;
                entry.pDetachEvent = pDetachEvent;

                return ResultSuccess();
            }
        }

        return ResultDeviceNotAvailable();
    }

    Result GetFirst(UnitHandle *pOutHandle)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        for (auto& entry : m_HandleTable)
        {
            if (entry.handle != InvalidHandleValue)
            {
                *pOutHandle = entry.handle;

                return ResultSuccess();
            }
        }

        return ResultDeviceNotAvailable();
    }

    ObjectType* Acquire(HandleType handle)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        uint32_t index = ParseHandle(handle);

        if ((index < MaxHandleCount)                 &&
            (m_HandleTable[index].handle == handle)  &&
            (m_HandleTable[index].isDisabled == false))
        {
            NN_CDACM_ABORT_UNLESS(m_HandleTable[index].refCount < 0xffffffff);
            m_HandleTable[index].refCount++;
            return m_HandleTable[index].pObject;
        }
        else
        {
            return nullptr;
        }
    }

    void Release(HandleType handle)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        uint32_t index = ParseHandle(handle);

        if ((index < MaxHandleCount) && (m_HandleTable[index].handle == handle))
        {
            NN_CDACM_ABORT_UNLESS(m_HandleTable[index].refCount > 0);
            m_HandleTable[index].refCount--;
        }
        else
        {
            NN_CDACM_ABORT_UNLESS_SUCCESS(ResultInvalidUnitHandle());
        }
    }

private:
    void Disable(UnitHandle handle)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        uint32_t index = ParseHandle(handle);

        if ((index < MaxHandleCount) && (m_HandleTable[index].handle == handle))
        {
            m_HandleTable[index].isDisabled = true;
        }
        else
        {
            NN_CDACM_ABORT_UNLESS_SUCCESS(ResultInvalidUnitHandle());
        }
    }

    bool IsIdle(HandleType handle)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        uint32_t index = ParseHandle(handle);

        if ((index < MaxHandleCount) && (m_HandleTable[index].handle == handle))
        {
            return m_HandleTable[index].refCount == 0;
        }
        else
        {
            NN_CDACM_ABORT_UNLESS_SUCCESS(ResultInvalidUnitHandle());
            return false;
        }
    }

private:
    const uint32_t   HandleBitCount     = sizeof(HandleType) * 8;
    const uint32_t   IndexBitCount      = HandleBitCount / 2;
    const uint32_t   SequenceBitCount   = HandleBitCount - IndexBitCount -  1;
    const uint32_t   IndexMask          = 0xffffffff >> (32 - IndexBitCount);
    const uint32_t   SequenceMask       = 0xffffffff >> (32 - SequenceBitCount);

    const uint32_t   MaxHandleCount     = maxHandleCount;
    const HandleType InvalidHandleValue = 0;

    struct HandleTableEntry
    {
        HandleType               handle;
        ObjectType              *pObject;
        uint32_t                 refCount;
        bool                     isDisabled;
        nn::os::SystemEventType *pDetachEvent;
    };

    HandleType MakeHandle(uint32_t index)
    {
        NN_CDACM_ABORT_UNLESS(index < MaxHandleCount);

        HandleType handle = (HandleType)(
            (1                           << (HandleBitCount - 1)) |  // valid
            ((m_Sequence & SequenceMask) << IndexBitCount)        |  // sequence
            ((index      & IndexMask   ) << 0)                       // index
        );

        m_Sequence++;

        return handle;
    }

    uint32_t ParseHandle(HandleType handle)
    {
        return (uint32_t)(
            (uint32_t)handle & IndexMask
        );
    }

    nn::os::Mutex    m_Mutex;
    uint32_t         m_Sequence;
    HandleTableEntry m_HandleTable[maxHandleCount];
};

} // driver
} // cdacm
} // nn
