﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cdacm_Device.h"
#include "cdacm_Driver.h"

namespace nn {
namespace cdacm {

namespace driver {

Device::Device()
    : m_DeviceState(DeviceState_Uninitialized)
    , m_pRS232(nullptr)
    , m_Mutex(false)
    , m_Handle(0)
{
}

Device::~Device()
{
}

Result
Device::Initialize(nn::usb::Host                  *pUsbHost,
                   nn::usb::InterfaceQueryOutput  *pProfile,
                   nn::os::SystemEventType       **pDetachEventPtr) NN_NOEXCEPT
{
    Result   result   = ResultSuccess();

    m_pUsbHost = pUsbHost;
    m_Profile = *pProfile;

    NN_CDACM_INFO("Device Detected:\n  VID        0x%04x\n  PID        0x%04x\n",
        m_Profile.deviceProfile.deviceDesc.idVendor,
        m_Profile.deviceProfile.deviceDesc.idProduct);
    // This may rightfully fail if the device has since disconnected
    NN_CDACM_INFO("Calling  UsbInterface Initialize (ifc 0x%x, host 0x%x, handle 0x%x\n",
        &m_UsbInterface, m_pUsbHost, m_Profile.ifProfile.handle);

    NN_CDACM_DO(m_UsbInterface.Initialize(m_pUsbHost, m_Profile.ifProfile.handle));

    if (result.IsFailure())
    {
        goto Exit;
    }

    *pDetachEventPtr = m_UsbInterface.GetStateChangeEvent();
    m_DeviceState = DeviceState_Initialized;

Exit:
    return result;
}

void
Device::GetProfile(UnitProfile *pOutProfile) const NN_NOEXCEPT
{
    pOutProfile->handle = m_Handle;
    pOutProfile->vid = m_Profile.deviceProfile.deviceDesc.idVendor;
    pOutProfile->pid = m_Profile.deviceProfile.deviceDesc.idProduct;
}

Result
Device::InitRS232(CdAcmParameters * parameters) NN_NOEXCEPT
{
    Result result;

    if (m_Profile.deviceProfile.deviceDesc.idVendor
                == SupportedUarts[UartVendorIndex_Ftdi][FieldIndex_Vid]
        && m_Profile.deviceProfile.deviceDesc.idProduct
                == SupportedUarts[UartVendorIndex_Ftdi][FieldIndex_Pid])
    {
        m_pRS232 = new (&m_RS232Memory) FtdiRS232();
    }
    else if (m_Profile.deviceProfile.deviceDesc.idVendor
        == SupportedUarts[UartVendorIndex_Prolific][FieldIndex_Vid]
        && m_Profile.deviceProfile.deviceDesc.idProduct
        == SupportedUarts[UartVendorIndex_Prolific][FieldIndex_Pid])
    {
        m_pRS232 = new (&m_RS232Memory) ProlificRS232();
    }

    NN_CDACM_DO(m_pRS232->Initialize(&m_UsbInterface, parameters));
    if (result.IsFailure())
    {
        NN_CDACM_WARN("RS232.Initialize Failure: %d:%d\n",
            result.GetModule(), result.GetDescription());
        m_pRS232->~RS232();
        m_pRS232 = nullptr;
        goto Exit;
    }

Exit:
    return result;
}

void
Device::SetHandle(UnitHandle handle) NN_NOEXCEPT
{
    m_Handle = handle;
}

UnitHandle
Device::GetHandle() const NN_NOEXCEPT
{
    return m_Handle;
}

Result
Device::Finalize() NN_NOEXCEPT
{
    Result result       = ResultSuccess();
    Result returnResult = ResultSuccess();

    m_DeviceState = DeviceState_Uninitialized;

    // Issue a device reset, cancel any pending transactions
    NN_CDACM_DO(Detach());
    if (result.IsFailure())
    {
        returnResult = result;
    }

    if (m_pRS232)
    {
        NN_CDACM_DO(m_pRS232->Finalize());
        if (result.IsFailure() && returnResult.IsSuccess())
        {
            returnResult = result;
        }
        m_pRS232->~RS232();
        m_pRS232 = nullptr;
    }
    NN_CDACM_DO(m_UsbInterface.Finalize());
    if (result.IsFailure() && returnResult.IsSuccess())
    {
        returnResult = result;
    }

    return returnResult;
}

Result
Device::Detach() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    result = m_UsbInterface.ResetDevice();

    if (nn::usb::ResultInterfaceInvalid::Includes(result))
    {
        result = ResultSuccess();
    }

    return result;
}

Result
Device::Read(size_t *pBytesXferred, void *buffer, uint32_t size) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    result = m_pRS232->Read(pBytesXferred, buffer, size);

    if (ResultDeviceNotAvailable::Includes(result))
    {
        // FIXME: error handling?
        Detach();
    }

    return result;
}

Result
Device::Write(size_t *pBytesXferred, const void *pOutBuffer, uint32_t length) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    result =  m_pRS232->Write(pBytesXferred, pOutBuffer, length);

    if (ResultDeviceNotAvailable::Includes(result))
    {
        // FIXME: error handling?
        Detach();
    }

    return result;
}

Result
Device::WriteAsync(
    uint32_t                 *pOutXferId,
    nn::os::SystemEventType **pCompletionEventPtr,
    const void               *pOutBuffer,
    uint32_t                  length) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    result = m_pRS232->WriteAsync(pOutXferId, pCompletionEventPtr, pOutBuffer, length);

    if (ResultDeviceNotAvailable::Includes(result))
    {
        // FIXME: error handling?
        Detach();
    }

    return result;

}

Result
Device::GetWriteAsyncResult(
    uint32_t            *pOutCount,
    nn::usb::XferReport *pOutReport,
    uint32_t             count) NN_NOEXCEPT
{
    return m_pRS232->GetWriteAsyncResult(pOutCount, pOutReport, count);
}

enum DeviceState Device::GetDeviceState() NN_NOEXCEPT
{
    return m_DeviceState;
}

} // end of namespace driver
} // end of namespace cdacm
} // end of namespace nn
