﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_StaticAssert.h>

#include <nn/settings/factory/settings_Bluetooth.h>
#include <nn/settings/factory/settings_ConfigurationId.h>
#include <nn/settings/factory/settings_MotionSensor.h>
#include <nn/settings/factory/settings_WirelessLan.h>
#include <nn/settings/factory/settings_SerialNumber.h>
#include <nn/settings/factory/settings_BatteryLot.h>

#include <nn/cal/cal_Types.h>

namespace nn { namespace cal {

//!< 生産時較正情報のマジックナンバー
const uint32_t CalibrationInfoMagicNumber =
    'C' << 0 | 'A' << 8 | 'L' << 16 | '0' << 24;

//!< 生産時較正情報のフォーマットバージョン
const uint32_t CalibrationInfoFormatVersion = 13;

//!< 生産時較正情報の機種情報
const uint32_t CalibrationInfoModel = 1;

//!< 秘密鍵の共通鍵暗号化に使用されている鍵の世代
const uint32_t CalibrationInfoCommonKeyGeneration = 4;

//!< SHA-256 ハッシュを表す構造体です。
struct Sha256Hash final
{
    ::nn::Bit8 data[32];
};

//!< 暗号化に使用する鍵の世代を表す構造体です。
struct EncryptionKeyGeneration final
{
    uint32_t generation;

    EncryptionKeyGeneration& operator=(const uint32_t rhs)
    {
        generation = rhs;
        return *this;
    }

    bool IsZero() const NN_NOEXCEPT
    {
        return (generation == 0);
    }
};

//!< 生産時較正情報のヘッダーを表す構造体です。
struct CalibrationInfoHeader final
{
    uint32_t magicNumber;   //!< 生産時較正情報のマジックナンバー
    uint32_t version;       //!< 生産時較正情報のフォーマットバージョン
    uint32_t bodySize;      //!< 生産時較正情報の本体のバイト数
    uint16_t model;         //!< 生産時較正情報の機種情報
    uint16_t updateCount;   //!< 生産時較正情報の書き換え回数
    char padding[14];
    ::nn::Bit16 crc16;      //!< 生産時較正情報のヘッダーの CRC-16
    Sha256Hash bodyHash;    //!< 生産時較正情報の本体の SHA-256 ハッシュ
};

//!< 構成識別子 1 のブロックを表す構造体です。
struct ConfigurationId1Block final
{
    //!< 構成識別子 1
    ::nn::settings::factory::ConfigurationId1 configurationId1;

    //!< 構成識別子 1 の CRC-16
    ::nn::Bit16 crc16;
};

//!< 無線 LAN に対して設定されたカントリーコードのブロックを表す構造体です。
struct WirelessLanCountryCodesBlock final
{
    //!< 無線 LAN に対して設定されたカントリーコードの数
    uint32_t count;

    char padding1[4];

    //!< 無線 LAN に対して設定されたカントリーコード
    ::nn::settings::factory::CountryCode countryCodes[128];

    char padding2[6];

    //!< 無線 LAN に対して設定されたカントリーコードの CRC-16
    ::nn::Bit16 crc16;
};

//!< 無線 LAN に対して設定された MAC アドレスのブロックを表す構造体です。
struct WirelessLanMacAddressBlock final
{
    //!< 無線 LAN に対して設定された MAC アドレス
    ::nn::settings::factory::MacAddress macAddress;

    //!< 無線 LAN に対して設定された MAC アドレスの CRC-16
    ::nn::Bit16 crc16;
};

//!< Bluetooth に対して設定された BD アドレスのブロックを表す構造体です。
struct BluetoothBdAddressBlock final
{
    //!< Bluetooth に対して設定された BD アドレス
    ::nn::settings::factory::BdAddress bdAddress;

    //!< Bluetooth に対して設定された BD アドレスの CRC-16
    ::nn::Bit16 crc16;
};

//!< 加速度センサのオフセット値のブロックを表す構造体です。
struct AccelerometerOffsetBlock final
{
    //!< 加速度センサのオフセット値
    ::nn::settings::factory::AccelerometerOffset accelerometerOffset;

    //!< 加速度センサのオフセット値の CRC-16
    ::nn::Bit16 crc16;
};

//!< 加速度センサのスケール値のブロックを表す構造体です。
struct AccelerometerScaleBlock final
{
    //!< 加速度センサのスケール値
    ::nn::settings::factory::AccelerometerScale accelerometerScale;

    //!< 加速度センサのスケール値の CRC-16
    ::nn::Bit16 crc16;
};

//!< ジャイロスコープのオフセット値のブロックを表す構造体です。
struct GyroscopeOffsetBlock final
{
    //!< ジャイロスコープのオフセット値
    ::nn::settings::factory::GyroscopeOffset gyroscopeOffset;

    //!< ジャイロスコープのオフセット値の CRC-16
    ::nn::Bit16 crc16;
};

//!< ジャイロスコープのスケール値のブロックを表す構造体です。
struct GyroscopeScaleBlock final
{
    //!< ジャイロスコープのスケール値
    ::nn::settings::factory::GyroscopeScale gyroscopeScale;

    //!< ジャイロスコープのスケール値の CRC-16
    ::nn::Bit16 crc16;
};

//!< 製品シリアル番号のブロックを表す構造体です。
struct SerialNumberBlock final
{
    //!< 製品シリアル番号
    ::nn::settings::factory::SerialNumber serialNumber;

    char padding[6];

    //!< 製品シリアル番号の CRC-16
    ::nn::Bit16 crc16;
};

//!< ECC-P256 版のデバイス登録用デバイス秘密鍵のブロックを表す構造体です。
struct EccP256DeviceKeyBlock final
{
    //!< ECC-P256 版のデバイス登録用デバイス秘密鍵
    EccP256DeviceKey deviceKey;

    char padding[14];

    //!< ECC-P256 版のデバイス登録用デバイス秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

//!< ECC-P256 版のデバイス登録用デバイス証明書のブロックを表す構造体です。
struct EccP256DeviceCertificateBlock final
{
    //!< ECC-P256 版のデバイス登録用デバイス証明書
    DeviceCertificate deviceCertificate;

    char padding[14];

    //!< ECC-P256 版のデバイス登録用デバイス証明書の CRC-16
    ::nn::Bit16 crc16;
};

//!< ECC-B233 版のデバイス登録用デバイス秘密鍵のブロックを表す構造体です。
struct EccB233DeviceKeyBlock final
{
    //!< ECC-B233 版のデバイス登録用デバイス秘密鍵
    EccB233DeviceKey deviceKey;

    char padding[14];

    //!< ECC-B233 版のデバイス登録用デバイス秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

//!< ECC-B233 版のデバイス登録用デバイス証明書のブロックを表す構造体です。
struct EccB233DeviceCertificateBlock final
{
    //!< ECC-B233 版のデバイス登録用デバイス証明書
    DeviceCertificate deviceCertificate;

    char padding[14];

    //!< ECC-B233 版のデバイス登録用デバイス証明書の CRC-16
    ::nn::Bit16 crc16;
};

//!< ECC-P256 版の eTicket 発行用デバイス秘密鍵のブロックを表す構造体です。
struct EccP256ETicketKeyBlock final
{
    //!< ECC-P256 版の eTicket 発行用デバイス秘密鍵
    EccP256DeviceKey deviceKey;

    char padding[14];

    //!< ECC-P256 版の eTicket 発行用デバイス秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

//!< ECC-P256 版の eTicket 発行用デバイス証明書のブロックを表す構造体です。
struct EccP256ETicketCertificateBlock final
{
    //!< ECC-P256 版の eTicket 発行用デバイス証明書
    DeviceCertificate deviceCertificate;

    char padding[14];

    //!< ECC-P256 版の eTicket 発行用デバイス証明書の CRC-16
    ::nn::Bit16 crc16;
};

//!< ECC-B233 版の eTicket 発行用デバイス秘密鍵のブロックを表す構造体です。
struct EccB233ETicketKeyBlock final
{
    //!< ECC-B233 版の eTicket 発行用デバイス秘密鍵
    EccB233DeviceKey deviceKey;

    char padding[14];

    //!< ECC-B233 版の eTicket 発行用デバイス秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

//!< ECC-B233 版の eTicket 発行用デバイス証明書のブロックを表す構造体です。
struct EccB233ETicketCertificateBlock final
{
    //!< ECC-B233 版の eTicket 発行用デバイス証明書
    DeviceCertificate deviceCertificate;

    char padding[14];

    //!< ECC-B233 版の eTicket 発行用デバイス証明書の CRC-16
    ::nn::Bit16 crc16;
};

//!< SSL クライアント秘密鍵のブロックを表す構造体です。
struct SslKeyBlock final
{
    //!< SSL クライアント秘密鍵
    SslKey sslKey;

    char padding[14];

    //!< SSL クライアント秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

//!< SSL クライアント証明書のバイト数のブロックを表す構造体です。
struct SslCertificateSizeBlock final
{
    //!< SSL クライアント証明書のバイト数
    uint64_t sslCertificateSize;

    char padding[6];

    //!< SSL クライアント証明書のバイト数の CRC-16
    ::nn::Bit16 crc16;
};

//!< SSL クライアント証明書のブロックを表す構造体です。
struct SslCertificateBlock final
{
    //!< SSL クライアント証明書
    SslCertificate sslCertificate;

    //!< SSL クライアント証明書の SHA-256 ハッシュ
    Sha256Hash sslCertificateHash;
};

//!< 乱数のブロックを表す構造体です。
struct RandomNumberBlock final
{
    //!< 乱数
    RandomNumber randomNumber;

    //!< 乱数の SHA-256 ハッシュ
    Sha256Hash randomNumberHash;
};

//!< ゲームカード相互認証秘密鍵のブロックを表す構造体です。
struct GameCardKeyBlock final
{
    //!< ゲームカード相互認証秘密鍵
    GameCardKey gameCardKey;

    char padding[14];

    //!< ゲームカード相互認証秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

//!< ゲームカード相互認証証明書のブロックを表す構造体です。
struct GameCardCertificateBlock final
{
    //!< ゲームカード相互認証証明書
    GameCardCertificate gameCardCertificate;

    //!< ゲームカード相互認証証明書の SHA-256 ハッシュ
    Sha256Hash gameCardCertificateHash;
};

//!< RSA-2048 版の eTicket 発行用デバイス秘密鍵のブロックを表す構造体です。
struct Rsa2048ETicketKeyBlock final
{
    //!< RSA-2048 版の eTicket 発行用デバイス秘密鍵
    Rsa2048ETicketKey rsa2048ETicketKey;

    char padding[14];

    ::nn::Bit16 crc16;
};

//!< RSA-2048 版の eTicket 発行用デバイス証明書のブロックを表す構造体です。
struct Rsa2048ETicketCertificateBlock final
{
    //!< RSA-2048 版の eTicket 発行用デバイス証明書
    Rsa2048ETicketCertificate rsa2048ETicketCertificate;

    char padding[14];

    ::nn::Bit16 crc16;
};

//!< 電池 LOT のブロックを表す構造体です。
struct BatteryLotBlock final
{
    //!< 電池 LOT
    ::nn::settings::factory::BatteryLot batteryLot;

    char padding[6];

    ::nn::Bit16 crc16;
};

//!< スピーカー補正値のブロックを表す構造体です。
struct SpeakerCalibrationValueBlock final
{
    //スピーカー補正値
    SpeakerCalibrationValue speakerCalibrationValue;

    //残りの予約領域
    ::nn::Bit8 reserved[1958];

    char padding[14];

    ::nn::Bit16 crc16;
};

//!< リージョンコードのブロックを表す構造体です。
struct RegionCodeBlock final
{
    //リージョンコード
    uint32_t regionCode;

    char padding[10];

    ::nn::Bit16 crc16;
};

//!< amiibo 用本体秘密鍵のブロックを表す構造体です。
struct AmiiboKeyBlock final
{
    //amiibo 用本体秘密鍵
    AmiiboKey amiiboKey;

    //amiibo 用本体秘密鍵を暗号化する鍵の世代
    EncryptionKeyGeneration encryptionKeyGeneration;

    char padding[10];

    ::nn::Bit16 crc16;
};

//!< amiibo 用 ECQV 証明書のブロックを表す構造体です。
struct AmiiboEcqvCertificateBlock final
{
    //amiibo 用 ECQV 証明書
    AmiiboEcqvCertificate amiiboEcqvCertificate;

    char padding[10];

    ::nn::Bit16 crc16;
};

//!< amiibo 用 ECDSA 証明書のブロックを表す構造体です。
struct AmiiboEcdsaCertificateBlock final
{
    //amiibo 用 ECDSA 証明書
    AmiiboEcdsaCertificate amiiboEcdsaCertificate;

    char padding[14];

    ::nn::Bit16 crc16;
};

//!< amiibo 用本体秘密鍵 (ECQV-BLS) のブロックを表す構造体です。
struct AmiiboEcqvBlsKeyBlock final
{
    //amiibo 用本体秘密鍵 (ECQV-BLS)
    AmiiboEcqvBlsKey amiiboEcqvBlsKey;

    //amiibo 用本体秘密鍵 (ECQV-BLS) を暗号化する鍵の世代
    EncryptionKeyGeneration encryptionKeyGeneration;

    char padding[10];

    ::nn::Bit16 crc16;
};

//!< amiibo 用 ECQV-BLS 証明書のブロックを表す構造体です。
struct AmiiboEcqvBlsCertificateBlock final
{
    //amiibo 用 ECQV-BLS 証明書
    AmiiboEcqvBlsCertificate amiiboEcqvBlsCertificate;

    char padding[14];

    ::nn::Bit16 crc16;
};

//!< amiibo 用 ECQV-BLS ルート証明書のブロックを表す構造体です。
struct AmiiboEcqvBlsRootCertificateBlock final
{
    //amiibo 用 ECQV-BLS ルート証明書
    AmiiboEcqvBlsRootCertificate amiiboEcqvBlsRootCertificate;

    char padding[14];

    ::nn::Bit16 crc16;
};

//!< 本体モデルタイプのブロックを表す構造体です。
struct ProductModelBlock final
{
    //本体モデルタイプ
    ProductModel productModel;

    char padding[10];

    ::nn::Bit16 crc16;
};

//!< 本体カラーバリエーションのブロックを表す構造体です。
struct ColorVariationBlock final
{
    //本体カラーバリエーション
    ColorVariation colorVariation;

    char padding[8];

    ::nn::Bit16 crc16;
};

//!< LCD バックライト出力の写像の係数のブロックを表す構造体です。
struct LcdBacklightBrightnessMappingBlock final
{
    //LCD バックライト出力の写像の係数
    LcdBacklightBrightnessMapping lcdBacklightBrightnessMapping;

    char padding[2];

    ::nn::Bit16 crc16;
};

//!< 領域拡張された ECC-B233 版のデバイス登録用デバイス秘密鍵のブロックを表す構造体です。
struct ExtendedEccB233DeviceKeyBlock final
{
    //!< 領域拡張された ECC-B233 版のデバイス登録用デバイス秘密鍵
    ExtendedEccB233DeviceKey extendedDeviceKey;

    //!< 領域拡張された ECC-B233 版のデバイス登録用デバイス秘密鍵を暗号化する鍵の世代
    EncryptionKeyGeneration encryptionKeyGeneration;

    char padding[10];

    //!< 領域拡張された ECC-B233 版のデバイス登録用デバイス秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

//!< 領域拡張された ECC-P256 版の eTicket 発行用デバイス秘密鍵のブロックを表す構造体です。
struct ExtendedEccP256ETicketKeyBlock final
{
    //!< 領域拡張された ECC-P256 版の eTicket 発行用デバイス秘密鍵
    ExtendedEccP256DeviceKey extendedDeviceKey;

    char padding[14];

    //!< 領域拡張された ECC-P256 版の eTicket 発行用デバイス秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

//!< 領域拡張された ECC-B233 版の eTicket 発行用デバイス秘密鍵のブロックを表す構造体です。
struct ExtendedEccB233ETicketKeyBlock final
{
    //!< 領域拡張された ECC-B233 版の eTicket 発行用デバイス秘密鍵
    ExtendedEccB233DeviceKey extendedDeviceKey;

    char padding[14];

    //!< 領域拡張された ECC-B233 版の eTicket 発行用デバイス秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

//!< 領域拡張された RSA-2048 版の eTicket 発行用デバイス秘密鍵のブロックを表す構造体です。
struct ExtendedRsa2048ETicketKeyBlock final
{
    //!< 領域拡張された RSA-2048 版の eTicket 発行用デバイス秘密鍵
    ExtendedRsa2048ETicketKey extendedRsa2048ETicketKey;

    //!< 領域拡張された RSA-2048 版の eTicket 発行用デバイス秘密鍵を暗号化する鍵の世代
    EncryptionKeyGeneration encryptionKeyGeneration;

    char padding[10];

    ::nn::Bit16 crc16;
};

//!< 領域拡張された SSL クライアント秘密鍵のブロックを表す構造体です。
struct ExtendedSslKeyBlock final
{
    //!< 領域拡張された SSL クライアント秘密鍵
    ExtendedSslKey extendedSslKey;

    //!< 領域拡張された SSL クライアント秘密鍵を暗号化する鍵の世代
    EncryptionKeyGeneration encryptionKeyGeneration;

    char padding[10];

    ::nn::Bit16 crc16;
};

//!< 領域拡張されたゲームカード相互認証秘密鍵のブロックを表す構造体です。
struct ExtendedGameCardKeyBlock final
{
    //!< ゲームカード相互認証秘密鍵
    ExtendedGameCardKey extendedGameCardKey;

    //!< ゲームカード相互認証秘密鍵を暗号化する鍵の世代
    EncryptionKeyGeneration encryptionKeyGeneration;

    char padding[10];

    //!< ゲームカード相互認証秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

//!< LCD ベンダ ID のブロックを表す構造体です。
struct LcdVendorIdBlock final
{
    //LCD ベンダ ID
    LcdVendorId lcdVendorId;

    char padding[10];

    ::nn::Bit16 crc16;
};

//!< 領域拡張された RSA-2048 版のデバイス登録用デバイス秘密鍵のブロックを表す構造体です。
struct ExtendedRsa2048DeviceKeyBlock final
{
    //!< 領域拡張された RSA-2048 版のデバイス登録用デバイス秘密鍵
    ExtendedRsa2048DeviceKey extendedDeviceKey;

    //!< 領域拡張された RSA-2048 版のデバイス登録用デバイス秘密鍵を暗号化する鍵の世代
    EncryptionKeyGeneration encryptionKeyGeneration;

    char padding[10];

    //!< 領域拡張された RSA-2048 版のデバイス登録用デバイス秘密鍵の CRC-16
    ::nn::Bit16 crc16;
};

//!< RSA-2048 版のデバイス登録用デバイス証明書のブロックを表す構造体です。
struct Rsa2048DeviceCertificateBlock final
{
    //!< RSA-2048 版のデバイス登録用デバイス証明書
    Rsa2048DeviceCertificate rsa2048DeviceCertificate;

    char padding[14];

    ::nn::Bit16 crc16;
};

//!< USB Type-C Power Source 回路バージョンのブロックを表す構造体です。
struct UsbTypeCPowerSourceCircuitBlock final
{
    UsbTypeCPowerSourceCircuit usbTypeCPowerSourceCircuit;

    char padding[13];

    ::nn::Bit16 crc16;
};

//!< 本体サブカラーのブロックを表す構造体です。
struct HousingSubColorBlock final
{
    HousingColor housingColor;

    char padding[10];

    ::nn::Bit16 crc16;
};

//!< 本体液晶縁カラーのブロックを表す構造体です。
struct HousingBezelColorBlock final
{
    HousingColor housingColor;

    char padding[10];

    ::nn::Bit16 crc16;
};

//!< 本体カラー 1 のブロックを表す構造体です。
struct HousingMainColor1Block final
{
    HousingColor housingColor;

    char padding[10];

    ::nn::Bit16 crc16;
};

//!< 本体カラー 2 のブロックを表す構造体です。
struct HousingMainColor2Block final
{
    HousingColor housingColor;

    char padding[10];

    ::nn::Bit16 crc16;
};

//!< 本体カラー 3 のブロックを表す構造体です。
struct HousingMainColor3Block final
{
    HousingColor housingColor;

    char padding[10];

    ::nn::Bit16 crc16;
};

//!< 生産時較正情報の本体を表す構造体です。
struct CalibrationInfoBody final
{
    //!< 構成識別子 1 のブロック
    ConfigurationId1Block configurationId1Block;

    char padding1[32];

    //!< 無線 LAN に対して設定されたカントリーコードのブロック
    WirelessLanCountryCodesBlock wirelessLanCountryCodesBlock;

    //!< 無線 LAN に対して設定された MAC アドレスのブロック
    WirelessLanMacAddressBlock wirelessLanMacAddressBlock;

    char padding2[8];

    //!< Bluetooth に対して設定された BD アドレスのブロック
    BluetoothBdAddressBlock bluetoothBdAddressBlock;

    char padding3[8];

    //!< 加速度センサのオフセット値のブロック
    AccelerometerOffsetBlock accelerometerOffsetBlock;

    //!< 加速度センサのスケール値のブロック
    AccelerometerScaleBlock accelerometerScaleBlock;

    //!< ジャイロスコープのオフセット値のブロック
    GyroscopeOffsetBlock gyroscopeOffsetBlock;

    //!< ジャイロスコープのスケール値のブロック
    GyroscopeScaleBlock gyroscopeScaleBlock;

    //!< 製品シリアル番号のブロック
    SerialNumberBlock serialNumberBlock;

    //!< ECC-P256 版のデバイス登録用デバイス秘密鍵のブロック
    EccP256DeviceKeyBlock eccP256DeviceKeyBlock;

    //!< ECC-P256 版のデバイス登録用デバイス証明書のブロック
    EccP256DeviceCertificateBlock eccP256DeviceCertificateBlock;

    //!< ECC-B233 版のデバイス登録用デバイス秘密鍵のブロック
    EccB233DeviceKeyBlock eccB233DeviceKeyBlock;

    //!< ECC-B233 版のデバイス登録用デバイス証明書のブロック
    EccB233DeviceCertificateBlock eccB233DeviceCertificateBlock;

    //!< ECC-P256 版の eTicket 発行用デバイス秘密鍵のブロック
    EccP256ETicketKeyBlock eccP256ETicketKeyBlock;

    //!< ECC-P256 版の eTicket 発行用デバイス証明書のブロック
    EccP256ETicketCertificateBlock eccP256ETicketCertificateBlock;

    //!< ECC-B233 版の eTicket 発行用デバイス秘密鍵のブロック
    EccB233ETicketKeyBlock eccB233ETicketKeyBlock;

    //!< ECC-B233 版の eTicket 発行用デバイス証明書のブロック
    EccB233ETicketCertificateBlock eccB233ETicketCertificateBlock;

    //!< SSL クライアント秘密鍵のブロック
    SslKeyBlock sslKeyBlock;

    //!< SSL クライアント証明書のバイト数のブロック
    SslCertificateSizeBlock sslCertificateSizeBlock;

    //!< SSL クライアント証明書のブロック
    SslCertificateBlock sslCertificateBlock;

    //!< 乱数のブロック
    RandomNumberBlock randomNumberBlock;

    //!< ゲームカード相互認証秘密鍵
    GameCardKeyBlock gameCardKeyBlock;

    //!< ゲームカード相互認証証明書のブロック
    GameCardCertificateBlock gameCardCertificateBlock;

    //!< RSA-2048 版の eTicket 発行用デバイス秘密鍵のブロック
    Rsa2048ETicketKeyBlock rsa2048ETicketKeyBlock;

    //!< RSA-2048 版の eTicket 発行用デバイス証明書のブロック
    Rsa2048ETicketCertificateBlock rsa2048ETicketCertificateBlock;

    //!< 電池 LOT のブロック
    BatteryLotBlock batteryLotBlock;

    //!< スピーカー補正値のブロック
    SpeakerCalibrationValueBlock speakerCalibrationValueBlock;

    //!< リージョンコードのブロック
    RegionCodeBlock regionCodeBlock;

    //!< amiibo 用本体秘密鍵のブロック
    AmiiboKeyBlock amiiboKeyBlock;

    //!< amiibo 用 ECQV 証明書のブロック
    AmiiboEcqvCertificateBlock amiiboEcqvCertificateBlock;

    //!< amiibo 用 ECDSA 証明書のブロック
    AmiiboEcdsaCertificateBlock amiiboEcdsaCertificateBlock;

    //!< amiibo 用本体秘密鍵 (ECQV-BLS) のブロック
    AmiiboEcqvBlsKeyBlock amiiboEcqvBlsKeyBlock;

    //!< amiibo 用 ECQV-BLS 証明書のブロック
    AmiiboEcqvBlsCertificateBlock amiiboEcqvBlsCertificateBlock;

    //!< amiibo 用 ECQV-BLS ルート証明書のブロック
    AmiiboEcqvBlsRootCertificateBlock amiiboEcqvBlsRootCertificateBlock;

    //!< 本体モデルタイプのブロック
    ProductModelBlock productModelBlock;

    //!< 本体カラーバリエーションのブロック
    ColorVariationBlock colorVariationBlock;

    //!< LCD バックライト出力の写像の係数のブロック
    LcdBacklightBrightnessMappingBlock lcdBacklightBrightnessMappingBlock;

    //!< 領域拡張された ECC-B233 版のデバイス登録用デバイス秘密鍵のブロック
    ExtendedEccB233DeviceKeyBlock extendedEccB233DeviceKeyBlock;

    //!< 領域拡張された ECC-P256 版の eTicket 発行用デバイス秘密鍵のブロックを表す構造体です。
    ExtendedEccP256ETicketKeyBlock extendedEccP256ETicketKeyBlock;

    //!< 領域拡張された ECC-B233 版の eTicket 発行用デバイス秘密鍵のブロックを表す構造体です。
    ExtendedEccB233ETicketKeyBlock extendedEccB233ETicketKeyBlock;

    //!< 領域拡張された RSA-2048 版の eTicket 発行用デバイス秘密鍵のブロックを表す構造体です。
    ExtendedRsa2048ETicketKeyBlock extendedRsa2048ETicketKeyBlock;

    //!< 領域拡張された SSL クライアント秘密鍵のブロックを表す構造体です。
    ExtendedSslKeyBlock  extendedSslKeyBlock;

    //!< 領域拡張されたゲームカード相互認証秘密鍵のブロックを表す構造体です。
    ExtendedGameCardKeyBlock extendedGameCardKeyBlock;

    //!< LCD ベンダ ID のブロック
    LcdVendorIdBlock lcdVendorIdBlock;

    //!< 領域拡張された RSA-2048 版のデバイス登録用デバイス秘密鍵のブロックを表す構造体です。
    ExtendedRsa2048DeviceKeyBlock extendedRsa2048DeviceKeyBlock;

    //!< RSA-2048 版のデバイス登録用デバイス証明書のブロック
    Rsa2048DeviceCertificateBlock rsa2048DeviceCertificateBlock;

    //!< USB Type-C Power Source 回路バージョンのブロック
    UsbTypeCPowerSourceCircuitBlock usbTypeCPowerSourceCircuitBlock;

    //!< 本体サブカラーのブロック
    HousingSubColorBlock housingSubColorBlock;

    //!< 本体液晶縁カラーのブロック
    HousingBezelColorBlock housingBezelColorBlock;

    //!< 本体カラー 1 のブロック
    HousingMainColor1Block housingMainColor1Block;

    //!< 本体カラー 2 のブロック
    HousingMainColor2Block housingMainColor2Block;

    //!< 本体カラー 3 のブロック
    HousingMainColor3Block housingMainColor3Block;

    //!< 予約領域
    ::nn::Bit8 reserved[16 * 985];
};

//!< 生産時較正情報を表す構造体です。
struct CalibrationInfo final
{
    CalibrationInfoHeader header;   //!< 生産時較正情報のヘッダー
    CalibrationInfoBody body;       //!< 生産時較正情報の本体
};

NN_STATIC_ASSERT(sizeof(CalibrationInfoHeader) == 16 * 4);
NN_STATIC_ASSERT(sizeof(ConfigurationId1Block) == 16 * 2);
NN_STATIC_ASSERT(sizeof(WirelessLanCountryCodesBlock) == 16 * 25);
NN_STATIC_ASSERT(sizeof(WirelessLanMacAddressBlock) == 8);
NN_STATIC_ASSERT(sizeof(BluetoothBdAddressBlock) == 8);
NN_STATIC_ASSERT(sizeof(AccelerometerOffsetBlock) == 8);
NN_STATIC_ASSERT(sizeof(AccelerometerScaleBlock) == 8);
NN_STATIC_ASSERT(sizeof(GyroscopeOffsetBlock) == 8);
NN_STATIC_ASSERT(sizeof(GyroscopeScaleBlock) == 8);
NN_STATIC_ASSERT(sizeof(SerialNumberBlock) == 16 * 2);
NN_STATIC_ASSERT(sizeof(EccP256DeviceKeyBlock) == 16 * 4);
NN_STATIC_ASSERT(sizeof(EccP256DeviceCertificateBlock) == 16 * 25);
NN_STATIC_ASSERT(sizeof(EccB233DeviceKeyBlock) == 16 * 4);
NN_STATIC_ASSERT(sizeof(EccB233DeviceCertificateBlock) == 16 * 25);
NN_STATIC_ASSERT(sizeof(EccP256ETicketKeyBlock) == 16 * 4);
NN_STATIC_ASSERT(sizeof(EccP256ETicketCertificateBlock) == 16 * 25);
NN_STATIC_ASSERT(sizeof(EccB233ETicketKeyBlock) == 16 * 4);
NN_STATIC_ASSERT(sizeof(EccB233ETicketCertificateBlock) == 16 * 25);
NN_STATIC_ASSERT(sizeof(SslKeyBlock) == 16 * 18);
NN_STATIC_ASSERT(sizeof(SslCertificateSizeBlock) == 16 * 1);
NN_STATIC_ASSERT(sizeof(SslCertificateBlock) == 16 * 130);
NN_STATIC_ASSERT(sizeof(RandomNumberBlock) == 16 * 258);
NN_STATIC_ASSERT(sizeof(GameCardKeyBlock) == 16 * 18);
NN_STATIC_ASSERT(sizeof(GameCardCertificateBlock) == 16 * 66);
NN_STATIC_ASSERT(sizeof(Rsa2048ETicketKeyBlock) == 16 * 35);
NN_STATIC_ASSERT(sizeof(Rsa2048ETicketCertificateBlock) == 16 * 37);
NN_STATIC_ASSERT(sizeof(BatteryLotBlock) == 16 * 2);
NN_STATIC_ASSERT(sizeof(SpeakerCalibrationValueBlock) == 16 * 129);
NN_STATIC_ASSERT(sizeof(RegionCodeBlock) == 16 * 1);
NN_STATIC_ASSERT(sizeof(AmiiboKeyBlock) == 16 * 6);
NN_STATIC_ASSERT(sizeof(AmiiboEcqvCertificateBlock) == 16 * 2);
NN_STATIC_ASSERT(sizeof(AmiiboEcdsaCertificateBlock) == 16 * 8);
NN_STATIC_ASSERT(sizeof(AmiiboEcqvBlsKeyBlock) == 16 * 5);
NN_STATIC_ASSERT(sizeof(AmiiboEcqvBlsCertificateBlock) == 16 * 3);
NN_STATIC_ASSERT(sizeof(AmiiboEcqvBlsRootCertificateBlock) == 16 * 10);
NN_STATIC_ASSERT(sizeof(ProductModelBlock) == 16 * 1);
NN_STATIC_ASSERT(sizeof(ColorVariationBlock) == 16 * 1);
NN_STATIC_ASSERT(sizeof(LcdBacklightBrightnessMappingBlock) == 16 * 1);
NN_STATIC_ASSERT(sizeof(ExtendedEccB233DeviceKeyBlock) == 16 * 6);
NN_STATIC_ASSERT(sizeof(ExtendedEccP256ETicketKeyBlock) == 16 * 6);
NN_STATIC_ASSERT(sizeof(ExtendedEccB233ETicketKeyBlock) == 16 * 6);
NN_STATIC_ASSERT(sizeof(ExtendedRsa2048ETicketKeyBlock) == 16 * 37);
NN_STATIC_ASSERT(sizeof(ExtendedSslKeyBlock) == 16 * 20);
NN_STATIC_ASSERT(sizeof(ExtendedGameCardKeyBlock) == 16 * 20);
NN_STATIC_ASSERT(sizeof(LcdVendorIdBlock) == 16 * 1);
NN_STATIC_ASSERT(sizeof(ExtendedRsa2048DeviceKeyBlock) == 16 * 37);
NN_STATIC_ASSERT(sizeof(Rsa2048DeviceCertificateBlock) == 16 * 37);
NN_STATIC_ASSERT(sizeof(UsbTypeCPowerSourceCircuitBlock) == 16 * 1);
NN_STATIC_ASSERT(sizeof(HousingSubColorBlock) == 16 * 1);
NN_STATIC_ASSERT(sizeof(HousingBezelColorBlock) == 16 * 1);
NN_STATIC_ASSERT(sizeof(HousingMainColor1Block) == 16 * 1);
NN_STATIC_ASSERT(sizeof(HousingMainColor2Block) == 16 * 1);
NN_STATIC_ASSERT(sizeof(HousingMainColor3Block) == 16 * 1);
NN_STATIC_ASSERT(sizeof(CalibrationInfoBody) == 16 * 2044);
NN_STATIC_ASSERT(sizeof(CalibrationInfo) == 16 * 2048);
NN_STATIC_ASSERT(sizeof(CalibrationInfo) == sizeof(RawCalibrationInfo));

//!< 生産時較正情報を取得します。
::nn::Result GetCalibrationInfo(CalibrationInfo* outValue) NN_NOEXCEPT;

//!< 生産時較正情報を設定します。
::nn::Result SetCalibrationInfo(const CalibrationInfo& value) NN_NOEXCEPT;

}}
