﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstdlib>
#include <memory>
#include <string>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/result/result_HandlingUtility.h>

#include "cal_FileSystem.h"

namespace nn { namespace cal {

namespace {

//!< 専用パーティションの利用が可能か否かを表す値です。
#if defined(NN_BUILD_CONFIG_OS_WIN)
bool g_ExistsPartition = false;
#else
bool g_ExistsPartition = true;
#endif

//!< ホストファイルシステムのマウント名を返します。
const char* GetHostFsMountName() NN_NOEXCEPT
{
    return "HostFs";
}

//!< 生産時較正情報パーティション (FAT) のルートディレクトリのパスを返します。
::std::string GetFatRootDirectoryPath() NN_NOEXCEPT
{
    if (ExistsPartition())
    {
        return ::std::string(GetFatPartitionMountName()) + ":/";
    }
    else
    {
        return ::std::string(GetFatPartitionMountName()) + ":/fat";
    }
}

//!< 生産時較正情報パーティション (FAT) 上のパスを変換します。
::std::string ConvertFatPath(const ::std::string& value) NN_NOEXCEPT
{
    const ::std::string& rootPath = GetFatDummyRootPath();
    return GetFatRootDirectoryPath() +
           (ExistsPartition() ? "" : "/") +
           ((value.find(rootPath) == 0) ? value.substr(rootPath.size())
                                        : value);
}

//!< 生産時較正情報パーティション (FAT) のルートディレクトリを作成します。
::nn::Result CreateFatRootDirectory() NN_NOEXCEPT
{
    if (ExistsPartition())
    {
        // 専用パーティションの利用が可能ならばディレクトリの作成は不要
        NN_RESULT_SUCCESS;
    }

    NN_RESULT_TRY(::nn::fs::CreateDirectory(GetFatRootDirectoryPath().c_str()))
        NN_RESULT_CATCH(::nn::fs::ResultPathAlreadyExists)
        {
            // ディレクトリが既に存在するならば成功
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY
    NN_RESULT_SUCCESS;
}

} // namespace

void* AllocateForFileSystem(size_t size) NN_NOEXCEPT
{
    return ::std::malloc(size);
}

void DeallocateForFileSystem(void* addr, size_t) NN_NOEXCEPT
{
    ::std::free(addr);
}

const char* GetFatDummyRootPath() NN_NOEXCEPT
{
    return "cal:/";
}

::nn::Result CopyFile(::nn::fs::FileHandle src,
                      ::nn::fs::FileHandle dst) NN_NOEXCEPT
{
    auto fileSize = int64_t();
    NN_RESULT_DO(::nn::fs::GetFileSize(&fileSize, src));
    NN_RESULT_DO(::nn::fs::SetFileSize(dst, fileSize));
    const size_t bufferSize = 16 * 1024;
    ::std::unique_ptr<char[]> buffer(new char[bufferSize]);
    char* const ptr = buffer.get();
    int64_t bias = 0;
    const auto option = ::nn::fs::WriteOption();
    while (0 < fileSize)
    {
        size_t size = ::std::min(bufferSize, static_cast<size_t>(fileSize));
        NN_RESULT_DO(::nn::fs::ReadFile(&size, src, bias, ptr, size));
        NN_RESULT_DO(::nn::fs::WriteFile(dst, bias, ptr, size, option));
        fileSize -= static_cast<int64_t>(size);
        bias += static_cast<int64_t>(size);
    }
    NN_RESULT_DO(::nn::fs::FlushFile(dst));
    NN_RESULT_SUCCESS;
}

::nn::Result CreateFatPartitionFile(
    ::nn::fs::FileHandle dst, void* pBuffer, size_t length) NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::fs::SetFileSize(dst, length));
    const auto option = ::nn::fs::WriteOption();
    NN_RESULT_DO(::nn::fs::WriteFile(dst, 0, pBuffer, length, option));
    NN_RESULT_DO(::nn::fs::FlushFile(dst));
    NN_RESULT_SUCCESS;
}

::nn::Result ReadFatPartitionFile(
    ::nn::fs::FileHandle src, void* pBuffer, size_t length) NN_NOEXCEPT
{
    NN_RESULT_DO(::nn::fs::ReadFile(src, 0, pBuffer, length));
    NN_RESULT_SUCCESS;
}

::nn::Result DeleteFatPartitionFile(const ::std::string& value) NN_NOEXCEPT
{
    const ::std::string& path = ConvertFatPath(value);
    NN_RESULT_DO(::nn::fs::DeleteFile(path.c_str()));
    NN_RESULT_SUCCESS;
}

::nn::Result CreateFatPartitionDirectory(const ::std::string& value
                                         ) NN_NOEXCEPT
{
    const ::std::string& path = ConvertFatPath(value);
    NN_RESULT_DO(CreateFatRootDirectory());
    NN_RESULT_DO(::nn::fs::CreateDirectory(path.c_str()));
    NN_RESULT_SUCCESS;
}

::nn::Result DeleteFatPartitionDirectory(const ::std::string& value
                                         ) NN_NOEXCEPT
{
    const ::std::string& path = ConvertFatPath(value);
    NN_RESULT_DO(::nn::fs::DeleteDirectoryRecursively(path.c_str()));
    NN_RESULT_SUCCESS;
}

::nn::Result IsFatPartitionFileExist(
    bool& pOutValue, const ::std::string& value) NN_NOEXCEPT
{
    const ::std::string& path = ConvertFatPath(value);
    auto type = ::nn::fs::DirectoryEntryType::DirectoryEntryType_File;
    auto result = ::nn::fs::GetEntryType(&type, path.c_str());

    if (result <= nn::fs::ResultPathNotFound())
    {
        pOutValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (result.IsSuccess())
    {
        if (type == ::nn::fs::DirectoryEntryType::DirectoryEntryType_File)
        {
            pOutValue = true;
        }
        else
        {
            pOutValue = false;
        }
        NN_RESULT_SUCCESS;
    }
    else
    {
        pOutValue = false;
        return result;
    }
}

::nn::Result IsFatPartitionDirectoryExist(
    bool& pOutValue, const ::std::string& value) NN_NOEXCEPT
{
    const ::std::string& path = ConvertFatPath(value);
    auto type = ::nn::fs::DirectoryEntryType::DirectoryEntryType_File;
    auto result = ::nn::fs::GetEntryType(&type, path.c_str());

    if (nn::fs::ResultPathNotFound().Includes(result))
    {
        pOutValue = false;
        NN_RESULT_SUCCESS;
    }
    else if (result.IsSuccess())
    {
        if (type == ::nn::fs::DirectoryEntryType::DirectoryEntryType_Directory)
        {
            pOutValue = true;
        }
        else
        {
            pOutValue = false;
        }
        NN_RESULT_SUCCESS;
    }
    else
    {
        pOutValue = false;
        return result;
    }
}

void EnableDebugMode() NN_NOEXCEPT
{
    g_ExistsPartition = false;
}

bool ExistsPartition() NN_NOEXCEPT
{
    return g_ExistsPartition;
}

CalibrationFile::CalibrationFile() NN_NOEXCEPT
    : m_IsOpen(false)
    , m_Handle()
    , m_Storage(nullptr)
{
    // 何もしない
}

CalibrationFile::~CalibrationFile() NN_NOEXCEPT
{
    if (m_IsOpen)
    {
        if (!ExistsPartition())
        {
            ::nn::fs::CloseFile(m_Handle);
        }
    }
}

::nn::Result CalibrationFile::Open() NN_NOEXCEPT
{
    if (!m_IsOpen)
    {
        if (ExistsPartition())
        {
            NN_RESULT_DO(
                ::nn::fs::OpenBisPartition(&m_Storage,
                                           ::nn::fs::BisPartitionId::CalibrationBinary));
        }
        else
        {
            ::std::string path;
            path += GetRawPartitionMountName();
            path += ":/";
            path += "calibration.dat";

            NN_RESULT_TRY(::nn::fs::CreateFile(path.c_str(), 0))
                NN_RESULT_CATCH(::nn::fs::ResultPathAlreadyExists)
                {
                    // ファイルが既に存在するならば成功
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY

            NN_RESULT_DO(::nn::fs::OpenFile(&m_Handle,
                                            path.c_str(),
                                            ::nn::fs::OpenMode_Read |
                                            ::nn::fs::OpenMode_Write));
        }

        m_IsOpen = true;
    }

    NN_RESULT_SUCCESS;
}

::nn::Result CalibrationFile::Read(void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsOpen);

    if (ExistsPartition())
    {
        auto fileSize = int64_t();
        NN_RESULT_DO(m_Storage->GetSize(&fileSize));
        size = ::std::min(size, static_cast<size_t>(fileSize));
        if (size > 0)
        {
            NN_RESULT_DO(m_Storage->Read(0, buffer, size));
        }
    }
    else
    {
        auto fileSize = int64_t();
        NN_RESULT_DO(::nn::fs::GetFileSize(&fileSize, m_Handle));
        size = ::std::min(size, static_cast<size_t>(fileSize));
        if (size > 0)
        {
            NN_RESULT_DO(::nn::fs::ReadFile(m_Handle, 0, buffer, size));
        }
    }

    NN_RESULT_SUCCESS;
}

::nn::Result CalibrationFile::Write(const void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsOpen);

    if (ExistsPartition())
    {
        NN_RESULT_DO(m_Storage->Write(0, buffer, size));
    }
    else
    {
        NN_RESULT_DO(::nn::fs::SetFileSize(m_Handle, size));
        ::nn::fs::WriteOption writeOption =
              ::nn::fs::WriteOption::MakeValue(::nn::fs::WriteOptionFlag_Flush);
        NN_RESULT_DO(
            ::nn::fs::WriteFile(m_Handle, 0, buffer, size, writeOption));
    }

    NN_RESULT_SUCCESS;
}

HostFsFile::HostFsFile() NN_NOEXCEPT
    : m_IsOpen(false)
    , m_Handle()
{
    // 何もしない
}

HostFsFile::~HostFsFile() NN_NOEXCEPT
{
    if (m_IsOpen)
    {
        ::nn::fs::CloseFile(m_Handle);

        ::nn::fs::Unmount(GetHostFsMountName());
    }
}

::nn::Result HostFsFile::Open(::nn::fs::FileHandle* outValue,
                              const char* path,
                              int mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_NOT_NULL(path);

    if (!m_IsOpen)
    {
        const ::std::string string(path);
        ::std::string rootPath;
        ::std::string filePath;
        filePath += GetHostFsMountName();
        filePath += ":/";
        const size_t index = string.find_last_of('\\');
        if (0 == string.size() || index >= string.size() - 1)
        {
            rootPath = string;
        }
        else
        {
            rootPath = string.substr(0, index + 1);
            filePath += string.substr(index + 1);
        }

        NN_RESULT_DO(::nn::fs::MountHost(GetHostFsMountName(),
                                         rootPath.c_str()));

        if ((mode & ::nn::fs::OpenMode_Write) != 0)
        {
            NN_RESULT_TRY(::nn::fs::CreateFile(filePath.c_str(), 0))
                NN_RESULT_CATCH(::nn::fs::ResultPathAlreadyExists)
                {
                    // ファイルが既に存在するならば成功
                }
                NN_RESULT_CATCH_ALL
                {
                    ::nn::fs::Unmount(GetHostFsMountName());
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY
        }

        NN_RESULT_TRY(::nn::fs::OpenFile(&m_Handle, filePath.c_str(), mode))
            NN_RESULT_CATCH_ALL
            {
                ::nn::fs::Unmount(GetHostFsMountName());
                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY

        m_IsOpen = true;
    }

    *outValue = m_Handle;

    NN_RESULT_SUCCESS;
}

FatPartitionFile::FatPartitionFile() NN_NOEXCEPT
    : m_IsOpen(false)
    , m_Handle()
{
    // 何もしない
}

FatPartitionFile::~FatPartitionFile() NN_NOEXCEPT
{
    if (m_IsOpen)
    {
        ::nn::fs::CloseFile(m_Handle);
    }
}

::nn::Result FatPartitionFile::Open(::nn::fs::FileHandle* outValue,
                                    const char* path,
                                    int mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_NOT_NULL(path);

    if (!m_IsOpen)
    {
        NN_RESULT_DO(CreateFatRootDirectory());

        const ::std::string& filePath = ConvertFatPath(path);
        if ((mode & ::nn::fs::OpenMode_Write) != 0)
        {
            NN_RESULT_TRY(::nn::fs::CreateFile(filePath.c_str(), 0))
                NN_RESULT_CATCH(::nn::fs::ResultPathAlreadyExists)
                {
                    // ファイルが既に存在するならば成功
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_RESULT_RETHROW;
                }
            NN_RESULT_END_TRY
        }

        NN_RESULT_DO(::nn::fs::OpenFile(&m_Handle, filePath.c_str(), mode));

        m_IsOpen = true;
    }

    *outValue = m_Handle;

    NN_RESULT_SUCCESS;
}

FatPartitionDirectory::FatPartitionDirectory() NN_NOEXCEPT
    : m_IsOpen(false)
    , m_Handle()
{
    // 何もしない
}

FatPartitionDirectory::~FatPartitionDirectory() NN_NOEXCEPT
{
    if (m_IsOpen)
    {
        ::nn::fs::CloseDirectory(m_Handle);
    }
}

::nn::Result FatPartitionDirectory::Open(::nn::fs::DirectoryHandle* outValue,
                                         const char* path,
                                         int mode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES_NOT_NULL(path);

    if (!m_IsOpen)
    {
        NN_RESULT_DO(CreateFatRootDirectory());

        NN_RESULT_DO(::nn::fs::OpenDirectory(&m_Handle,
                                             ConvertFatPath(path).c_str(),
                                             mode));

        m_IsOpen = true;
    }

    *outValue = m_Handle;

    NN_RESULT_SUCCESS;
}

}}
