﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_HipcClientProxyByName.h> // for nn::sf::CreateHipcProxyByName
#include <nn/sf/sf_ExpHeapAllocator.h>      // for nn::sf::ExpHeapStaticAllocator

#include <nn/applet/applet.h>

#include <nn/btm/btm_Types.h>
#include <nn/btm/btm_IBtm.sfdl.h>
#include <nn/btm/btm_ServiceName.h>

#include <nn/btm/user/btm_UserApi.h>

namespace nn { namespace btm {
namespace {
    bool isInitialized = false;
    nn::sf::SharedPointer<IBtmUser>     g_BtmUser;
    nn::sf::SharedPointer<IBtmUserCore> g_BtmUserCore;
    struct BtmUserByHipcTag;
    typedef nn::sf::ExpHeapStaticAllocator<1024 * 16, BtmUserByHipcTag> BtmUserAllocator;
} // namespace

// BtmUserAllocator を静的コンストラクタで初期化するためのヘルパー
class BtmUserAllocatorInitializer
{
public:

    BtmUserAllocatorInitializer() NN_NOEXCEPT
    {
        BtmUserAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }

} g_BtmUserAllocatorInitializer;

}} // namespace nn::btm

namespace nn { namespace btm { namespace user {
void InitializeBtmUserInterface() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(!isInitialized);
    NN_ABORT_UNLESS(!g_BtmUser);

    nn::sf::SharedPointer<IBtmUser> ret;
    auto result = nn::sf::CreateHipcProxyByName<IBtmUser, BtmUserAllocator::Policy>(&ret, BtmUserServiceName);
    NN_ABORT_UNLESS(result.IsSuccess());

    g_BtmUser = ret;
    NN_UNUSED(result);

    nn::sf::SharedPointer<nn::btm::IBtmUserCore> core;
    g_BtmUser->GetCoreImpl(&core);
    NN_ABORT_UNLESS(core);

    g_BtmUserCore = core;
    isInitialized = true;
}

void FinalizeBtmUserInterface() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    NN_ABORT_UNLESS(g_BtmUser);
    NN_ABORT_UNLESS(g_BtmUserCore);
    g_BtmUser = nullptr;
    g_BtmUserCore = nullptr;
    isInitialized = false;
}

void AcquireBleScanEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSystemEvent);
    NN_ABORT_UNLESS(isInitialized);

    nn::sf::NativeHandle handle;
    bool isSuccess = g_BtmUserCore->AcquireBleScanEventImpl(&handle);

    if (isSuccess)
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    else
    {
        NN_ABORT("[btm] Failed to acquire system event for BLE Scan.\n");
    }
}

nn::Result GetBleScanFilterParameter(BleAdvFilterForGeneral* pFilter, uint16_t parameterId) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    auto result = g_BtmUserCore->GetBleScanParameterGeneralImpl(pFilter, parameterId);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return result;
}

nn::Result GetBleScanFilterParameter(BleAdvFilterForSmartDevice* pFilter, uint16_t parameterId) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    auto result = g_BtmUserCore->GetBleScanParameterSmartDeviceImpl(pFilter, parameterId);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return result;
}

nn::Result StartBleScanForGeneral(const BleAdvFilterForGeneral& filter) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    auto result = g_BtmUserCore->StartBleScanForGeneralImpl(nn::applet::GetAppletResourceUserId(), filter);
    return result;
}

nn::Result StopBleScanForGeneral() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    auto result = g_BtmUserCore->StopBleScanForGeneralImpl();
    return result;
}

uint8_t GetBleScanResultsForGeneral(ScanResult pResults[], uint8_t inNum) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    NN_ABORT_UNLESS_NOT_NULL(pResults);

    uint8_t outNum = 0;
    nn::sf::OutArray<ScanResult> results(&pResults[0], inNum);

    g_BtmUserCore->GetBleScanResultsForGeneralImpl(results, &outNum, nn::applet::GetAppletResourceUserId());

    return outNum;
}

nn::Result StartBleScanForPaired(const BleAdvFilterForGeneral& filter) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    auto result = g_BtmUserCore->StartBleScanForPairedDeviceImpl(nn::applet::GetAppletResourceUserId(), filter);
    return result;
}

nn::Result StopBleScanForPaired() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    auto result = g_BtmUserCore->StopBleScanForPairedDeviceImpl();
    return result;
}

nn::Result StartBleScanForSmartDevice(const BleAdvFilterForSmartDevice &filter) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    auto result = g_BtmUserCore->StartBleScanForSmartDeviceImpl(nn::applet::GetAppletResourceUserId(), filter);
    return result;
}

nn::Result StopBleScanForSmartDevice() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    auto result = g_BtmUserCore->StopBleScanForSmartDeviceImpl();
    return result;
}

uint8_t GetBleScanResultsForSmartDevice(ScanResult pResults[], uint8_t inNum) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    NN_ABORT_UNLESS_NOT_NULL(pResults);

    uint8_t outNum = 0;
    nn::sf::OutArray<ScanResult> results(&pResults[0], inNum);

    g_BtmUserCore->GetBleScanResultsForSmartDeviceImpl(results, &outNum, nn::applet::GetAppletResourceUserId());

    return outNum;
}

void AcquireBleConnectionEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    NN_ABORT_UNLESS_NOT_NULL(pSystemEvent);

    nn::sf::NativeHandle handle;
    bool isSuccess = g_BtmUserCore->AcquireBleConnectionEventImpl(&handle);

    if (isSuccess)
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    else
    {
        NN_ABORT("[btm] Failed to acquire system event for BLE Connection.\n");
    }
}

nn::Result BleConnect(nn::bluetooth::Address address) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    auto result = g_BtmUserCore->BleConnectImpl(nn::applet::GetAppletResourceUserId(), address);
    return result;
}

nn::Result BleDisconnect(uint32_t connectionHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    auto result = g_BtmUserCore->BleDisconnectImpl(connectionHandle);
    return result;
}

uint8_t BleGetConnectionState(user::BleClientConnState *pConnState, uint8_t inNum) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    NN_ABORT_UNLESS_NOT_NULL(pConnState);

    uint8_t outNum;
    nn::sf::OutArray<BleClientConnState> outArray(&pConnState[0], inNum);

    g_BtmUserCore->BleGetConnectionStateImpl(outArray, &outNum, nn::applet::GetAppletResourceUserId());

    return outNum;
}

void AcquireBleServiceDiscoveryEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    NN_ABORT_UNLESS_NOT_NULL(pSystemEvent);

    nn::sf::NativeHandle handle;
    bool isSuccess = g_BtmUserCore->AcquireBleServiceDiscoveryEventImpl(&handle);

    if (isSuccess)
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    else
    {
        NN_ABORT("[btm] Failed to acquire system event for BLE GATT Service Discovery.\n");
    }
}

uint8_t GetGattServices(GattService *pServices, uint8_t inNum, uint32_t connectionHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    NN_ABORT_UNLESS_NOT_NULL(pServices);

    uint8_t outNum;
    nn::sf::OutArray<GattService> outArray(&pServices[0], inNum);
    g_BtmUserCore->GetGattServicesImpl(outArray, &outNum,
                                       nn::applet::GetAppletResourceUserId(), connectionHandle);

    return outNum;
}

uint8_t GetGattIncludedServices(GattService *pIncludedServices, uint8_t inNum, uint32_t connectionHandle, uint16_t serviceHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    NN_ABORT_UNLESS_NOT_NULL(pIncludedServices);

    uint8_t outNum = 0;
    nn::sf::OutArray<GattService> outArray(&pIncludedServices[0], inNum);
    g_BtmUserCore->GetGattIncludedServicesImpl(outArray, &outNum,
                                               nn::applet::GetAppletResourceUserId(), connectionHandle, serviceHandle);

    return outNum;
}

bool GetGattService(GattService *pService, uint32_t connectionHandle, const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    NN_ABORT_UNLESS_NOT_NULL(pService);

    nn::sf::Out<GattService> outService(pService);

    return g_BtmUserCore->GetGattServiceImpl(outService,
                                             nn::applet::GetAppletResourceUserId(), connectionHandle, uuid);
}

bool GetBelongingGattService(GattService *pService, uint32_t connectionHandle, uint16_t attributeHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    NN_ABORT_UNLESS_NOT_NULL(pService);

    nn::sf::Out<GattService> outService(pService);

    return g_BtmUserCore->GetBelongingServiceImpl(outService,
                                                  nn::applet::GetAppletResourceUserId(), connectionHandle, attributeHandle);
}

uint8_t GetGattCharacteristics(GattCharacteristic *pCharacteristics, uint8_t inNum, uint32_t connectionHandle, uint16_t serviceHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    NN_ABORT_UNLESS_NOT_NULL(pCharacteristics);

    uint8_t outNum;
    nn::sf::OutArray<GattCharacteristic> outArray(&pCharacteristics[0], inNum);

    g_BtmUserCore->GetGattCharacteristicsImpl(outArray, &outNum,
                                              nn::applet::GetAppletResourceUserId(), connectionHandle, serviceHandle);

    return outNum;
}

uint8_t GetGattDescriptors(GattDescriptor *pDescriptors, uint8_t inNum, uint32_t connectionHandle, uint16_t charHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    NN_ABORT_UNLESS_NOT_NULL(pDescriptors);

    uint8_t outNum;
    nn::sf::OutArray<GattDescriptor> outArray(&pDescriptors[0], inNum);

    g_BtmUserCore->GetGattDescriptorsImpl(outArray, &outNum,
                                          nn::applet::GetAppletResourceUserId(), connectionHandle, charHandle);

    return outNum;
}

void AcquireBlePairingEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    NN_ABORT_UNLESS_NOT_NULL(pSystemEvent);

    nn::sf::NativeHandle handle;
    bool isSuccess = g_BtmUserCore->AcquireBlePairingEventImpl(&handle);

    if (isSuccess)
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    else
    {
        NN_ABORT("[btm] Failed to acquire system event for BLE pairing.\n");
    }
}

nn::Result BlePairDevice(uint32_t connectionHandle, const BleAdvFilterForGeneral& filter) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    auto result = g_BtmUserCore->BlePairDeviceImpl(connectionHandle, filter);
    return result;
}

nn::Result BleUnPairDevice(uint32_t connectionHandle, const BleAdvFilterForGeneral& filter) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    auto result = g_BtmUserCore->BleUnpairDeviceOnBothImpl(connectionHandle, filter);
    return result;
}

nn::Result BleUnPairDevice(const nn::bluetooth::Address& address, const BleAdvFilterForGeneral& filter) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    auto result = g_BtmUserCore->BleUnpairDeviceImpl(address, filter);
    return result;
}

uint8_t BleGetPairedDevices(nn::bluetooth::Address* pOutAddresses, uint8_t count, const BleAdvFilterForGeneral& filter) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);

    uint8_t outNum;
    nn::sf::OutArray<BdAddress> outArray(&pOutAddresses[0], count);

    g_BtmUserCore->BleGetPairedAddressesImpl(outArray, &outNum, filter);

    return outNum;
}

void AcquireBleMtuConfigEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    NN_ABORT_UNLESS_NOT_NULL(pSystemEvent);

    nn::sf::NativeHandle handle;
    bool isSuccess = g_BtmUserCore->AcquireBleMtuConfigEventImpl(&handle);

    if (isSuccess)
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    else
    {
        NN_ABORT("[btm] Failed to acquire system event for BLE mtu config.\n");
    }
}

nn::Result ConfigureBleMtu(uint32_t connectionHandle, uint16_t mtu) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    NN_ABORT_UNLESS_RANGE(mtu, nn::bluetooth::BleMtuDefault, nn::bluetooth::BleMtuMax + 1);

    auto result = g_BtmUserCore->ConfigureBleMtuImpl(nn::applet::GetAppletResourceUserId(), connectionHandle, mtu);
    return result;
}

uint16_t GetBleMtu(uint32_t connectionHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    uint16_t mtu;
    g_BtmUserCore->GetBleMtuImpl(&mtu, nn::applet::GetAppletResourceUserId(), connectionHandle);

    return mtu;

}

nn::Result RegisterBleGattDataPath(const BleDataPath& path) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    auto result = g_BtmUserCore->RegisterBleGattDataPathImpl(nn::applet::GetAppletResourceUserId(), path);
    return result;
}

nn::Result UnregisterBleGattDataPath(const BleDataPath& path) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(isInitialized);
    auto result = g_BtmUserCore->UnregisterBleGattDataPathImpl(nn::applet::GetAppletResourceUserId(), path);
    return result;
}

}}} // namespace nn::btm::user

