﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_HipcClientProxyByName.h> // for nn::sf::CreateHipcProxyByName
#include <nn/sf/sf_ExpHeapAllocator.h>      // for nn::sf::ExpHeapStaticAllocator

#include <nn/btm/btm_Types.h>
#include <nn/btm/btm_Api.h>
#include <nn/btm/btm_Result.h>
#include <nn/btm/btm_IBtm.sfdl.h>
#include <nn/btm/btm_ServiceName.h>


namespace nn { namespace btm {

namespace {
nn::sf::SharedPointer<IBtm> g_Btm;
struct  BtmByHipcTag;
typedef nn::sf::ExpHeapStaticAllocator<1024 * 16, BtmByHipcTag> BtmAllocator;
nn::os::MutexType g_Mutex = NN_OS_MUTEX_INITIALIZER(false);
uint8_t g_InitCount = 0;
}   // namespace

// BtmAllocator を静的コンストラクタで初期化するためのヘルパー
class BtmAllocatorInitializer
{
public:

    BtmAllocatorInitializer() NN_NOEXCEPT
    {
        BtmAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }

} g_BtmAllocatorInitializer;


void InitializeBtmInterface() NN_NOEXCEPT
{
    LockMutex(&g_Mutex);
    if(g_InitCount == 0)
    {
        NN_SDK_ASSERT(!g_Btm);

        nn::sf::SharedPointer<IBtm> ret;
        auto result = nn::sf::CreateHipcProxyByName<IBtm, BtmAllocator::Policy>(&ret, BtmServiceName);
        NN_ABORT_UNLESS(result.IsSuccess());

        //[Todo]ポインタを直に渡してしまっても良さそう
        g_Btm = ret;
        g_InitCount++;
        NN_UNUSED(result);
    }
    else if(g_InitCount == 255)
    {
        NN_ABORT("[btm]Initialized count over 255");
    }
    else
    {
        //初期化済みの場合は、初期化カウントを増やすのみ
        g_InitCount++;
    }
    UnlockMutex(&g_Mutex);
}

void FinalizeBtmInterface() NN_NOEXCEPT
{
    LockMutex(&g_Mutex);
    if(g_InitCount == 0)
    {
        NN_ABORT("[btm]Initialized count over 0");
    }
    else if(g_InitCount == 1)
    {
        //初期化カウントが残り1の時は末期化する
        NN_SDK_ASSERT(g_Btm);
        g_Btm = nullptr;
        g_InitCount--;
    }
    else
    {
        //初期化カウントが残っている場合はカウントを減らすのみ
        g_InitCount--;
    }
    UnlockMutex(&g_Mutex);
}


//-----------------------------------------
//Btm
//-----------------------------------------
void GetState(BtmState* pState)  NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pState);
    g_Btm->GetStateImpl(pState);
}

void GetHostDeviceProperty(HostDeviceProperty* pHostDeviceProperty) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pHostDeviceProperty);
    g_Btm->GetHostDevicePropertyImpl(pHostDeviceProperty);
}


//-----------------------------------------
//Connected Device Condition
//-----------------------------------------
void RegisterSystemEventForConnectedDeviceCondition(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    nn::sf::NativeHandle handle;
    bool isSuccess = g_Btm->AcquireDeviceConditionEventImpl(&handle);
    if(isSuccess)
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    else
    {
        NN_ABORT("[btm]There are no system events.\n");
    }
}

void GetConnectedDeviceCondition(DeviceConditionList* pList) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pList);
    g_Btm->GetDeviceConditionImpl(pList);
}

nn::Result SetBurstMode(const BdAddress* pAddress, bool isBurstMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAddress);
    auto result = g_Btm->SetBurstModeImpl(*pAddress,isBurstMode);
    return result;
}

nn::Result SetSlotMode(const DeviceSlotModeList* pDeviceSlotModeList) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDeviceSlotModeList);
    if(pDeviceSlotModeList->deviceCount > 8)
    {
        return nn::btm::ResultInvalidArgument();
    }
    //[Todo]構造体内部のパラメータチェック
    auto result = g_Btm->SetSlotModeImpl(*pDeviceSlotModeList);
    return result;
}

nn::Result SetBluetoothMode(BluetoothMode bluetoothMode) NN_NOEXCEPT
{
    if(bluetoothMode > BluetoothMode_Auto)
    {
        return nn::btm::ResultInvalidArgument();
    }
    auto result = g_Btm->SetBluetoothModeImpl(bluetoothMode);
    return result;
}

nn::Result SetWlanMode(WlanMode wlanMode) NN_NOEXCEPT
{
    if(wlanMode > WlanMode_User8)
    {
        return nn::btm::ResultInvalidArgument();
    }
    auto result = g_Btm->SetWlanModeImpl(wlanMode);
    return result;
}

//-----------------------------------------
//Registered Device Info
//-----------------------------------------
void RegisterSystemEventForRegisteredDeviceInfo(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    nn::sf::NativeHandle handle;
    bool isSuccess = g_Btm->AcquireDeviceInfoEventImpl(&handle);
    if(isSuccess)
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    else
    {
        NN_ABORT("[btm]There are no system events.\n");
    }
}

void GetDeviceInfo(DeviceInfoList* pList) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pList);
    g_Btm->GetDeviceInfoImpl(pList);
}

void AddDeviceInfo(const DeviceInfo* pInfo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pInfo);
    g_Btm->AddDeviceInfoImpl(*pInfo);
}

void RemoveDeviceInfo(const BdAddress* pAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAddress);
    g_Btm->RemoveDeviceInfoImpl(*pAddress);
}

void ProtectDeviceInfo(const BdAddress* pAddress, bool isProtect) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAddress);
    g_Btm->ProtectDeviceInfoImpl(*pAddress, isProtect);
}

void IncreaseDeviceInfoOrder(const BdAddress *pAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAddress);
    g_Btm->IncreaseDeviceInfoOrderImpl(*pAddress);
}


//-----------------------------------------
//PowerMode
//-----------------------------------------
void AcquireLlrStateEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    nn::sf::NativeHandle handle;
    bool isSuccess = g_Btm->AcquireLlrStateEventImpl(&handle);
    if(isSuccess)
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    else
    {
        NN_ABORT("[btm]There are no system events.\n");
    }
}

bool IsLlrStarted() NN_NOEXCEPT
{
    return g_Btm->IsLlrStartedImpl();
}

nn::Result LlrNotify(const BdAddress *pAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAddress);
    auto result = g_Btm->LlrNotifyImpl(*pAddress);
    return result;
}

void EnableRadio(bool isEnable) NN_NOEXCEPT
{
    if(isEnable)
    {
        g_Btm->EnableRadioImpl();
    }
    else
    {
        g_Btm->DisableRadioImpl();
    }
}

void RegisterSystemEventForAwakeReq(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    nn::sf::NativeHandle handle;
    bool isSuccess = g_Btm->AcquireAwakeReqEventImpl(&handle);
    if(isSuccess)
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    else
    {
        NN_ABORT("[btm]There are no system events.\n");
    }
}

void EnableSlotSaving(bool isEnable) NN_NOEXCEPT
{
    g_Btm->EnableSlotSavingImpl(isEnable);
}

//-----------------------------------------
//Hid
//-----------------------------------------
void HidDisconnect(const BdAddress* pAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAddress);
    g_Btm->HidDisconnectImpl(*pAddress);
}

nn::Result HidSetRetransmissionMode(const BdAddress* pAddress, const ZeroRetransmissionList* pZeroRetransmissionList) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAddress);
    NN_SDK_REQUIRES_NOT_NULL(pZeroRetransmissionList);
    auto result = g_Btm->HidSetRetransmissionModeImpl(*pAddress, *pZeroRetransmissionList);
    return result;
}

//------------------------------------------
// BLE
//------------------------------------------
void AcquireBleScanEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSystemEvent);

    nn::sf::NativeHandle handle;
    bool isSuccess = g_Btm->AcquireBleScanEventImpl(&handle);

    if (isSuccess)
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    else
    {
        NN_ABORT("[btm] Failed to acquire system event for BLE Scan.\n");
    }
}

nn::Result GetBleScanFilterParameter(user::BleAdvFilterForGeneral* pFilter, uint16_t parameterId) NN_NOEXCEPT
{
    auto result = g_Btm->GetBleScanParameterGeneralImpl(pFilter, parameterId);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return result;
}

nn::Result GetBleScanFilterParameter(user::BleAdvFilterForSmartDevice* pFilter, uint16_t parameterId) NN_NOEXCEPT
{
    auto result = g_Btm->GetBleScanParameterSmartDeviceImpl(pFilter, parameterId);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return result;
}

nn::Result StartBleScanForGeneral(const user::BleAdvFilterForGeneral& filter) NN_NOEXCEPT
{
    auto result = g_Btm->StartBleScanForGeneralImpl(filter);
    return result;
}

nn::Result StopBleScanForGeneral() NN_NOEXCEPT
{
    auto result = g_Btm->StopBleScanForGeneralImpl();
    return result;
}

uint8_t GetBleScanResultsForGeneral(user::ScanResult pResults[], uint8_t inNum) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pResults);

    uint8_t outNum = 0;
    nn::sf::OutArray<user::ScanResult> results(&pResults[0], inNum);

    g_Btm->GetBleScanResultsForGeneralImpl(results, &outNum);

    return outNum;
}

nn::Result StartBleScanForPaired(const user::BleAdvFilterForGeneral& filter) NN_NOEXCEPT
{
    auto result = g_Btm->StartBleScanForPairedDeviceImpl(filter);
    return result;
}

nn::Result StopBleScanForPaired() NN_NOEXCEPT
{
    auto result = g_Btm->StopBleScanForPairedDeviceImpl();
    return result;
}

nn::Result StartBleScanForSmartDevice(const user::BleAdvFilterForSmartDevice &filter) NN_NOEXCEPT
{
    auto result = g_Btm->StartBleScanForSmartDeviceImpl(filter);
    return result;
}

nn::Result StopBleScanForSmartDevice() NN_NOEXCEPT
{
    auto result = g_Btm->StopBleScanForSmartDeviceImpl();
    return result;
}

uint8_t GetBleScanResultsForSmartDevice(user::ScanResult pResults[], uint8_t inNum) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pResults);

    uint8_t outNum = 0;
    nn::sf::OutArray<user::ScanResult> results(&pResults[0], inNum);

    g_Btm->GetBleScanResultsForSmartDeviceImpl(results, &outNum);

    return outNum;
}

void AcquireBleConnectionEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    nn::sf::NativeHandle handle;
    bool isSuccess = g_Btm->AcquireBleConnectionEventImpl(&handle);

    if (isSuccess)
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    else
    {
        NN_ABORT("[btm] Failed to acquire system event for BLE Connection.\n");
    }
}

nn::Result BleConnect(nn::bluetooth::Address address) NN_NOEXCEPT
{
    auto result = g_Btm->BleConnectImpl(address);
    return result;
}

nn::Result BleOverrideConnection(uint32_t connectionHandle) NN_NOEXCEPT
{
    auto result = g_Btm->BleOverrideConnectionImpl(connectionHandle);
    return result;
}

nn::Result BleDisconnect(uint32_t connectionHandle) NN_NOEXCEPT
{
    auto result = g_Btm->BleDisconnectImpl(connectionHandle);
    return result;
}

uint8_t BleGetConnectionState(user::BleClientConnState *pConnState, uint8_t inNum) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConnState);

    uint8_t outNum;
    nn::sf::OutArray<user::BleClientConnState> outArray(&pConnState[0], inNum);

    g_Btm->BleGetConnectionStateImpl(outArray, &outNum);

    return outNum;
}

void BleGetGattClientConditionList(GattClientConditionList* pGattClientConditionList) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pGattClientConditionList);

    g_Btm->BleGetGattClientConditionListImpl(pGattClientConditionList);
}

void AcquireBleServiceDiscoveryEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    nn::sf::NativeHandle handle;
    bool isSuccess = g_Btm->AcquireBleServiceDiscoveryEventImpl(&handle);

    if (isSuccess)
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    else
    {
        NN_ABORT("[btm] Failed to acquire system event for BLE GATT Service Discovery.\n");
    }
}

uint8_t GetGattServices(user::GattService *pServices, uint8_t inNum, uint32_t connectionHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pServices);

    uint8_t outNum;
    nn::sf::OutArray<user::GattService> outArray(&pServices[0], inNum);

    g_Btm->GetGattServicesImpl(outArray, &outNum, connectionHandle);

    return outNum;
}

uint8_t GetGattIncludedServices(user::GattService *pIncludedServices, uint8_t inNum, uint32_t connectionHandle, uint16_t serviceHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pIncludedServices);

    uint8_t outNum = 0;
    nn::sf::OutArray<user::GattService> outArray(&pIncludedServices[0], inNum);
    g_Btm->GetGattIncludedServicesImpl(outArray, &outNum, connectionHandle, serviceHandle);

    return outNum;
}

bool GetGattService(user::GattService *pService, uint32_t connectionHandle, const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pService);

    nn::sf::Out<user::GattService> outService(pService);

    return g_Btm->GetGattServiceImpl(outService, connectionHandle, uuid);
}

bool GetBelongingGattService(user::GattService *pService, uint32_t connectionHandle, uint16_t attributeHandle) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pService);

    nn::sf::Out<user::GattService> outService(pService);

    return g_Btm->GetBelongingServiceImpl(outService, connectionHandle, attributeHandle);
}

uint8_t GetGattCharacteristics(user::GattCharacteristic *pCharacteristics, uint8_t inNum, uint32_t connectionHandle, uint16_t serviceHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pCharacteristics);

    uint8_t outNum;
    nn::sf::OutArray<user::GattCharacteristic> outArray(&pCharacteristics[0], inNum);

    g_Btm->GetGattCharacteristicsImpl(outArray, &outNum, connectionHandle, serviceHandle);

    return outNum;
}

uint8_t GetGattDescriptors(user::GattDescriptor *pDescriptors, uint8_t inNum, uint32_t connectionHandle, uint16_t charHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDescriptors);

    uint8_t outNum;
    nn::sf::OutArray<user::GattDescriptor> outArray(&pDescriptors[0], inNum);

    g_Btm->GetGattDescriptorsImpl(outArray, &outNum, connectionHandle, charHandle);

    return outNum;
}

void AcquireBlePairingEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pSystemEvent);

    nn::sf::NativeHandle handle;
    bool isSuccess = g_Btm->AcquireBlePairingEventImpl(&handle);

    if (isSuccess)
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    else
    {
        NN_ABORT("[btm] Failed to acquire system event for BLE pairing.\n");
    }
}

nn::Result BlePairDevice(uint32_t connectionHandle, const user::BleAdvFilterForGeneral& filter) NN_NOEXCEPT
{
    auto result = g_Btm->BlePairDeviceImpl(connectionHandle, filter);
    return result;
}

nn::Result BleUnPairDevice(uint32_t connectionHandle, const user::BleAdvFilterForGeneral& filter) NN_NOEXCEPT
{
    auto result = g_Btm->BleUnpairDeviceOnBothImpl(connectionHandle, filter);
    return result;
}

nn::Result BleUnPairDevice(const nn::bluetooth::Address& address, const user::BleAdvFilterForGeneral& filter) NN_NOEXCEPT
{
    auto result = g_Btm->BleUnpairDeviceImpl(address, filter);
    return result;
}

uint8_t BleGetPairedDevices(nn::bluetooth::Address* pOutAddresses, uint8_t count, const user::BleAdvFilterForGeneral& filter) NN_NOEXCEPT
{
    uint8_t outNum;
    nn::sf::OutArray<BdAddress> outArray(&pOutAddresses[0], count);

    g_Btm->BleGetPairedAddressesImpl(outArray, &outNum, filter);

    return outNum;
}

void AcquireBleMtuConfigEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);

    nn::sf::NativeHandle handle;
    bool isSuccess = g_Btm->AcquireBleMtuConfigEventImpl(&handle);

    if (isSuccess)
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    else
    {
        NN_ABORT("[btm] Failed to acquire system event for BLE mtu config.\n");
    }
}

nn::Result ConfigureBleMtu(uint32_t connectionHandle, uint16_t mtu) NN_NOEXCEPT
{
    auto result = g_Btm->ConfigureBleMtuImpl(connectionHandle, mtu);
    return result;
}

uint16_t GetBleMtu(uint32_t connectionHandle) NN_NOEXCEPT
{
    uint16_t mtu;
    g_Btm->GetBleMtuImpl(&mtu, connectionHandle);

    return mtu;

}

nn::Result RegisterBleGattDataPath(const user::BleDataPath& path) NN_NOEXCEPT
{
    auto result = g_Btm->RegisterBleGattDataPathImpl(path);
    return result;
}

nn::Result UnregisterBleGattDataPath(const user::BleDataPath& path) NN_NOEXCEPT
{
    auto result = g_Btm->UnregisterBleGattDataPathImpl(path);
    return result;
}

void RegisterAppletResourceUserId(const nn::applet::AppletResourceUserId& aruid,
                                  nn::applet::AppletId appletId) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Btm->RegisterAppletResourceUserIdImpl(aruid, static_cast<uint32_t>(appletId)));
}

void UnregisterAppletResourceUserId(const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Btm->UnregisterAppletResourceUserIdImpl(aruid));
}

void SetAppletResourceUserId(const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_Btm->SetAppletResourceUserIdImpl(aruid));
}

}} // namespace nn::btm

