﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>

#include <nn/bpc/detail/bpc_Log.h>
#include <nn/bpc/driver/bpc.h>
#include <nn/i2c/i2c.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

#include "bpc_PmicAccessor-hardware.nx.h"

namespace nn { namespace bpc { namespace driver { namespace detail {

namespace {

// IRQTOP 系レジスタ
const uint8_t OffsetIrqTop     = 0x05;
const uint8_t OffsetIrqTopM    = 0x0d;
const uint8_t MaskIrqTopGlblM  = (1 << 7);
const uint8_t MaskIrqTopRtcM   = (1 << 3);
const uint8_t MaskIrqTopOnOffM = (1 << 1);

// ONOFF 系レジスタ
const uint8_t OffsetOnOffIrq   = 0x0b;
const uint8_t OffsetOnOffIrqM  = 0x12;
const uint8_t MaskOnOffIrqEn0Rising  = (1 << 3);
const uint8_t MaskOnOffIrqAcOkFalling = (1 << 6);
const uint8_t MaskOnOffIrqAcOkRising = (1 << 7);

// NVERC レジスタ
const uint8_t OffsetNverc      = 0x0c;
const uint8_t MaskNvercRstIn   = 0x80;
const uint8_t MaskNvercMbu     = 0x40;
const uint8_t MaskNvercMbo     = 0x20;
const uint8_t MaskNvercTovld   = 0x08;
const uint8_t MaskNvercHdrst   = 0x04;
const uint8_t MaskNvercShdn    = 0x01;

WakeupReason g_WakeupReason = WakeupReasonNoReason;

uint8_t g_EscapedIrqTopM = 0;
uint8_t g_EscapedOnOffIrqM = 0;

bool g_ShutdownReasonUpdated = false;
ShutdownReason g_ShutdownReason = ShutdownReasonNoReason;

void ConfigWakeupInterrupts(nn::i2c::I2cSession& session) NN_NOEXCEPT
{
    uint8_t unused = 0;

    const uint8_t RegIrqTopM = static_cast<const uint8_t>(~(MaskIrqTopGlblM | MaskIrqTopRtcM | MaskIrqTopOnOffM)); // 0 で unmasked
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&g_EscapedIrqTopM, session, &OffsetIrqTopM));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(session, &OffsetIrqTopM, &RegIrqTopM));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&unused, session, &OffsetIrqTop)); // Clear on read によってクリア

    const uint8_t RegOnOffIrqM = static_cast<const uint8_t>(~(MaskOnOffIrqEn0Rising | MaskOnOffIrqAcOkFalling | MaskOnOffIrqAcOkRising)); // 0 で unmasked
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&g_EscapedOnOffIrqM, session, &OffsetOnOffIrqM));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(session, &OffsetOnOffIrqM, &RegOnOffIrqM));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&unused, session, &OffsetOnOffIrq)); // Clear on read によってクリア

    g_WakeupReason = WakeupReasonNoReason;

    NN_DETAIL_BPC_TRACE("Configured wakeup interrupts: ~RegIrqTopM=0x%02x, ~RegOnOffIrqM=0x%02x\n",
                static_cast<uint8_t>(~RegIrqTopM), static_cast<uint8_t>(~RegOnOffIrqM));
}

void ReadWakeupReason(nn::i2c::I2cSession& session) NN_NOEXCEPT
{
    // SuspendIrqHandler() を呼んで割り込みをハンドリングしない状態でレジスタを読むこと

    uint8_t regIrqTop = 0;
    uint8_t regIrqTopM = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&regIrqTop, session, &OffsetIrqTop));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&regIrqTopM, session, &OffsetIrqTopM));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(session, &OffsetIrqTopM, &g_EscapedIrqTopM)); // マスクをスリープ前に戻す
    NN_DETAIL_BPC_TRACE("regIrqTop=0x%02x / ~regIrqTopM=0x%02x\n", regIrqTop, static_cast<uint8_t>(~regIrqTopM));
    regIrqTop &= ~regIrqTopM;

    uint8_t regOnOffIrq = 0;
    uint8_t regOnOffIrqM = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&regOnOffIrq, session, &OffsetOnOffIrq));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&regOnOffIrqM, session, &OffsetOnOffIrqM));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(session, &OffsetOnOffIrqM, &g_EscapedOnOffIrqM)); // マスクをスリープ前に戻す
    NN_DETAIL_BPC_TRACE("regOnOffIrq=0x%02x / ~regOnOffIrqM=0x%02x\n", regOnOffIrq, static_cast<uint8_t>(~regOnOffIrqM));
    regOnOffIrq &= ~regOnOffIrqM;

    // どれか一つのみ取得するようにするならば Finalize 時に ONOFFCNFG2 を操作するのが望ましい
    g_WakeupReason = WakeupReasonNoReason;
    g_WakeupReason |= (regIrqTop & MaskIrqTopRtcM) ? WakeupReasonRtc : WakeupReasonNoReason; // RTC については TOPM だけでざっくり判断
    g_WakeupReason |= (regOnOffIrq & MaskOnOffIrqAcOkRising) ? WakeupReasonAcOkOn : WakeupReasonNoReason;
    g_WakeupReason |= (regOnOffIrq & MaskOnOffIrqAcOkFalling) ? WakeupReasonAcOkOff : WakeupReasonNoReason;
    g_WakeupReason |= (regOnOffIrq & MaskOnOffIrqEn0Rising) ? WakeupReasonPowerButton : WakeupReasonNoReason;

    // XXX: EDEV では各フラグは正しく動作するが、甚だ残念なことに SDEV では ACOK の割り込みは一生来ない (Type-C の挿抜状態に関わらず常時 ACOK=1 なので)
    // XXX: 起床時に素早く挿抜すると AcOkOn / AcOkOff が両方同時に立ったり、ひどいときには両方立たない場合があるので、現時点ではこれだけでは基本信用できない
    NN_DETAIL_BPC_TRACE("Wakeup reason: %s/%s/%s/%s\n",
        (g_WakeupReason & WakeupReasonRtc) ? "RTC" : "-",
        (g_WakeupReason & WakeupReasonAcOkOn) ? "ACOK On" : "-",
        (g_WakeupReason & WakeupReasonAcOkOff) ? "ACOK Off" : "-",
        (g_WakeupReason & WakeupReasonPowerButton) ? "POWER Btn" : "-"
    );
}

void UpdateShutdownReason(nn::i2c::I2cSession& session) NN_NOEXCEPT
{
    uint8_t reg = 0;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&reg, session, &OffsetNverc));

    // save 6 reasons except SHUTDOWN_BATTERY_LOW and SHUTDOWN_WATCHDOG
    g_ShutdownReason = ShutdownReasonNoReason;
    g_ShutdownReason |= (reg & MaskNvercRstIn) ? ShutdownReasonResetInput : ShutdownReasonNoReason;
    g_ShutdownReason |= (reg & MaskNvercMbu) ? ShutdownReasonMainBatteryUnderVoltage : ShutdownReasonNoReason;
    g_ShutdownReason |= (reg & MaskNvercMbo) ? ShutdownReasonMainBatteryOverVoltage : ShutdownReasonNoReason;
    g_ShutdownReason |= (reg & MaskNvercTovld) ? ShutdownReasonTemperatureOverload : ShutdownReasonNoReason;
    g_ShutdownReason |= (reg & MaskNvercHdrst) ? ShutdownReasonShutdownPin : ShutdownReasonNoReason;
    g_ShutdownReason |= (reg & MaskNvercShdn) ? ShutdownReasonShutdownPin : ShutdownReasonNoReason;
}

void UpdateShutdownReasonIfNeeded(nn::i2c::I2cSession& session) NN_NOEXCEPT
{
    // サーバ側マルチスレッド対応時要 Mutex 追加。
    if ( !g_ShutdownReasonUpdated )
    {
        UpdateShutdownReason(session);
        g_ShutdownReasonUpdated = true;
    }
}

} // namespace

void InitializePowerReason() NN_NOEXCEPT
{
}

void UpdateWakeupReason() NN_NOEXCEPT
{
    ReadWakeupReason(*GetPmicI2cSession());
}

void ConfigWakeupEvents() NN_NOEXCEPT
{
    ConfigWakeupInterrupts(*GetPmicI2cSession());
}

nn::Result GetWakeupReason(WakeupReason* pOutWakeupReason) NN_NOEXCEPT
{
    *pOutWakeupReason = g_WakeupReason;
    NN_RESULT_SUCCESS;
}

nn::Result GetShutdownReason(ShutdownReason* pOutShutdownReason) NN_NOEXCEPT
{
    UpdateShutdownReasonIfNeeded(*GetPmicI2cSession());

    *pOutShutdownReason = g_ShutdownReason;
    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::bpc::driver::detail
