﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_SystemThreadDefinition.h>

#include <nn/bpc/detail/bpc_Log.h>
#include <nn/dd.h>
#include <nn/i2c/i2c.h>
#include <nn/os/os_InterruptEvent.h>
#include <nn/result/result_HandlingUtility.h>

#include <nne/max7762x/max7762x_results.h>
#include <nne/max7762x/max7762x_irq_Types.h>
#include <nne/max7762x/max7762x_irq_api.h>

#include "bpc_IrqHandler.h"
#include "bpc_SleepButtonCounter.h"

namespace nn { namespace bpc { namespace driver { namespace detail {

namespace {

// ExternalRtc module has rtc session
nne::max7762x::irq::RegulatorIRQSession g_Session;

nn::os::MultiWaitType g_InterruptWaiter;

// event from PMU_EXT
nn::os::InterruptEventType  g_InterruptEvent;
nn::os::MultiWaitHolderType g_InterruptEventHolder;

// event from the control
enum InterruptHandlerState
{
    InterruptHandlerState_Receiving,
    InterruptHandlerState_Suspended,
    InterruptHandlerState_Terminating
};
InterruptHandlerState       g_InterruptHandlerState;
nn::os::EventType           g_StateChangeEvent;
nn::os::EventType           g_StateChangeAcknowledgeEvent;
nn::os::MultiWaitHolderType g_StateChangeEventHolder;

// event for thread iniitalization
nn::os::EventType           g_InitializedEvent;
nn::os::ThreadType          g_HandleInterruptThread;

const size_t ThreadStackSize = 8192;
NN_ALIGNAS(4096) uint8_t g_InterruptThreadStack[ThreadStackSize];

const int InterruptOffset = 32;
const int PmuExtInterruptNumber = 86 + InterruptOffset; // Interrupt Number PMU_EXT

nn::os::MessageQueueType* pg_OnOffMessageQueue;

nn::os::SystemEventType g_AcOkRisingEvent;

// WORKAROUND: PMIC からの割り込みの極性を反転させる
void InvertInterruptPolarity() NN_NOEXCEPT
{
    const uint32_t PmcCtrlIntrLow = (1 << 17);
    const nn::dd::PhysicalAddress pmuBaseAddress = 0x7000E000;
    const uintptr_t apbdevPmcCntrl0 = (0x0 + 0x400);

    nn::dd::PhysicalAddress ph_a = pmuBaseAddress + apbdevPmcCntrl0;
    nn::dd::WriteIoRegister(ph_a, nn::dd::ReadIoRegister(ph_a) | PmcCtrlIntrLow);
}

void OnOffEventCallback(int onOffEvent) NN_NOEXCEPT
{
    // 割り込みの操作を時系列で取得してハンドリングすることを保証したいので MessageQueue を使用する
    const int MaskAcOkRising  = 0x80;
    const int MaskRising      = 0x08;
    const int MaskFalling     = 0x04;
    const int Mask1sec        = 0x02;
    const int MaskManualResetWarning = 0x01;

    //NN_DETAIL_BPC_TRACE("On/Off Interrupt : %02x.\n", onOffEvent);

    if ( onOffEvent & MaskAcOkRising )
    {
        nn::os::SignalSystemEvent(&g_AcOkRisingEvent);
    }
    if ( onOffEvent & MaskRising )
    {
        nn::os::SendMessageQueue(pg_OnOffMessageQueue, MessageOnOff_Rising);
    }
    if ( onOffEvent & MaskFalling )
    {
        nn::os::SendMessageQueue(pg_OnOffMessageQueue, MessageOnOff_Falling);
    }
    // SIGLO-43468 : 仕様変更により 1sec はレギュレータドライバ上でマスクされて無効になっています。
    if ( onOffEvent & Mask1sec )
    {
        nn::os::SendMessageQueue(pg_OnOffMessageQueue, MessageOnOff_1sec);
    }
    // SIGLO-43468 : 仕様変更により ManualResetWarning はレギュレータドライバ上でマスクされて無効になっています。
    if ( onOffEvent & MaskManualResetWarning )
    {
        nn::os::SendMessageQueue(pg_OnOffMessageQueue, MessageOnOff_ManualResetWarning);
    }
}

void RtcEventCallback(int rtcEvent) NN_NOEXCEPT
{
    // TODO: すべての RTC ハンドラはここからの通知ベースで仕事をするべき（ウェイクリーズンなどで特別扱いしない）
    NN_DETAIL_BPC_TRACE("RTC Callback event = 0x%x\n", rtcEvent);
}

void InterruptHandlerThread(void *arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    nn::os::SignalEvent(&g_InitializedEvent);

    auto& interruptEvent = g_InterruptEvent;
    while ( 1 )
    {
        //NN_DETAIL_BPC_TRACE("while top\n");

        nn::os::MultiWaitHolderType* pEventHolder = nn::os::WaitAny(&g_InterruptWaiter);

        if ( pEventHolder == &g_StateChangeEventHolder )
        {
            nn::os::ClearEvent(&g_StateChangeEvent);

            //NN_DETAIL_BPC_TRACE("finalized\n");
            if (g_InterruptHandlerState == InterruptHandlerState_Receiving)
            {
                nn::os::LinkMultiWaitHolder(&g_InterruptWaiter, &g_InterruptEventHolder);
                nne::max7762x::irq::enableOnOffEvents(&g_Session, OnOffEventCallback);
                // nne::max7762x::irq::enableRTCEvents(&g_Session, RtcEventCallback);
                // RTC のほうはそのまま残す
                NN_DETAIL_BPC_TRACE("IRQ handler: receiving state\n");
            }
            else if (g_InterruptHandlerState == InterruptHandlerState_Suspended)
            {
                nne::max7762x::irq::enableOnOffEvents(&g_Session, nullptr);
                // nne::max7762x::irq::enableRTCEvents(&g_Session, nullptr);
                // RTC のほうはそのまま残す
                nn::os::UnlinkMultiWaitHolder(&g_InterruptEventHolder);
                nn::os::ClearInterruptEvent(&interruptEvent);
                NN_DETAIL_BPC_TRACE("IRQ handler: suspended state\n");
            }
            else if (g_InterruptHandlerState == InterruptHandlerState_Terminating)
            {
                NN_DETAIL_BPC_TRACE("IRQ handler: terminating\n");
                break; // break entire loop
            }

            nn::os::SignalEvent(&g_StateChangeAcknowledgeEvent);
        }
        else if ( pEventHolder == &g_InterruptEventHolder )
        {
            //NN_DETAIL_BPC_TRACE("oh interrupt!\n");

            nn::os::WaitInterruptEvent(&interruptEvent);
            nne::max7762x::irq::processInterrupt(&g_Session);
            nn::os::ClearInterruptEvent(&interruptEvent);
        }
    }
    NN_DETAIL_BPC_TRACE("Interrupt Handler Thread is terminating.\n");
}

void InitializeInterruptHandler() NN_NOEXCEPT
{
    nn::Result result;
    g_InterruptHandlerState = InterruptHandlerState_Receiving;

    nn::os::CreateSystemEvent(&g_AcOkRisingEvent, nn::os::EventClearMode_ManualClear, true);

    nn::os::InitializeEvent(&g_InitializedEvent, false, nn::os::EventClearMode_AutoClear);

    nn::os::InitializeMultiWait(&g_InterruptWaiter);

    nn::os::InitializeInterruptEvent(&g_InterruptEvent, PmuExtInterruptNumber, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&g_StateChangeEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&g_StateChangeAcknowledgeEvent, false, nn::os::EventClearMode_AutoClear);

    nn::os::InitializeMultiWaitHolder(&g_InterruptEventHolder, &g_InterruptEvent);
    nn::os::InitializeMultiWaitHolder(&g_StateChangeEventHolder, &g_StateChangeEvent);

    nn::os::LinkMultiWaitHolder(&g_InterruptWaiter, &g_InterruptEventHolder);
    nn::os::LinkMultiWaitHolder(&g_InterruptWaiter, &g_StateChangeEventHolder);

    result = nn::os::CreateThread(&g_HandleInterruptThread, InterruptHandlerThread,
                                  nullptr, g_InterruptThreadStack, ThreadStackSize,
                                  NN_SYSTEM_THREAD_PRIORITY(bpc, IrqHandler));

    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::os::SetThreadNamePointer(&g_HandleInterruptThread, NN_SYSTEM_THREAD_NAME(bpc, IrqHandler));

    nn::os::StartThread(&g_HandleInterruptThread);

    NN_DETAIL_BPC_TRACE("Waiting for initialization of Interrupt Handler Thread.\n");

    NN_ABORT_UNLESS(nn::os::TimedWaitEvent(&g_InitializedEvent, nn::TimeSpan::FromMilliSeconds(5000LL)), "Failed to initialize Interrupt Handler Thread.");

    NN_DETAIL_BPC_TRACE("Interrupt Handler Thread has been initialized.\n");
}

void FinalizeInterruptHandler() NN_NOEXCEPT
{
    g_InterruptHandlerState = InterruptHandlerState_Terminating;
    nn::os::SignalEvent(&g_StateChangeEvent);

    nn::os::WaitThread(&g_HandleInterruptThread);
    nn::os::DestroyThread(&g_HandleInterruptThread);

    nn::os::UnlinkMultiWaitHolder(&g_StateChangeEventHolder);
    nn::os::UnlinkMultiWaitHolder(&g_InterruptEventHolder);

    nn::os::FinalizeMultiWaitHolder(&g_StateChangeEventHolder);
    nn::os::FinalizeMultiWaitHolder(&g_InterruptEventHolder);

    nn::os::FinalizeEvent(&g_StateChangeAcknowledgeEvent);
    nn::os::FinalizeEvent(&g_StateChangeEvent);
    nn::os::FinalizeInterruptEvent(&g_InterruptEvent);

    nn::os::FinalizeMultiWait(&g_InterruptWaiter);

    nn::os::FinalizeEvent(&g_InitializedEvent);

    nn::os::DestroySystemEvent(&g_AcOkRisingEvent);

    NN_DETAIL_BPC_TRACE("Interrupt Handler Thread has been terminated.\n");
}

} // namespace

void InitializeIrqHandler(nn::os::MessageQueueType* pMessageQueue) NN_NOEXCEPT
{
    InvertInterruptPolarity();

    pg_OnOffMessageQueue = pMessageQueue;

    // open session
    nne::max7762x::Result result = nne::max7762x::irq::OpenSession(&g_Session, "max77620_irq0");
    NN_ABORT_UNLESS(result == nne::max7762x::Result_Success, "I2C transaction error");

    // add on/off & RTC callback
    nne::max7762x::irq::enableOnOffEvents(&g_Session, OnOffEventCallback);
    nne::max7762x::irq::enableRTCEvents(&g_Session, RtcEventCallback);

    // start interrupt handler
    InitializeInterruptHandler();
}

void FinalizeIrqHandler() NN_NOEXCEPT
{
    // stop interrupt handler
    FinalizeInterruptHandler();

    // close session
    nne::max7762x::Result result = nne::max7762x::irq::CloseSession(&g_Session);
    NN_ABORT_UNLESS(result == nne::max7762x::Result_Success, "I2C transaction error");

    pg_OnOffMessageQueue = nullptr;
}

void SuspendIrqHandler() NN_NOEXCEPT
{
    g_InterruptHandlerState = InterruptHandlerState_Suspended;
    nn::os::SignalEvent(&g_StateChangeEvent);
    nn::os::WaitEvent(&g_StateChangeAcknowledgeEvent);
}

void ResumeIrqHandler() NN_NOEXCEPT
{
    g_InterruptHandlerState = InterruptHandlerState_Receiving;
    nn::os::SignalEvent(&g_StateChangeEvent);
    nn::os::WaitEvent(&g_StateChangeAcknowledgeEvent);
}

void GetBoardPowerControlEventPtr(nn::os::SystemEventType** pOutEventPtr, BoardPowerControlEventTarget target) NN_NOEXCEPT
{
    switch ( target )
    {
    case BoardPowerControlEventTarget_AcOkRising:
        *pOutEventPtr = &g_AcOkRisingEvent;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

}}}} // namespace nn::bpc::driver::detail
