﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>

#include <nn/os.h>

#include <nn/bpc/bpc_Result.h>
#include <nn/bpc/bpc_ResultPrivate.h>
#include <nn/bpc/bpc_RtcTypes.h>
#include <nn/bpc/detail/bpc_Log.h>
#include <nn/bpc/driver/bpc_Rtc.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_Pcv.h>
#include <nn/spl/spl_Api.h>

#include <nne/max7762x/max7762x_results.h>
#include <nne/max7762x/max7762x_rtc_Types.h>
#include <nne/max7762x/max7762x_rtc_api.h>

#include "bpc_ExternalRtc.h"
#include "bpc_PmicAccessor-hardware.nx.h"
#include "bpc_RtcAccessor-hardware.nx.h"

namespace nn { namespace bpc { namespace driver { namespace detail {

namespace {

nn::spl::HardwareType g_HardwareType;
nne::max7762x::rtc::RegulatorRTCSession     g_RtcSession;

bool g_RtcResetOnShutdown = false;

nne::max7762x::rtc::RTC_ALARM_ID ToPmicRtcAlarmId(nn::bpc::driver::detail::RtcAlarmType type) NN_NOEXCEPT
{
    switch (type)
    {
        case nn::bpc::driver::detail::RtcAlarmType::Wakeup:
        {
            return nne::max7762x::rtc::RTC_ALARM1;
        }
        case nn::bpc::driver::detail::RtcAlarmType::QuasiOff:
        {
            return nne::max7762x::rtc::RTC_ALARM2;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

// 32kHz 発振器の設定
void Set32kLoad(nn::i2c::I2cSession& session) NN_NOEXCEPT
{
    const uint8_t index = 0x03; // CNFG1_32K
    const uint8_t pwrModeValue = 0x00;  // Low Power Mode
    const uint8_t pwrModeMask = 0x03;  // PWR_MD_32K
    const uint8_t loadValue = 0x30;  // 10pF per node
    const uint8_t loadMask = 0x30;  // 32KLOAD
    uint8_t reg = 0;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&reg, session, &index));
    reg = (reg & ~pwrModeMask) | (pwrModeValue & pwrModeMask);
    reg = (reg & ~loadMask) | (loadValue & loadMask);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(session, &index, &reg));

    // SIGLO-36572: メーカから回答。安定までは数百 usec 待つのが良い。
    const auto delaySpan = nn::TimeSpan::FromMicroSeconds(500);

    nn::os::SleepThread(delaySpan);
}

int IsLeapYear(int year) NN_NOEXCEPT
{
    // 通常うるう年は
    //  - 400年で割り切れたらうるう年である
    //  - 100年で割り切れたらうるう年でない
    //  -   4年で割り切れたらうるう年である
    // という判定が必要だが
    // maxim77620 のうるう年の仕様は 4 で割り切れるかどうかだけなのでそれに従う
    // ex) 0, 4, 8, ... 92, 96 という4の倍数の年がうるう年扱い
    return ( year % 4 == 0 ) ? 1 : 0;
}

int DateToDays(int year, int month, int day) NN_NOEXCEPT
{
    //
    // CTR fnd::DateTime::DateToDays から持ってきたコード
    // 西暦2000年1月1日 00:00:00 を基点として経過秒数を計算しているのを
    // 西暦0年1月1日 00:00:00 の基点に修正
    //

    day -= 1;
    // year -= 2000; // 基点変更 西暦2000年 -> 西暦0年

    if ( month <= 2 )
    {
        month += (12 - 3);
        year -= 1;
    }
    else
    {
        month -= 3;
    }

    int offset = 1; // 西暦0年がうるう年なのでデフォルトで1のオフセット

    if ( year < 0 )
    {
        offset = IsLeapYear(year);
    }

    return ((((365 * 4 + 1) * 25 - 1) * 4 + 1) * (year / 100) / 4)
            + (( 365 * 4 + 1) * (year % 100) / 4)
            + (153 * month + 2) / 5
            + day + (31 + 28)
            + offset;
}

void DaysToDate(int *pOutYear, int *pOutMonth, int *pOutDay, int inDays) NN_NOEXCEPT
{
    int days = inDays;

    // CTR nn::fnd::DateTime::DaysToDate から拝借.
    // フェアフィールドの公式を変形して利用している.
    //
    // *** max77620 向けアレンジ ***
    // max77620 のうるう年の仕様は 4 で割り切れるかどうかだけなので、
    // 「100 の倍数の年はうるう年でない」ルールを除外する変更を足している。

    // const int c4days = (((365 * 4 + 1) * 25 - 1) * 4 + 1);
    // const int c1days =  ((365 * 4 + 1) * 25 - 1);
    const int c4days = (((365 * 4 + 1) * 25 - 0) * 4 + 0); // max77620 向け: 100 の倍数年はすべてうるう年ということにする
    const int c1days =  ((365 * 4 + 1) * 25 - 0); // max77620 向け: 100 の倍数年はすべてうるう年ということにする
    const int y4days =   (365 * 4 + 1);
    const int y1days =    365;
    int year, month, day;

    days -= 31 + 28;
    // days += 365; // フェアフィールドの公式は西暦0年がベースになってるので1年をベースにする
    days -= 1; // 西暦0年をベースとし、起点がうるう年である分を補正するため -1 する

    int c4    = days / c4days;
    int c4ds  = days % c4days;

    if (c4ds < 0)
    {
        c4ds += c4days;
        c4 -= 1;
    }

    int c1    = c4ds / c1days;
    int c1ds  = c4ds % c1days;
    int y4    = c1ds / y4days;
    int y4ds  = c1ds % y4days;
    int y1    = y4ds / y1days;
    int ydays = y4ds % y1days;

    {
        // CTR で2000年からになっていたのを0年にする.
        // ここで2000年ベースを単純に1年ベースにするとうるう年判定が狂うので、
        // 0にして(2000年も0年もうるう年)、↑の days+=365 で調整

        // year = 2000 + c4 * 400 + c1 * 100 + y4 * 4 + y1;
        year = c4 * 400 + c1 * 100 + y4 * 4 + y1;
    }
    month = (5 * ydays + 2) / 153;
    day = ydays - (153 * month + 2) / 5 + 1;

    if (y1 == 4 || c1 == 4)
    {
        month = 2 + (12 - 3);
        day = 29;
        year -= 1;
    }

    if ( month <= (12 - 3) )
    {
        month += 3;
    }
    else
    {
        month -= 12 - 3;
        year += 1;
    }

    if ( pOutYear )
    {
        *pOutYear = year;
    }
    if ( pOutMonth )
    {
        *pOutMonth = month;
    }
    if ( pOutDay )
    {
        *pOutDay = day;
    }
}

nn::Result PmicValueToSeconds(
    int64_t *pOutSeconds,
    int inYear,
    int inMonth,
    int inDay,
    int inHours,
    int inMinutes,
    int inSeconds) NN_NOEXCEPT
{
    // Maxim77620 レジスタの有効ビットマスク
    const int MaskYear      = (1 << 8) - 1; // 8bit
    const int MaskMonth     = (1 << 5) - 1; // 5bit
    const int MaskDay       = (1 << 6) - 1; // 6bit
    const int MaskHour      = (1 << 5) - 1; // 5bit
    const int MaskMinute    = (1 << 7) - 1; // 7bit
    const int MaskSecond    = (1 << 7) - 1; // 7bit

    // Maxim77620の仕様をもとに有効ビットをMASK
    const int year        = inYear      & MaskYear;
    const int month       = inMonth     & MaskMonth;
    const int day         = inDay       & MaskDay;
    const int64_t hours   = static_cast<int64_t>( inHours & MaskHour );
    const int64_t minutes = static_cast<int64_t>( inMinutes & MaskMinute );
    const int64_t seconds = static_cast<int64_t>( inSeconds & MaskSecond );

    // NN_DETAIL_BPC_TRACE(": max7762x RTC / Year:%02d Month:%02d Day:%02d %02lld:%02lld:%02lld\n",
        // year, month, day, hours, minutes, seconds);

    // Maxim77620の仕様をもとに有効値チェック
    NN_RESULT_THROW_UNLESS(
        (year >= 0 || year <= 99) ||
        (month >= 1 || month <= 12) ||
        (day >= 1 || day <= 31) ||
        (hours >= 0 || hours <= 23) ||
        (minutes >= 0 || minutes <= 59) ||
        (seconds >= 0 || seconds <= 59),
        nn::bpc::ResultRtcInvalid());

    const int64_t days = static_cast<int64_t>( DateToDays(year, month, day) );
    *pOutSeconds = (((days * 24) + hours) * 60 + minutes ) * 60 + seconds;

    NN_RESULT_SUCCESS;
}

void SecondsToPmicValue(
    int *pOutYear,
    int *pOutMonth,
    int *pOutDay,
    int *pOutHour,
    int *pOutMinute,
    int *pOutSecond,
    int64_t seconds) NN_NOEXCEPT
{
    int days = static_cast<int>( seconds / (24LL * 60LL * 60LL) );
    DaysToDate(pOutYear, pOutMonth, pOutDay, days);
    *pOutYear %= 100; // PMIC では年は2桁 (0-99) 表現なので丸める

    int remain = static_cast<int>( seconds % (24LL * 60LL * 60LL) );

    *pOutHour = remain / (60 * 60);
    remain %= (60 * 60);

    *pOutMinute = remain / 60;
    remain %= 60;

    *pOutSecond = remain;
}

nn::Result ReadRtcDoubleBufferredRegister(uint8_t* pOutValue, nn::i2c::I2cSession& session, const uint8_t* pAddress) NN_NOEXCEPT
{
    const uint8_t RtcUpdate0Address = 0x04;
    const uint8_t RbudrMask = 0x10;
    const int MaxRetryCount = 4;

    NN_RESULT_DO(nn::i2c::WriteSingleRegister(session, &RtcUpdate0Address, &RbudrMask));

    int count = 0;
    uint8_t value = 0x00;
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        NN_RESULT_DO(nn::i2c::ReadSingleRegister(&value, session, &RtcUpdate0Address));
        count++;
    }
    while ( ((value & RbudrMask) == RbudrMask) && count < MaxRetryCount );

    if ( (value & RbudrMask) == RbudrMask )
    {
        return ResultRtcAccessFailed();
    }

    NN_RESULT_DO(nn::i2c::ReadSingleRegister(pOutValue, session, pAddress));

    NN_RESULT_SUCCESS;
}

nn::Result WriteRtcDoubleBufferredRegister(nn::i2c::I2cSession& session, const uint8_t* pAddress, const uint8_t* pValue) NN_NOEXCEPT
{
    const uint8_t RtcUpdate0Address = 0x04;
    const uint8_t UdrMask = 0x01;
    const int MaxRetryCount = 4;

    NN_RESULT_DO(nn::i2c::WriteSingleRegister(session, pAddress, pValue));

    NN_RESULT_DO(nn::i2c::WriteSingleRegister(session, &RtcUpdate0Address, &UdrMask));

    int count = 0;
    uint8_t value = 0x00;
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        NN_RESULT_DO(nn::i2c::ReadSingleRegister(&value, session, &RtcUpdate0Address));
        count++;
    }
    while ( ((value & UdrMask) == UdrMask) && count < MaxRetryCount );

    if ( (value & UdrMask) == UdrMask )
    {
        return nn::bpc::ResultRtcAccessFailed();
    }

    NN_RESULT_SUCCESS;
}

bool IsRtcResetDetected() NN_NOEXCEPT
{
    const uint8_t RtcCntlAddress = 0x03;
    const uint8_t ExpectedMask = 0x03;
    const uint8_t ExpectedValue = 0x02;

    uint8_t value = 0x00;

    // RTCCNTL(0x03) の下位 2 bit が 0x02 であり、Binary モードかつ 24 時間モードであることを期待する。
    NN_ABORT_UNLESS_RESULT_SUCCESS(ReadRtcDoubleBufferredRegister(&value, *(GetRtcI2cSession()), &RtcCntlAddress));

    return (ExpectedValue != (value & ExpectedMask));
}

// 外部 RTC の時刻レジスタおよびアラームレジスタをクリアします。
void ClearRtcTime() NN_NOEXCEPT
{
    const uint8_t RtcSecAddress = 0x07;
    const uint8_t RtcMinAddress = 0x08;
    const uint8_t RtcHourAddress = 0x09;
    const uint8_t RtcDowAddress = 0x0a;
    const uint8_t RtcMonthAddress = 0x0b;
    const uint8_t RtcYearAddress = 0x0c;
    const uint8_t RtcDomAddress = 0x0d;

    const uint8_t RtcSecA1Address = 0x0e;
    const uint8_t RtcMinA1Address = 0x0f;
    const uint8_t RtcHourA1Address = 0x10;
    const uint8_t RtcDowA1Address = 0x11;
    const uint8_t RtcMonthA1Address = 0x12;
    const uint8_t RtcYearA1Address = 0x13;
    const uint8_t RtcDomA1Address = 0x14;

    const uint8_t RtcSecA2Address = 0x15;
    const uint8_t RtcMinA2Address = 0x16;
    const uint8_t RtcHourA2Address = 0x17;
    const uint8_t RtcDowA2Address = 0x18;
    const uint8_t RtcMonthA2Address = 0x19;
    const uint8_t RtcYearA2Address = 0x1a;
    const uint8_t RtcDomA2Address = 0x1b;

    const uint8_t value0 = 0x00;
    const uint8_t value1 = 0x01;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcSecAddress, &value0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcMinAddress, &value0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcHourAddress, &value0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcDowAddress, &value1));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcMonthAddress, &value1));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcYearAddress, &value0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcDomAddress, &value1));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcSecA1Address, &value0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcMinA1Address, &value0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcHourA1Address, &value0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcDowA1Address, &value1));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcMonthA1Address, &value1));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcYearA1Address, &value0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcDomA1Address, &value1));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcSecA2Address, &value0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcMinA2Address, &value0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcHourA2Address, &value0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcDowA2Address, &value1));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcMonthA2Address, &value1));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(*(GetRtcI2cSession()), &RtcYearA2Address, &value0));

    // 最後に DoubleBufferredRegister をまとめてフラッシュします。
    NN_ABORT_UNLESS_RESULT_SUCCESS(WriteRtcDoubleBufferredRegister(*(GetRtcI2cSession()), &RtcDomA2Address, &value1));
}

// 外部 RTC を 12 時間モードに設定し時刻を初期化します。
// BCD フィールドは操作しません。
// 事後 RTCCNTLM は 0x00 になります。
void SetRtc12hMode() NN_NOEXCEPT
{
    const uint8_t RtcCtrlMAddress = 0x02;
    const uint8_t RtcCtrlAddress = 0x03;
    const uint8_t RtcCtrlMHrModeM = 0x02;
    const uint8_t RtcCtrlMNoMask = 0x00;
    const uint8_t RtcCtrlHrMode12hMode = 0x00;

    NN_ABORT_UNLESS_RESULT_SUCCESS(WriteRtcDoubleBufferredRegister(*(GetRtcI2cSession()), &RtcCtrlMAddress, &RtcCtrlMHrModeM));
    NN_ABORT_UNLESS_RESULT_SUCCESS(WriteRtcDoubleBufferredRegister(*(GetRtcI2cSession()), &RtcCtrlAddress, &RtcCtrlHrMode12hMode));

    ClearRtcTime();

    NN_ABORT_UNLESS_RESULT_SUCCESS(WriteRtcDoubleBufferredRegister(*(GetRtcI2cSession()), &RtcCtrlMAddress, &RtcCtrlMNoMask));
}

} // anonymous namespace

void InitializeExternalRtc(bool recoveryFromRtcReset) NN_NOEXCEPT
{
    nn::spl::Initialize();
    g_HardwareType = nn::spl::GetHardwareType();
    nn::spl::Finalize();

    if ( recoveryFromRtcReset == false )
    {
        // セーフモードの場合、RTC リセットを検知したらアボート
        //
        // セーフモードでは、リセット値をシステムパーティションに保存できない場合がある。
        // 保存できない場合に、意図的に時間を巻き戻すなどの不正に繋がる恐れがあるため、
        // RTC リセットを検知した時点で、RTC のレジスタを維持したままアボートする。
        // RTC がリセットされた状態でセーフモードに入ること自体がかなりのレアケース。
        NN_ABORT_UNLESS(IsRtcResetDetected() == false);
    }
    else
    {
        // RTC のリセットが不揮発領域に保存されていない場合。
        if ( !(nn::settings::system::IsExternalRtcReset()) )
        {
            // OpenSession する前に外部 RTC の動作モードを確認し、期待と異なる場合リセットが発生したと判断する。
            auto isRtcResetDetected = IsRtcResetDetected();

            if ( isRtcResetDetected )
            {
                NN_DETAIL_BPC_TRACE("External RTC reset detected.\n");

                // RTC のリセットを不揮発領域に保存する。
                nn::settings::system::SetExternalRtcReset(isRtcResetDetected);

                // 永続化を保証するために読み込む
                // SIGLO-67781: 未初期化の本体ではこの判定に失敗するため ERROR メッセージを表示するまでにとどめます。
                if ( isRtcResetDetected != nn::settings::system::IsExternalRtcReset() )
                {
                    NN_DETAIL_BPC_ERROR("Error on saving reset detection.\n");
                }
            }
        }
    }

    // PMIC 側のレジスタで 32K_LOAD の値を設定する。
    auto* pSession = GetPmicI2cSession();
    Set32kLoad(*pSession);

    auto result = nne::max7762x::rtc::OpenSession(&g_RtcSession, "max77620_rtc0");
    NN_ABORT_UNLESS(result == nne::max7762x::Result_Success, "max77620_rtc0 error");

    // 起動時にアラームをクリアする
    NN_ABORT_UNLESS_RESULT_SUCCESS(DisableRtcAlarm(RtcAlarmType::Wakeup));
    NN_ABORT_UNLESS_RESULT_SUCCESS(DisableRtcAlarm(RtcAlarmType::QuasiOff));

// 時刻演算のロジック確認用コード。 SecondsToPmicValue / PmicValueToSeconds 内のロジックを変更したら実行
#if 0
    for (int64_t days = 0; days < ((365 * 4 + 1) * 25); ++days )
    {
        auto checkFunc = [](int64_t expectedSeconds) NN_NOEXCEPT
        {
            NN_DETAIL_BPC_TRACE("Self test: expectedSeconds %lld\n", expectedSeconds);
            nne::max7762x::rtc::RTC_TIME rtcBuffer;
            SecondsToPmicValue(
                &rtcBuffer.year,
                &rtcBuffer.month,
                &rtcBuffer.day_of_month,
                &rtcBuffer.hours,
                &rtcBuffer.minutes,
                &rtcBuffer.seconds,
                expectedSeconds);
            NN_DETAIL_BPC_TRACE("Self test: rtcBuffer %d/%d/%d %d:%d:%d\n",
                        rtcBuffer.year, rtcBuffer.month, rtcBuffer.day_of_month, rtcBuffer.hours, rtcBuffer.minutes, rtcBuffer.seconds);

            int64_t actualSeconds;
            NN_ABORT_UNLESS_RESULT_SUCCESS(PmicValueToSeconds(
                &actualSeconds,
                rtcBuffer.year,
                rtcBuffer.month,
                rtcBuffer.day_of_month,
                rtcBuffer.hours,
                rtcBuffer.minutes,
                rtcBuffer.seconds));

            NN_DETAIL_BPC_TRACE("Self test: actualSeconds %lld\n", actualSeconds);
            NN_ABORT_UNLESS_EQUAL(expectedSeconds, actualSeconds);
        };

        const int64_t SecondsAtDayBegin = days * 24 * 60 * 60;
        const int64_t SecondsAtDayEnd = (days + 1) * 24 * 60 * 60 - 1;
        checkFunc(SecondsAtDayBegin);
        checkFunc(SecondsAtDayEnd);
        NN_DETAIL_BPC_TRACE("Self test: Checked day %d / %d\n", days, ((365 * 4 + 1) * 25) - 1);
    }
#endif

}

nn::Result GetRtcTime(int64_t *pOutRtcTimeInSeconds) NN_NOEXCEPT
{
    // maxim77620 からRTC値取得
    nne::max7762x::rtc::RTC_TIME rtcBuffer;
    auto result = nne::max7762x::rtc::readRTC(&g_RtcSession, &rtcBuffer);
    // NN_DETAIL_BPC_TRACE("nne::max7762x::rtc::readRTC result:%d\n", result);
    NN_RESULT_THROW_UNLESS(result == nne::max7762x::Result_Success, nn::bpc::ResultRtcAccessFailed());

    NN_RESULT_DO(PmicValueToSeconds(
        pOutRtcTimeInSeconds,
        rtcBuffer.year,
        rtcBuffer.month,
        rtcBuffer.day_of_month,
        rtcBuffer.hours,
        rtcBuffer.minutes,
        rtcBuffer.seconds));

    NN_RESULT_SUCCESS;
}

nn::Result SetRtcTime(int64_t rtcTimeInSeconds) NN_NOEXCEPT
{
    nne::max7762x::rtc::RTC_TIME rtcBuffer; // day_of_the_week は設定しない（対応予定なし）
    SecondsToPmicValue(
        &rtcBuffer.year,
        &rtcBuffer.month,
        &rtcBuffer.day_of_month,
        &rtcBuffer.hours,
        &rtcBuffer.minutes,
        &rtcBuffer.seconds,
        rtcTimeInSeconds);

    auto result = nne::max7762x::rtc::writeRTC(&g_RtcSession, &rtcBuffer);
    // NN_DETAIL_BPC_TRACE("nne::max7762x::rtc::writeRTC result:%d\n", result);
    NN_RESULT_THROW_UNLESS(result == nne::max7762x::Result_Success, nn::bpc::ResultRtcAccessFailed());

    NN_RESULT_SUCCESS;
}

nn::Result EnableRtcAlarm(int64_t targetTime, RtcAlarmType type) NN_NOEXCEPT
{
    auto alarmId = ToPmicRtcAlarmId(type);

    // Copper では擬似オフが存在しないので、RTC Alarm 2 を常に無効にする
    if (g_HardwareType == nn::spl::HardwareType_Copper && alarmId == nne::max7762x::rtc::RTC_ALARM2)
    {
        NN_RESULT_SUCCESS;
    }

    nne::max7762x::rtc::RTC_TIME rtcBuffer; // day_of_the_week は設定しない（対応予定なし）
    SecondsToPmicValue(
        &rtcBuffer.year,
        &rtcBuffer.month,
        &rtcBuffer.day_of_month,
        &rtcBuffer.hours,
        &rtcBuffer.minutes,
        &rtcBuffer.seconds,
        targetTime);

    auto result = nne::max7762x::rtc::writeAlarm(&g_RtcSession, alarmId, &rtcBuffer);
    // NN_DETAIL_BPC_TRACE("nne::max7762x::rtc::writeAlarm result:%d\n", result);
    NN_RESULT_THROW_UNLESS(result == nne::max7762x::Result_Success, nn::bpc::ResultRtcAlarmAccessFailed());

    result = nne::max7762x::rtc::enableRTCAlarm(&g_RtcSession, alarmId, true);
    // NN_DETAIL_BPC_TRACE("nne::max7762x::rtc::enableRTCAlarm(enable) result:%d\n", result);
    NN_RESULT_THROW_UNLESS(result == nne::max7762x::Result_Success, nn::bpc::ResultRtcAlarmAccessFailed());

    NN_RESULT_SUCCESS;
}

nn::Result DisableRtcAlarm(RtcAlarmType type) NN_NOEXCEPT
{
    auto alarmId = ToPmicRtcAlarmId(type);

    auto result = nne::max7762x::rtc::enableRTCAlarm(&g_RtcSession, alarmId, false);
    // NN_DETAIL_BPC_TRACE("nne::max7762x::rtc::enableRTCAlarm(disable) result:%d\n", result);
    NN_RESULT_THROW_UNLESS(result == nne::max7762x::Result_Success, nn::bpc::ResultRtcAlarmAccessFailed());

    // 割り込みをマスクしてても割り込みフラグ自体は時間が来ると立ってしまうので、
    // マスク後に絶対に到達しない無効な時刻 (99-2-30 23:59:59) をセットしておく
    nne::max7762x::rtc::RTC_TIME rtcBuffer;
    rtcBuffer.year = 99;
    rtcBuffer.month = 2;
    rtcBuffer.day_of_month = 30;
    rtcBuffer.day_of_the_week = 0;
    rtcBuffer.hours = 23;
    rtcBuffer.minutes = 59;
    rtcBuffer.seconds = 59;

    result = nne::max7762x::rtc::writeAlarm(&g_RtcSession, alarmId, &rtcBuffer);
    // NN_DETAIL_BPC_TRACE("nne::max7762x::rtc::writeAlarm result:%d\n", result);
    NN_RESULT_THROW_UNLESS(result == nne::max7762x::Result_Success, nn::bpc::ResultRtcAlarmAccessFailed());

    NN_RESULT_SUCCESS;
}

nn::Result GetRtcResetDetected(bool* pOutRtcResetDetected) NN_NOEXCEPT
{
    *pOutRtcResetDetected = nn::settings::system::IsExternalRtcReset();

    NN_RESULT_SUCCESS;
}

nn::Result ClearRtcResetDetected() NN_NOEXCEPT
{
    // RTC のリセットを不揮発領域に保存する。
    nn::settings::system::SetExternalRtcReset(false);

    // 永続化を保証するために読み込む
    NN_ABORT_UNLESS(!(nn::settings::system::IsExternalRtcReset()), "Error on clearing rtc reset detection");

    NN_RESULT_SUCCESS;
}

nn::Result SetUpRtcResetOnShutdown() NN_NOEXCEPT
{
    g_RtcResetOnShutdown = true;

    NN_RESULT_SUCCESS;
}

// シャットダウン（リブート・シャットダウンの両方を含む）が行われる直前に RTC に関連する処理を行わせるために呼ばれます。
void NotifyShutdownToRtc() NN_NOEXCEPT
{
    // 時刻モードを 12 時間モードにして初期化します。
    // RTC アラームはクリアされます。
    if ( g_RtcResetOnShutdown )
    {
        SetRtc12hMode();
    }
}

}}}} // namespace nn::bpc::driver::detail
