﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>

#include <nn/os.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include <nn/bpc/detail/bpc_Log.h>
#include <nn/bpc/driver/bpc.h>
#include <nn/i2c/i2c.h>
#include <nn/result/result_HandlingUtility.h>

// TODO: max7762x_results.h を他の max7762x ヘッダの前に置かねばならない制約の除去。
#include <nne/max7762x/max7762x_results.h>
#include <nne/max7762x/max7762x_onoff_Types.h>
#include <nne/max7762x/max7762x_onoff_api.h>

// TODO: max7762x_gpio_Types.h を max7762x_gpio_api.h の前に置かねばならない制約の除去。
#include <nne/max7762x/max7762x_gpio_Types.h>
#include <nne/max7762x/max7762x_gpio_api.h>

#include "bpc_BoardPowerController.h"
#include "bpc_ExternalRtc.h"
#include "bpc_PmicAccessor-hardware.nx.h"
#include "bpc_RtcAccessor-hardware.nx.h"

namespace nn { namespace bpc { namespace driver { namespace detail {

namespace {

// TOEIAEZU(temporary) wait 5 seconds
const int TimeoutMilliSeconds = 5000;

class SettingsHolder
{
public:
    SettingsHolder() NN_NOEXCEPT : m_IsQuasiOffEnabled(false) {}
    ~SettingsHolder() NN_NOEXCEPT = default;

    void Load() NN_NOEXCEPT
    {
        if ( nn::settings::fwdbg::GetSettingsItemValue(&m_IsQuasiOffEnabled, sizeof(m_IsQuasiOffEnabled), "bpc", "enable_quasi_off") != sizeof(m_IsQuasiOffEnabled) )
        {
            NN_DETAIL_BPC_WARN("Fail to read bpc.enable_quasi_off value in firmware debug settings.\n");
            m_IsQuasiOffEnabled = false;
        }
    }

    bool IsQuasiOffEnabled() const NN_NOEXCEPT
    {
        return m_IsQuasiOffEnabled;
    }

private:
    bool m_IsQuasiOffEnabled;
};
SettingsHolder g_SettingsHolder;

void FinalizeFuelgauge() NN_NOEXCEPT
{
    nn::i2c::I2cSession max17050Session;
    nn::i2c::Initialize();
    nn::i2c::OpenSession(&max17050Session, nn::i2c::I2cDevice::I2cDevice_Max17050);

    // SIGLO-65162: SHDNTIMER が短く、緩和時間が確保できていない市場本体に対しての一括修正。
    const uint8_t ShutdownTimerAddress = 0x3f;

    // SIGLO-65162: 1.6 時間（デフォルト値）
    const uint16_t ShutdownTimerValue = 0xE000;
    ::nn::Result result;
    result = nn::i2c::WriteSingleRegister(max17050Session, &ShutdownTimerAddress, &ShutdownTimerValue);
    if (result.IsFailure())
    {
        // SIGLO-42947: Copper 等電池残量 IC を搭載しないハードウェアの場合、必ず WARN メッセージが表示される。
        NN_DETAIL_BPC_WARN("Fuelgauge I2C write access failure. (%08x)\n", result);
        return;
    }

    const uint8_t ConfigAddress = 0x1d;
    const uint16_t I2cShutdownMask = 0x0040;
    uint16_t config;
    result = nn::i2c::ReadSingleRegister(&config, max17050Session, &ConfigAddress);
    if (result.IsFailure())
    {
        NN_DETAIL_BPC_ERROR("Fuelgauge I2C read access failure. (%08x)\n", result);
        return;
    }

    bool acOk;
    result = GetAcOk(&acOk);
    if (result.IsFailure())
    {
        NN_DETAIL_BPC_ERROR("GetAcOk() failure.\n");
    }

    uint16_t newConfig;
    // AcOk の取得失敗時にはシャットダウンモードにしない
    if (result.IsFailure() || acOk)
    {
        newConfig = config & ~I2cShutdownMask;
    }
    else
    {
        newConfig = config | I2cShutdownMask;
    }

    if (newConfig != config)
    {
        result = nn::i2c::WriteSingleRegister(max17050Session, &ConfigAddress, &newConfig);
        if (result.IsFailure())
        {
            NN_DETAIL_BPC_ERROR("Fuelgauge I2C write access failure. (%08x)\n", result);
            return;
        }
    }
}

nn::Result SetQuasiOffTimer(bool enable) NN_NOEXCEPT
{
    if ( enable )
    {
        int64_t currentTime;
        NN_RESULT_DO(GetRtcTime(&currentTime));
        int64_t targetTime = currentTime + 15; // 現在から 15 秒後

        NN_RESULT_DO(EnableRtcAlarm(targetTime, RtcAlarmType::QuasiOff));
    }
    else
    {
        NN_RESULT_DO(DisableRtcAlarm(RtcAlarmType::QuasiOff));
    }

    NN_RESULT_SUCCESS;
}

// XXX: Programs/Eris/Sources/Processes/boot/boot_PmicDriver.cpp の実装のコピー
// 本関数内のロジックを修正する際は、コピー元のロジックも同様に修正する必要があるか検討すること
NN_NORETURN void ShutdownSystemImpl(bool isReboot) NN_NOEXCEPT
{
    const uint8_t onoffcnfg1Address = 0x41;
    const uint8_t onoffcnfg2Address = 0x42;
    const uint8_t sftRstMask = 0x80;
    const uint8_t sftRstWkMask = 0x80;

    auto& session = *GetPmicI2cSession();

    uint8_t onoffcnfg2Value = 0x00;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&onoffcnfg2Value, session, &onoffcnfg2Address));
    if (isReboot)
    {
        onoffcnfg2Value = onoffcnfg2Value | sftRstWkMask;
    }
    else
    {
        onoffcnfg2Value = onoffcnfg2Value & (~sftRstWkMask);
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(session, &onoffcnfg2Address, &onoffcnfg2Value));

    uint8_t onoffcnfg1Value = 0x00;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&onoffcnfg1Value, session, &onoffcnfg1Address));
    onoffcnfg1Value = onoffcnfg1Value | sftRstMask;

    // 誰かが Wakeup アラームをセットしたままの状態だったら誤爆するため、明示的に解除
    NN_ABORT_UNLESS_RESULT_SUCCESS(DisableRtcAlarm(RtcAlarmType::Wakeup));

    // (SIGLO-49391) Workaround: 擬似オフに行くはずが通常起動してしまう誤爆を避けるため RTC アラームの割り込みフラグを明示的に read-clear しておく
    // IAAA-4278 で bootloader がマスクを考慮して判断するようになればこの処理は不要
    const uint8_t rtcIntAddress = 0x00;
    uint8_t rtcIntValue = 0x00;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&rtcIntValue, *GetRtcI2cSession(), &rtcIntAddress));
    NN_DETAIL_BPC_INFO("Cleared RTC interrupt flags. It was 0x%x\n", rtcIntValue);

    // シャットダウンの場合は、後で擬似オフ状態に入れるための RTC Alarm をセット（再起動の場合は強制解除）
    // ACOK 取得 -> FGIC シャットダウンモード設定のクリティカル期間を長くしないよう、給電状態かどうかはチェックすることなく毎回セットする
    if ( !isReboot && g_SettingsHolder.IsQuasiOffEnabled() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(SetQuasiOffTimer(true));
    }
    else
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(SetQuasiOffTimer(false));
    }

    // シャットダウン直前に RTC 関連処理を行います。
    NotifyShutdownToRtc();

    // シャットダウン直前に充電状況に応じて電池残量計をシャットダウンするか決定する
    FinalizeFuelgauge();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::WriteSingleRegister(session, &onoffcnfg1Address, &onoffcnfg1Value));

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(TimeoutMilliSeconds));

    NN_ABORT("[bpc] Timeout.");
}

} // namespace

void InitializeBoardPowerController() NN_NOEXCEPT
{
    g_SettingsHolder.Load();

    // GPIO の FPS 初期化の為 GPIO ドライバを初期化する
    nne::max7762x::gpio::Initialize();
}

void FinalizeBoardPowerController() NN_NOEXCEPT
{
    nne::max7762x::gpio::Finalize();
}

NN_NORETURN void ShutdownSystem() NN_NOEXCEPT
{
    NN_DETAIL_BPC_INFO("Shutdown System.\n");

#if 1
    ShutdownSystemImpl(false);
#else // SIGLO-36382: ACOK チェック後シャットダウンまでのクリティカルセクション期間をできるだけ短くするため、max7762x の実装を使わない
    nne::max7762x::onoff::RegulatorONOFFSession session;

    // temporary implementation because onoff::OpenSession set not correct register settings
    {
        nne::max7762x::Result result = nne::max7762x::onoff::OpenSession(&session, "max77620_onoff0");
        NN_ABORT_UNLESS(result == nne::max7762x::Result_Success, "I2C transaction error");
    }

    FinalizeFuelgauge();

    nne::max7762x::Result result = nne::max7762x::onoff::Shutdown(&session);
    NN_ABORT_UNLESS(result == nne::max7762x::Result_Success, "I2C transaction error");

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(TimeoutMilliSeconds));

    NN_ABORT("[bpc] Timeout.");
#endif
}

NN_NORETURN void RebootSystem() NN_NOEXCEPT
{
    NN_DETAIL_BPC_INFO("Reboot System.\n");

#if 1
    ShutdownSystemImpl(true);
#else
    nne::max7762x::onoff::RegulatorONOFFSession session;

    // temporary implementation because onoff::OpenSession set not correct register settings
    {
        nne::max7762x::Result result = nne::max7762x::onoff::OpenSession(&session, "max77620_onoff0");
        NN_ABORT_UNLESS(result == nne::max7762x::Result_Success, "I2C transaction error");
    }

    nne::max7762x::Result result = nne::max7762x::onoff::Restart(&session);
    NN_ABORT_UNLESS(result == nne::max7762x::Result_Success, "I2C transaction error");

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(TimeoutMilliSeconds));

    NN_ABORT("[bpc] Timeout.");
#endif
}

nn::Result GetAcOk(bool* pOutAcOk) NN_NOEXCEPT
{
    auto* pSession = GetPmicI2cSession();
    uint8_t reg;
    const uint8_t OffsetOnOffStat = 0x15;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::i2c::ReadSingleRegister(&reg, *pSession, &OffsetOnOffStat));

    const uint8_t MaskAcOk = 0x02;
    *pOutAcOk = ((reg & MaskAcOk) == MaskAcOk);

    NN_RESULT_SUCCESS;
}

}}}} // namespace nn::bpc::driver::detail
