﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os.h>
#include <nn/os/os_Mutex.h>
#include <nn/bpc/bpc_Lib.h> // DEPRECATED
#include <nn/bpc/bpc_BoardPowerControl.h>
#include <nn/bpc/bpc_PowerButton.h> // To call InitializePowerButton in deprecated Initialize
#include <nn/bpc/bpc_ShimInternal.h>
#include <nn/bpc/bpc_IBoardPowerControlManager.sfdl.h>

#include "bpc_GetManagerByHipc.h"

namespace nn { namespace bpc {

namespace {

// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
nn::sf::SharedPointer<IBoardPowerControlManager> g_BoardPowerControlManager;

// Initialize の参照カウント
int g_InitializeCount = 0;

// 参照カウントを守る Mutex
struct StaticMutex
{
    nn::os::MutexType mutex;
    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&mutex);
    }
} g_InitializeCountMutex = { NN_OS_MUTEX_INITIALIZER(false) };

} // namespace

void InitializeBoardPowerControl() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    if ( g_InitializeCount == 0 )
    {
        NN_SDK_ASSERT(!g_BoardPowerControlManager);

        g_BoardPowerControlManager = GetBoardPowerControlManagerByHipc();
    }

    g_InitializeCount++;
}

// DEPRECATED
NN_DEPRECATED void Initialize() NN_NOEXCEPT
{
    InitializeBoardPowerControl();
    InitializePowerButton();
}

void InitializeBoardPowerControlWith(nn::sf::SharedPointer<IBoardPowerControlManager> boardPowerControlManager) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount == 0);

    g_BoardPowerControlManager = boardPowerControlManager;

    g_InitializeCount++;
}

nn::sf::SharedPointer<IBoardPowerControlManager> GetInternalBoardPowerControlManager() NN_NOEXCEPT
{
    return g_BoardPowerControlManager;
}

void FinalizeBoardPowerControl() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount > 0);

    g_InitializeCount--;

    if ( g_InitializeCount == 0 )
    {
        NN_SDK_ASSERT_NOT_NULL(g_BoardPowerControlManager);

        // 共有ポインタへの nullptr の代入で解放できる。
        g_BoardPowerControlManager = nullptr;
    }
}

// DEPRECATED
NN_DEPRECATED void Finalize() NN_NOEXCEPT
{
    FinalizePowerButton();
    FinalizeBoardPowerControl();
}

NN_NORETURN void ShutdownSystem() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_BoardPowerControlManager);

    auto result = g_BoardPowerControlManager->ShutdownSystem();

    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_ABORT("[bpc] Failed to shutdown.");
}

NN_NORETURN void RebootSystem() NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_BoardPowerControlManager);

    auto result = g_BoardPowerControlManager->RebootSystem();

    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_ABORT("[bpc] Failed to reboot.");
}

nn::Result GetWakeupReason(WakeupReason* pOutWakeupReason) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_BoardPowerControlManager);

    std::uint32_t reason = static_cast<std::uint32_t>(WakeupReasonNoReason);
    NN_RESULT_DO(g_BoardPowerControlManager->GetWakeupReason(&reason));

    *pOutWakeupReason = static_cast<WakeupReason>(reason);

    NN_RESULT_SUCCESS;
}

nn::Result GetShutdownReason(ShutdownReason* pOutShutdownReason) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_BoardPowerControlManager);
    NN_SDK_ASSERT(pOutShutdownReason);

    std::uint32_t reason = static_cast<std::uint32_t>(ShutdownReasonNoReason);
    NN_RESULT_DO(g_BoardPowerControlManager->GetShutdownReason(&reason));

    *pOutShutdownReason = static_cast<ShutdownReason>(reason);

    NN_RESULT_SUCCESS;
}

nn::Result GetAcOk(bool* pOutAcOk) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_BoardPowerControlManager);
    NN_SDK_ASSERT(pOutAcOk);

    bool acOk = false;
    NN_RESULT_DO(g_BoardPowerControlManager->GetAcOk(&acOk));

    *pOutAcOk = acOk;

    NN_RESULT_SUCCESS;
}

nn::Result GetBoardPowerControlEvent(nn::os::SystemEventType* pOutEvent, BoardPowerControlEventTarget target) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(g_BoardPowerControlManager);
    NN_SDK_ASSERT(pOutEvent);

    nn::sf::NativeHandle sfHandle;

    NN_RESULT_DO(g_BoardPowerControlManager->GetPowerEvent(&sfHandle, target));

    nn::os::AttachReadableHandleToSystemEvent(pOutEvent, sfHandle.GetOsHandle(), sfHandle.IsManaged(), nn::os::EventClearMode_ManualClear);

    sfHandle.Detach();

    NN_RESULT_SUCCESS;
}

}} // namepsace nn::bpc
