﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/sf/impl/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/bpc/bpc_IBoardPowerControlManager.sfdl.h>
#include <nn/bpc/bpc_IRtcManager.sfdl.h>
#include <nn/bpc/bpc_ManagerImpl.h>
#include <nn/bpc/driver/bpc.h>

namespace nn { namespace bpc {

// IBoardPowerControlManager 実装クラスの実装
BoardPowerControlManagerImpl::BoardPowerControlManagerImpl() NN_NOEXCEPT
{
}

BoardPowerControlManagerImpl::~BoardPowerControlManagerImpl() NN_NOEXCEPT
{
}

nn::Result BoardPowerControlManagerImpl::ShutdownSystem() NN_NOEXCEPT
{
    nn::bpc::driver::ShutdownSystem();

    NN_RESULT_SUCCESS;
}

nn::Result BoardPowerControlManagerImpl::RebootSystem() NN_NOEXCEPT
{
    nn::bpc::driver::RebootSystem();

    NN_RESULT_SUCCESS;
}

nn::Result BoardPowerControlManagerImpl::GetWakeupReason(nn::sf::Out<std::uint32_t> pOutWakeupReason) NN_NOEXCEPT
{
    WakeupReason reason = WakeupReasonNoReason;
    NN_RESULT_DO(nn::bpc::driver::GetWakeupReason(&reason));
    *pOutWakeupReason = reason;

    NN_RESULT_SUCCESS;
}

nn::Result BoardPowerControlManagerImpl::GetShutdownReason(nn::sf::Out<std::uint32_t> pOutShutdownReason) NN_NOEXCEPT
{
    ShutdownReason reason = ShutdownReasonNoReason;
    NN_RESULT_DO(nn::bpc::driver::GetShutdownReason(&reason));
    *pOutShutdownReason = reason;

    NN_RESULT_SUCCESS;
}

nn::Result BoardPowerControlManagerImpl::GetAcOk(nn::sf::Out<bool> pOutAcOk) NN_NOEXCEPT
{
    bool acOk = false;
    NN_RESULT_DO(nn::bpc::driver::GetAcOk(&acOk));
    *pOutAcOk = acOk;

    NN_RESULT_SUCCESS;
}

nn::Result BoardPowerControlManagerImpl::GetPowerEvent(nn::sf::Out<nn::sf::NativeHandle> pOutHandle, BoardPowerControlEventTarget target) NN_NOEXCEPT
{
    nn::os::SystemEventType* pEvent;

    NN_RESULT_DO(nn::bpc::driver::GetBoardPowerControlEventPtr(&pEvent, target));

    nn::os::NativeHandle internalHandle = nn::os::GetReadableHandleOfSystemEvent(pEvent);

    // 内部の IsManaged を false にしているので、寿命管理はされない。
    *pOutHandle = nn::sf::NativeHandle(internalHandle, false);

    NN_RESULT_SUCCESS;
}

nn::Result BoardPowerControlManagerImpl::GetSleepButtonState(nn::sf::Out<nn::bpc::SleepButtonState> pOutState) NN_NOEXCEPT
{
    SleepButtonState state = SleepButtonState_Released;
    nn::bpc::driver::GetSleepButtonState(&state);
    *pOutState = state;

    NN_RESULT_SUCCESS;
}

nn::Result BoardPowerControlManagerImpl::GetPowerButtonEvent(nn::sf::Out<nn::sf::NativeHandle> pOutHandle, EventTarget target) NN_NOEXCEPT
{
    nn::os::SystemEventType* pEvent;

    NN_RESULT_DO(nn::bpc::driver::GetPowerButtonEventPtr(&pEvent, target));

    nn::os::NativeHandle internalHandle = nn::os::GetReadableHandleOfSystemEvent(pEvent);

    // 内部の IsManaged を false にしているので、寿命管理はされない。
    *pOutHandle = nn::sf::NativeHandle(internalHandle, false);

    NN_RESULT_SUCCESS;
}

// DEPRECATED
nn::Result BoardPowerControlManagerImpl::CreateWakeupTimer(nn::sf::Out<int> outTimerHandle, nn::TimeSpanType interval) NN_NOEXCEPT
{
    int timerHandle;
    NN_RESULT_DO(nn::bpc::driver::CreateWakeupTimer(&timerHandle, interval, WakeupTimerType_BackgroundTask));
    *outTimerHandle = timerHandle;
    NN_RESULT_SUCCESS;
}

nn::Result BoardPowerControlManagerImpl::CreateWakeupTimerEx(nn::sf::Out<int> outTimerHandle, nn::TimeSpanType interval, WakeupTimerType type) NN_NOEXCEPT
{
    int timerHandle;
    NN_RESULT_DO(nn::bpc::driver::CreateWakeupTimer(&timerHandle, interval, type));
    *outTimerHandle = timerHandle;
    NN_RESULT_SUCCESS;
}

nn::Result BoardPowerControlManagerImpl::CancelWakeupTimer(int timerHandle) NN_NOEXCEPT
{
    return nn::bpc::driver::CancelWakeupTimer(timerHandle);
}

nn::Result BoardPowerControlManagerImpl::EnableWakeupTimerOnDevice(nn::sf::Out<bool> outIsEnabled, nn::sf::Out<int> outTimerHandle, nn::sf::Out<WakeupTimerType> outWakeupTimerType) NN_NOEXCEPT
{
    bool isEnabled;
    nn::TimeSpan interval;
    int handle;
    WakeupTimerType type;
    NN_RESULT_DO(nn::bpc::driver::EnableWakeupTimerOnDevice(&isEnabled, &handle, &type));
    *outIsEnabled = isEnabled;
    *outTimerHandle = handle;
    *outWakeupTimerType = type;
    NN_RESULT_SUCCESS;
}

nn::Result BoardPowerControlManagerImpl::GetLastEnabledWakeupTimerType(nn::sf::Out<WakeupTimerType> outWakeupTimerType) NN_NOEXCEPT
{
    *outWakeupTimerType = nn::bpc::driver::GetLastEnabledWakeupTimerType();
    NN_RESULT_SUCCESS;
}

nn::Result BoardPowerControlManagerImpl::CleanAllWakeupTimers(WakeupTimerType type) NN_NOEXCEPT
{
    NN_RESULT_DO(nn::bpc::driver::CleanAllWakeupTimers(type));
    NN_RESULT_SUCCESS;
}

// IRtcManager 実装クラスの実装
RtcManagerImpl::RtcManagerImpl() NN_NOEXCEPT
{
}

RtcManagerImpl::~RtcManagerImpl() NN_NOEXCEPT
{
}

nn::Result RtcManagerImpl::GetRtcTime(nn::sf::Out<int64_t> outRtcTimeInSeconds) NN_NOEXCEPT
{
    int64_t time;
    NN_RESULT_DO(nn::bpc::driver::GetRtcTime(&time));
    *outRtcTimeInSeconds = time;
    NN_RESULT_SUCCESS;
}

nn::Result RtcManagerImpl::SetRtcTime(int64_t rtcTimeInSeconds) NN_NOEXCEPT
{
    NN_RESULT_DO(nn::bpc::driver::SetRtcTime(rtcTimeInSeconds));
    NN_RESULT_SUCCESS;
}

nn::Result RtcManagerImpl::GetRtcResetDetected(nn::sf::Out<bool> outRtcResetDetected) NN_NOEXCEPT
{
    bool rtcResetDetected = false;
    NN_RESULT_DO(nn::bpc::driver::GetRtcResetDetected(&rtcResetDetected));
    *outRtcResetDetected = rtcResetDetected;
    NN_RESULT_SUCCESS;
}

nn::Result RtcManagerImpl::ClearRtcResetDetected() NN_NOEXCEPT
{
    NN_RESULT_DO(nn::bpc::driver::ClearRtcResetDetected());
    NN_RESULT_SUCCESS;
}

nn::Result RtcManagerImpl::SetUpRtcResetOnShutdown() NN_NOEXCEPT
{
    NN_RESULT_DO(nn::bpc::driver::SetUpRtcResetOnShutdown());
    NN_RESULT_SUCCESS;
}

}} // namespace nn::bpc
