﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/bluetooth/bluetooth_UserApi.h>
#include <nn/btm/user/btm_UserApi.h>
#include <nn/bluetooth/bluetooth_Result.h>

#include <nn/bluetooth/bluetooth_GattService.h>
#include <nn/bluetooth/bluetooth_GattCharacteristic.h>
#include <nn/bluetooth/bluetooth_GattDescriptor.h>

namespace nn { namespace bluetooth {

    GattDescriptor::GattDescriptor()  NN_NOEXCEPT:
        GattAttribute()
    {
        m_Type = GattAttributeType_Descriptor;
    }

    GattDescriptor::GattDescriptor(const GattAttributeUuid& uuid, uint16_t handle, uint32_t connectionHandle)  NN_NOEXCEPT:
        GattAttribute(uuid, handle, connectionHandle)
    {
        m_Type = GattAttributeType_Descriptor;
    }

    GattDescriptor::~GattDescriptor() NN_NOEXCEPT
    {
        // 何もしない
    }

    void GattDescriptor::GetService(GattService* pOutService) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutService);

        nn::btm::user::GattService btmService;

        if (nn::btm::user::GetBelongingGattService(&btmService, m_ConnectionHandle, m_Handle))
        {
            *pOutService = GattService(btmService.uuid, btmService.handle, m_ConnectionHandle, btmService.instanceId, btmService.endGroupHandle, btmService.isPrimaryService);

            return;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultAttributeNotFound());
    }

    void GattDescriptor::GetCharacteristic(GattCharacteristic* pOutCharacteristic) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutCharacteristic);

        GattService service;

        GetService(&service);

        nn::btm::user::GattCharacteristic btmCharacteristic[GattAttributeCountMaxClient];
        uint8_t characteristicNum = nn::btm::user::GetGattCharacteristics(btmCharacteristic, NN_ARRAY_SIZE(btmCharacteristic), m_ConnectionHandle, service.GetHandle());

        for (uint8_t i = 0; i < characteristicNum; ++i)
        {
            if (i < characteristicNum - 1)
            {
                if (btmCharacteristic[i].handle < m_Handle && m_Handle < btmCharacteristic[i + 1].handle)
                {
                    *pOutCharacteristic = GattCharacteristic(btmCharacteristic[i].uuid, btmCharacteristic[i].handle, m_ConnectionHandle, btmCharacteristic[i].instanceId, btmCharacteristic[i].property);

                    return;
                }
            }
            else
            {
                if (btmCharacteristic[i].handle < m_Handle)
                {
                    *pOutCharacteristic = GattCharacteristic(btmCharacteristic[i].uuid, btmCharacteristic[i].handle, m_ConnectionHandle, btmCharacteristic[i].instanceId, btmCharacteristic[i].property);

                    return;
                }
            }
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultAttributeNotFound());
    }

    void GattDescriptor::SetValue(const void* pValue, size_t size) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pValue);
        NN_ABORT_UNLESS_LESS_EQUAL(size, GattAttributeValueSizeMax);

        m_ValueLength = size;
        memcpy(m_Value, pValue, m_ValueLength);
    }

    size_t GattDescriptor::GetValue(void* pOutValue, size_t size) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutValue);

        size_t getSize = (m_ValueLength < size) ? m_ValueLength : size;

        memcpy(pOutValue, m_Value, getSize);

        return getSize;
    }
}}  // namespace nn::bluetooth
