﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/dd.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/bluetooth/bluetooth_Result.h>
#include <nn/bluetooth/bluetooth_ResultPrivate.h>

#include "bluetooth_DriverImpl.h"
#include "bluetooth_Settings.h"

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
#include <nn/settings/factory/settings_Bluetooth.h>
#include <nn/settings/factory/settings_WirelessLan.h>
#include <nn/settings/factory/settings_ConfigurationId.h>
#endif



namespace nn { namespace bluetooth {

nn::os::EventType  g_InternalEvent;
nn::os::EventType  g_InternalEventForHid;
nn::os::EventType  g_InternalEventForHidReport;
nn::os::EventType  g_InternalEventForBle;
nn::os::EventType  g_InternalEventForBleCore;
nn::os::EventType  g_InternalEventForBleHid;

bool               g_PendingIntEvent             = false;
bool               g_PendingIntEventForHid       = false;
bool               g_PendingIntEventForHidReport = false;
bool               g_PendingIntEventForBle       = false;
bool               g_PendingIntEventForBleCore   = false;
bool               g_PendingIntEventForBleHid    = false;

//Todo: To add a mutex for using these parameters.
// Basically a client behave a correct rule, collision will not occur.
// If a client uses the GetEventInfo regardless a event signal, collision will occur.
// Before this implementation, we should define the spec of multi-client.
// When we implement multi-clients, we have to define how to create/destroy these parameters.
namespace{
    //for Bluetooth
    nn::bluetooth::BluetoothInterfaceC g_BluetoothInterface;
    nn::os::SystemEventType            g_SystemEvent;
    uint8_t                            g_Buffer[BUFFER_SIZE_OF_CORE_OUT];
    uint32_t                           g_EventType;

    // Initialize when Skip boot is enabled instead of g_SystemEvent
    nn::os::SystemEventType             g_DummySystemEvent;

    //for Hid
    const BluetoothHhInterface*        g_pBluetoothHidInterface;
    nn::os::SystemEventType            g_SystemEventForHid;
    uint8_t                            g_BufferForHid[BUFFER_SIZE_OF_HID_OUT];
    uint32_t                           g_EventTypeForHid;

    // Initialize when Skip boot is enabled instead of g_SystemEventForHid
    nn::os::SystemEventType             g_DummySystemEventForHid;

    nn::os::SystemEventType            g_SystemEventForHidReport;
    uint8_t                            g_BufferForHidReport[BUFFER_SIZE_OF_HID_OUT];
    uint32_t                           g_EventTypeForHidReport;

    // Initialize when Skip boot is enabled instead of g_SystemEventForHidReport
    nn::os::SystemEventType             g_DummySystemEventForHidReport;

    // for Extension (other parameters are shared with Hid.)
    const BluetoothExtension*          g_pBluetoothExtensionInterface;

    // for bluetooth low energy
    const BluetoothLeInterface*        g_pBluetoothLowEnergyInterface;
    nn::os::SystemEventType            g_SystemEventForBleCore;
    uint8_t                            g_BufferForBleCore[BUFFER_SIZE_OF_BLE_OUT];
    uint32_t                           g_EventTypeForBleCore;

    // Initialize when Skip boot is enabled instead of g_SystemEventForBleCore
    nn::os::SystemEventType             g_DummySystemEventForBleCore;

    nn::os::SystemEventType            g_SystemEventForBle;
    uint8_t                            g_BufferForBle[BUFFER_SIZE_OF_BLE_OUT];
    uint32_t                           g_EventTypeForBle;

    // Initialize when Skip boot is enabled instead of g_SystemEventForBle
    nn::os::SystemEventType             g_DummySystemEventForBle;

    nn::os::SystemEventType            g_SystemEventForBleHid;
    uint8_t                            g_BufferForBleHid[BUFFER_SIZE_OF_BLE_OUT];
    uint32_t                           g_EventTypeForBleHid;

    // Initialize when Skip boot is enabled instead of g_SystemEventForBleHid
    nn::os::SystemEventType             g_DummySystemEventForBleHid;

    //int g_EventCount=0;
    //int tick;

    bool g_isFirstInitialize = true;
}

nn::Result BluetoothDriverImpl::WaitServerReady() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}


//-----------------------------------------------------------------------------
// Utility functions
//-----------------------------------------------------------------------------
void PrintMacAddress(const BluetoothAddress* pBluetoothAddress)
{
    const int sizeOfMacAddress = 6;
    for(int i=0;i<sizeOfMacAddress;i++)
    {
        NN_SDK_LOG("%02X",pBluetoothAddress->address[i]);
        if(i<sizeOfMacAddress - 1)
        {
            NN_SDK_LOG(":");
        }
        else
        {
            NN_SDK_LOG("\n");
        }
    }
}

void PrintByteDataAsHex(uint16_t length, const uint8_t buffer[])
{
#if 0
    for(uint16_t i=0;i<length;i++)
    {
        NN_SDK_LOG("%02X:",buffer[i]);
    }
    NN_SDK_LOG("\n");
#endif

    uint8_t byteCount=0;
    for(uint16_t i=0;i<length;i++)
    {
        NN_SDK_LOG("%02X ",buffer[i]);
        byteCount++;
        if(byteCount==16)
        {
            NN_SDK_LOG("\n");
            byteCount=0;
        }
    }
    NN_SDK_LOG("\n");
}


//typeOfResult =0 : Core
//typeOfResult =1 : Hid
nn::Result ConvertResult(uint8_t resultFromHal, uint8_t typeOfResult)
{
    nn::Result result;
    if(typeOfResult==0)
    {
        switch(resultFromHal)
        {
        case BT_STATUS_SUCCESS:
        result=ResultSuccess();
        break;

        case BT_STATUS_FAIL:
        result=ResultFail();
        break;

        case BT_STATUS_NOT_READY:
        result=ResultNotReady();
        break;

        case BT_STATUS_NOMEM:
        result=ResultNoMemory();
        break;

        case BT_STATUS_BUSY:
        result=ResultBusy();
        break;

        case BT_STATUS_DONE:
        result=ResultDone();
        break;

        case BT_STATUS_UNSUPPORTED:
        result=ResultUnsupported();
        break;

        case BT_STATUS_PARM_INVALID:
        result=ResultParameterInvalid();
        break;

        case BT_STATUS_UNHANDLED:
        result=ResultUnhandled();
        break;

        case BT_STATUS_AUTH_FAILURE:
        result=ResultAuthFailure();
        break;

        case BT_STATUS_RMT_DEV_DOWN:
        result=ResultRemoteDeviceDown();
        break;

        case BT_STATUS_AUTH_REJECTED:
        result=ResultAuthRejected();
        break;

        default:
        NN_SDK_REQUIRES(typeOfResult <= BT_STATUS_AUTH_REJECTED);
        break;
        }
    }
    else if(typeOfResult==1)
    {
        switch(resultFromHal)
        {
        case BTHH_OK:
        result=ResultSuccess();
        break;

        case BTHH_HS_HID_NOT_READY:
        result=ResultHandshakeNotReady();
        break;

        case BTHH_HS_INVALID_RPT_ID:
        result=ResultHandshakeInvalidReportId();
        break;

        case BTHH_HS_TRANS_NOT_SPT:
        result=ResultHandshakeTransactionNotSupported();
        break;

        case BTHH_HS_INVALID_PARAM:
        result=ResultHandshakeInvalidParameter();
        break;

        case BTHH_HS_ERROR:
        result=ResultHandshakeError();
        break;

        case BTHH_HH_ERR:
        result=ResultGeneralHhError();
        break;

        case BTHH_ERR_SDP:
        result=ResultSdpError();
        break;

        case BTHH_ERR_PROTO:
        result=ResultProtocolError();
        break;

        case BTHH_ERR_DB_FULL:
        result=ResultDatabaseFulled();
        break;

        case BTHH_ERR_TOD_UNSPT:
        result=ResultTypeOfDeviceNotSupported();
        break;

        case BTHH_ERR_NO_RES:
        result=ResultNoResources();
        break;

        case BTHH_ERR_AUTH_FAILED:
        result=ResultAuthFailed();
        break;

        case BTHH_ERR_HDL:
        result=ResultHdlError();
        break;

        case BTHH_ERR:
        result=ResultError();
        break;

        case BTHH_ERR_BAD_PARAM:
        result=ResultBadParameter();
        break;

        case BTHH_ERR_BAD_RESPONSE:
        result=ResultBadResponse();
        break;

        case BTHH_ERR_BUSY:
        result=ResultBusyError();
        break;

        case BTHH_ERR_CONNECTION:
        result=ResultConnectionError();
        break;

        case BTHH_ERR_OPEN:
        result=ResultOpenError();
        break;

        case BTHH_ERR_CLIENT:
        result=ResultClientError();
        break;

        case BTHH_ERR_SERVER:
        result=ResultServerError();
        break;

        case BTHH_ERR_SECURITY:
        result=ResultSecurityError();
        break;

        case BTHH_ERR_FILE_SYS:
        result=ResultFsError();
        break;

        case BTHH_ERR_AUDIO_VIDEO:
        result=ResultAvError();
        break;

        case BTHH_ERR_DISCOVERY:
        result=ResultDiscoveryError();
        break;

        case BTHH_ERR_TEST_MODE:
        result=ResultTestModeError();
        break;

        case BTHH_ERR_DM:
        result=ResultDmError();
        break;

        case BTHH_ERR_ROBSON:
        result=ResultRobsonError();
        break;

        default:
        NN_SDK_REQUIRES(typeOfResult <= BTHH_ERR_HDL);
        break;
        }
    }
    else
    {
        NN_SDK_REQUIRES(typeOfResult < 1);
    }
    return result;
}; //NOLINT(impl/function_size)

nn::Result ConvertLeResult(BluetoothLeStatus resultFromHal)
{
    nn::Result result;

    switch(resultFromHal)
    {
    case BT_OK:
    result=ResultSuccess();
    break;

    case BT_BLE_ERR:
    result=ResultBleError();
    break;

    case BT_BLE_ERR_NOT_ENABLED:
    result=ResultBleNotEnabled();
    break;

    case BT_BLE_ERR_NO_RESOURCE:
    result=ResultBleNoResource();
    break;

    case BT_BLE_ERR_INVALID_PARAM:
    result=ResultBleInvalidParam();
    break;

    case BT_GATT_ERR_INVALID_HANDLE:
    result=ResultBleInvalidHandle();
    break;

    case BT_GATT_ERR_READ_PERMISSION:
    result=ResultBleReadPermission();
    break;

    case BT_GATT_ERR_WRITE_PERMISSION:
    result=ResultBleWritePermission();
    break;

    case BT_GATT_ERR_INVALID_PDU:
    result=ResultBleInvalidPdu();
    break;

    case BT_GATT_ERR_INVALID_AUTH:
    result=ResultBleInvalidAuth();
    break;

    case BT_GATT_ERR_REQ_NOT_SUPP:
    result=ResultBleReqNotSupported();
    break;

    case BT_GATT_ERR_INVALID_OFFSET:
    result=ResultBleInvalidOffset();
    break;

    case BT_GATT_ERR_QUEUE_FULL:
    result=ResultBleQueueFull();
    break;

    case BT_GATT_ERR_NOT_FOUND:
    result=ResultBleNotFound();
    break;

    case BT_GATT_ERR_NOT_LONG:
    result=ResultBleNotLong();
    break;

    case BT_GATT_ERR_KEY_SIZE:
    result=ResultBleKeySize();
    break;

    case BT_GATT_ERR_INVALID_ATTR_LEN:
    result=ResultBleInvalidAttributeLen();
    break;

    case BT_GATT_ERR_UNLIKELY:
    result=ResultBleUnlikely();
    break;

    case BT_GATT_ERR_INVALID_ENCRYPTION:
    result=ResultBleInvalidEncryption();
    break;

    case BT_GATT_ERR_INVALID_GROUP_TYPE:
    result=ResultBleInvalidGroupType();
    break;

    case BT_GATT_ERR_NO_RESOURCES:
    result=ResultBleGattNoResources();
    break;

    case BT_GATT_ERR_INVALID_PARAM:
    result=ResultBleGattInvalidParam();
    break;

    case BT_GATT_ERR_INTERNAL:
    result=ResultBleInternal();
    break;

    case BT_GATT_ERR_WRONG_STATE:
    result=ResultBleWrongState();
    break;

    case BT_GATT_ERR_DB_FULL:
    result=ResultBleDatabaseFull();
    break;

    case BT_GATT_ERR_BUSY:
    result=ResultBleGattBusy();
    break;

    case BT_GATT_ERR:
    result=ResultBleGattError();
    break;

    case BT_GATT_ERR_STARTED:
    result=ResultBleStarted();
    break;

    case BT_GATT_ERR_PENDING:
    result=ResultBlePending();
    break;

    case BT_GATT_ERR_MORE:
    result=ResultBleMore();
    break;

    case BT_GATT_ERR_INVALID_CONFIG:
    result=ResultBleInvalidConfig();
    break;

    case BT_GATT_ERR_ENCRYPTED_MITM:
    result=ResultBleEncryptedMitm();
    break;

    case BT_GATT_ERR_ENCRYPTED_NO_MITM:
    result=ResultBleEncryptedNoMitm();
    break;

    case BT_GATT_ERR_NOT_ENCRYPTED:
    result=ResultBleNotEncrypted();
    break;

    case BT_GATT_ERR_CONGESTED:
    result=ResultBleCongested();
    break;

    case BT_HS_HID_NOT_READY:
    result=ResultHandshakeNotReady();
    break;

    case BT_HS_INVALID_RPT_ID:
    result=ResultHandshakeInvalidReportId();
    break;

    case BT_HS_TRANS_NOT_SPT:
    result=ResultHandshakeTransactionNotSupported();
    break;

    case BT_HS_INVALID_PARAM:
    result=ResultHandshakeInvalidParameter();
    break;

    case BT_HS_ERROR:
    result=ResultHandshakeError();
    break;

    case BT_HH_ERR:
    result=ResultGeneralHhError();
    break;

    case BT_ERR_SDP:
    result= ResultBleServiceDiscoveryError();
    break;

    case BT_ERR_PROTO:
    result=ResultProtocolError();
    break;

    case BT_ERR_DB_FULL:
    result= ResultBleDatabaseFullError();
    break;

    case BT_ERR_TOD_UNSPT:
    result=ResultTypeOfDeviceNotSupported();
    break;

    case BT_ERR_NO_RES:
    result=ResultNoResources();
    break;

    case BT_ERR_AUTH_FAILED:
    result=ResultAuthFailed();
    break;

    case BT_ERR_HDL:
    result=ResultHdlError();
    break;

    case BT_ERR:
    result=ResultError();
    break;

    case BT_ERR_BAD_PARAM:
    result=ResultBadParameter();
    break;

    case BT_ERR_BAD_RESPONSE:
    result=ResultBadResponse();
    break;

    case BT_ERR_BUSY:
    result= ResultBleBusyError();
    break;

    case BT_ERR_CONNECTION:
    result= ResultBleConnectionError();
    break;

    case BT_ERR_OPEN:
    result= ResultBleOpenError();
    break;

    case BT_ERR_CLIENT:
    result= ResultBleClientError();
    break;

    case BT_ERR_SERVER:
    result= ResultBleServerError();
    break;

    case BT_ERR_SECURITY:
    result=ResultSecurityError();
    break;

    case BT_ERR_FILE_SYS:
    result=ResultFsError();
    break;

    case BT_ERR_AUDIO_VIDEO:
    result=ResultAvError();
    break;

    case BT_ERR_DISCOVERY:
    result= ResultBleScanError();
    break;

    case BT_ERR_TEST_MODE:
    result=ResultTestModeError();
    break;

    case BT_ERR_DM:
    result=ResultDmError();
    break;

    case BT_ERR_ROBSON:
    result=ResultRobsonError();
    break;

    default:
    result=ResultError();
    break;
    }

    return result;
}; //NOLINT(impl/function_size)

static void SignalEvent(uint32_t event)
{
    g_EventType = event;
    nn::os::SignalSystemEvent(&g_SystemEvent);
    g_PendingIntEvent = true;
}

static void SignalEventForHid(uint32_t event)
{
    g_EventTypeForHid = event;
    nn::os::SignalSystemEvent(&g_SystemEventForHid);
    g_PendingIntEventForHid = true;
}

static void SignalEventForHidReport(uint32_t event)
{
    g_EventTypeForHidReport = event;
    nn::os::SignalSystemEvent(&g_SystemEventForHidReport);
    g_PendingIntEventForHidReport = true;
}

static void SignalEventForBle(uint32_t event)
{
    g_EventTypeForBle = event;
    nn::os::SignalSystemEvent(&g_SystemEventForBle);
    g_PendingIntEventForBle = true;
}

static void SignalEventForBleCore(uint32_t event)
{
    g_EventTypeForBleCore = event;
    nn::os::SignalSystemEvent(&g_SystemEventForBleCore);
    g_PendingIntEventForBleCore = true;
}

static void SignalEventForBleHid(uint32_t event)
{
    g_EventTypeForBleHid = event;
    nn::os::SignalSystemEvent(&g_SystemEventForBleHid);
    g_PendingIntEventForBleHid = true;
}

//-----------------------------------------------------------------------------
// Callbacks from the HAL
//-----------------------------------------------------------------------------
//[Todo]have to care a size of enum, size_t.
void AdapterStateChangedCallback(BluetoothState state) NN_NOEXCEPT
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s state=%d\n",__FUNCTION__,state);
    NN_SDK_REQUIRES(sizeof(InfoFromAdapterStateChangedCallback) <= BUFFER_SIZE_OF_CORE_OUT);
    reinterpret_cast<InfoFromAdapterStateChangedCallback*>(&g_Buffer[0])->state=state;

    SignalEvent(EventFromAdapterStateChangedCallback);
}

void AdapterPropertiesCallback(BluetoothIfStatus status,
                                       int NumProperties,
                                       BluetoothProperty *properties) NN_NOEXCEPT
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s status=%d\n",__FUNCTION__,status);
    NN_SDK_REQUIRES(sizeof(InfoFromAdapterPropertiesCallback)  <= BUFFER_SIZE_OF_CORE_OUT);
    reinterpret_cast<InfoFromAdapterPropertiesCallback*>(&g_Buffer[0])->status=status;

    if(properties->type==BT_PROPERTY_BDADDR)
    {
        reinterpret_cast<InfoFromAdapterPropertiesCallback*>(&g_Buffer[0])->bluetoothAddress=*static_cast<BluetoothAddress*>(properties->pVal);
    }

    SignalEvent(EventFromAdapterPropertiesCallback);
}

void RemoteDevicePropertiesCallback(BluetoothIfStatus status,
                                           Btbdaddr *pBdAddr,
                                           int NumProperties,
                                           BluetoothProperty *properties) NN_NOEXCEPT
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s status=%d\n",__FUNCTION__,status);
    NN_SDK_REQUIRES(sizeof(InfoFromRemoteDevicePropertiesCallback)  <= BUFFER_SIZE_OF_CORE_OUT);
    reinterpret_cast<InfoFromRemoteDevicePropertiesCallback*>(&g_Buffer[0])->status=status;
    memcpy(reinterpret_cast<InfoFromRemoteDevicePropertiesCallback*>(&g_Buffer[0])->bluetoothAddress.address, pBdAddr->address, BD_ADDR_LEN);

    SignalEvent(EventFromRemoteDevicePropertiesCallback);
}

void DeviceFoundCallback(int NumProperties,
                                BluetoothProperty *properties) NN_NOEXCEPT
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s Device found NumProperties=%d\n",__FUNCTION__,NumProperties);

#if 0
    Btbdaddr bd_addr;
    memcpy(bd_addr.address,properties->pVal,6);

    NN_SDK_LOG("\tBdaddr:%02x:%02x:%02x:%02x:%02x:%02x\n",
                                bd_addr.address[0],
                                bd_addr.address[1],
                                bd_addr.address[2],
                                bd_addr.address[3],
                                bd_addr.address[4],
                                bd_addr.address[5]);
#endif

    NN_SDK_REQUIRES(sizeof(InfoFromDeviceFoundCallback) <= BUFFER_SIZE_OF_CORE_OUT);
    /*if(properties->type==BT_PROPERTY_BDADDR)
    {
        reinterpret_cast<InfoFromDeviceFoundCallback*>(&g_Buffer[0])->bluetoothAddress= *static_cast<BluetoothAddress*>(properties->pVal);
    }*/
    if (properties->type == BT_PROPERTY_ALL)
    {
        //PrintByteDataAsHex(properties->len,reinterpret_cast<uint8_t*>(properties->pVal));
        *reinterpret_cast<InfoFromDeviceFoundCallback*>(&g_Buffer[0]) = *static_cast<InfoFromDeviceFoundCallback*>(properties->pVal);
    }
    else
    {
        NN_SDK_ASSERT(false,"undefined type of DeviceFoundCallback");
    }

    SignalEvent(EventFromDeviceFoundCallback);
}

void DiscoveryStateChangedCallback(BluetoothDiscoveryState state) NN_NOEXCEPT
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s Device discovered...state=%d\n",__FUNCTION__,state);
    NN_SDK_REQUIRES(sizeof(InfoFromDiscoveryStateChangedCallback) <= BUFFER_SIZE_OF_CORE_OUT);
    reinterpret_cast<InfoFromDiscoveryStateChangedCallback*>(&g_Buffer[0])->state=state;

    SignalEvent(EventFromDiscoveryStateChangedCallback);
}

void PinRequestCallback(Btbdaddr *pBdAddr,
                               Btbdname *pBdName, ClassOfDevice *pCod) NN_NOEXCEPT
{
    // NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s Device discovered...cod=%02X%02X%02X\n",__FUNCTION__,pCod->cod[0],pCod->cod[1],pCod->cod[2]);
    NN_SDK_REQUIRES(sizeof(InfoFromPinRequestCallback) <= BUFFER_SIZE_OF_CORE_OUT);
    InfoFromPinRequestCallback *pPin = reinterpret_cast<InfoFromPinRequestCallback*>(&g_Buffer[0]);
    memcpy(pPin->bluetoothAddress.address, pBdAddr->address, BD_ADDR_LEN);
    pPin->bluetoothName    = *pBdName;
    pPin->classOfDevice    = *pCod;

    SignalEvent(EventFromPinRequestCallback);
}

void SspRequestCallback( Btbdaddr *pBdAddr,
                                Btbdname *pBdName,
                                ClassOfDevice *pCod,
                                BluetoothSspVariant PairingVariant,
                                uint32_t pass_key ) NN_NOEXCEPT
{
    // NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s cod=%02X%02X%02X\n",__FUNCTION__,pCod->cod[0],pCod->cod[1],pCod->cod[2]);
    NN_SDK_REQUIRES(sizeof(InfoFromSspRequestCallback) <= BUFFER_SIZE_OF_CORE_OUT);

    InfoFromSspRequestCallback *pSsp = reinterpret_cast<InfoFromSspRequestCallback*>(&g_Buffer[0]);
    memcpy(pSsp->bluetoothAddress.address, pBdAddr->address, BD_ADDR_LEN);
    pSsp->bluetoothName    = *pBdName;
    pSsp->classOfDevice    = *pCod;
    pSsp->variant          = PairingVariant;
    pSsp->passkey          = pass_key;

    SignalEvent(EventFromSspRequestCallback);
}

void BondStateChangedCallback(BluetoothIfStatus status,
                                     Btbdaddr *pBdAddr,
                                     BluetoothBondState state) NN_NOEXCEPT
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s state=%d\n",__FUNCTION__,state);
    NN_SDK_REQUIRES(sizeof(InfoFromBondStateChangedCallback) <= BUFFER_SIZE_OF_CORE_OUT);
    InfoFromBondStateChangedCallback *pBS = reinterpret_cast<InfoFromBondStateChangedCallback*>(&g_Buffer[0]);
    pBS->status           = status;
    memcpy(pBS->bluetoothAddress.address, pBdAddr->address, BD_ADDR_LEN);
    pBS->state            = state;

    SignalEvent(EventFromBondStateChangedCallback);
}

void AclStateChangedCallback(BluetoothIfStatus status, Btbdaddr *pBdAddr,
                                    BluetoothAclState state) NN_NOEXCEPT
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s state=%d\n",__FUNCTION__,state);
    NN_SDK_REQUIRES(sizeof(InfoFromAclStateChangedCallback) <= BUFFER_SIZE_OF_CORE_OUT);
    InfoFromAclStateChangedCallback *pAcl = reinterpret_cast<InfoFromAclStateChangedCallback*>(&g_Buffer[0]);
    pAcl->status           = status;
    memcpy(pAcl->bluetoothAddress.address, pBdAddr->address, BD_ADDR_LEN);
    pAcl->state            = state;

    SignalEvent(EventFromAclStateChangedCallback);
}

void CallbackThreadEventCallback(BluetoothCallbackThreadEvent evt) NN_NOEXCEPT
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s evt=%d\n",__FUNCTION__,evt);
    NN_SDK_REQUIRES(sizeof(InfoFromCallbackThreadEventCallback) <= BUFFER_SIZE_OF_CORE_OUT);
    reinterpret_cast<InfoFromCallbackThreadEventCallback*>(&g_Buffer[0])->evt=evt;

    SignalEvent(EventFromCallbackThreadEventCallback);
}

void DutModeReceivedCallback(uint16_t opcode, uint8_t *buffer, uint8_t len) NN_NOEXCEPT
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s opcode=%d\n",__FUNCTION__,opcode);
    NN_SDK_REQUIRES(sizeof(InfoFromDutModeReceivedCallback) + len <= BUFFER_SIZE_OF_CORE_OUT);
    InfoFromDutModeReceivedCallback *pDut = reinterpret_cast<InfoFromDutModeReceivedCallback*>(&g_Buffer[0]);
    pDut->opcode = opcode;
    pDut->length = len;
    memcpy(&pDut->buffer, buffer, len); // NOLINT

    SignalEvent(EventFromDutModeReceivedCallback);
}

void LEtestModeCallback(BluetoothIfStatus status, uint16_t NumPackets) NN_NOEXCEPT
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s status=%d\n",__FUNCTION__,status);
    NN_SDK_REQUIRES(sizeof(InfoFromLEtestModeCallback) <= BUFFER_SIZE_OF_CORE_OUT);
    InfoFromLEtestModeCallback *pLEt = reinterpret_cast<InfoFromLEtestModeCallback*>(&g_Buffer[0]);
    pLEt->status     = status;
    pLEt->numPackets = NumPackets;

    SignalEvent(EventFromLEtestModeCallback);
}

void EnergyInfoCallback(BluetoothActivityEnergyInfo *pEnergyInfo) NN_NOEXCEPT
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromEnergyInfoCallback) <= BUFFER_SIZE_OF_CORE_OUT);
    reinterpret_cast<InfoFromEnergyInfoCallback*>(&g_Buffer[0])->energyInfo=*pEnergyInfo;

    SignalEvent(EventFromEnergyInfoCallback);
}

void FatalErrorCallback() NN_NOEXCEPT
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);

    // Masking the event until BTM is ready to use it
    //SignalEvent(EventFromFatalErrorCallback);
}

void LeClientStateChangedCallback(InfoFromLeAppStateChangedCallback *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeAppStateChangedCallback) <= BUFFER_SIZE_OF_BLE_OUT);

    InfoFromLeAppStateChangedCallback *pLE = reinterpret_cast<InfoFromLeAppStateChangedCallback*>(&g_BufferForBleCore[0]);
    pLE->clientIf       = cbData->clientIf;
    pLE->status         = cbData->status;
    pLE->isRegistered   = cbData->isRegistered;

    SignalEventForBleCore(EventFromLeClientStateChangedCallback);
}

void LeServerStateChangedCallback(InfoFromLeAppStateChangedCallback *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeAppStateChangedCallback) <= BUFFER_SIZE_OF_BLE_OUT);

    InfoFromLeAppStateChangedCallback *pLE = reinterpret_cast<InfoFromLeAppStateChangedCallback*>(&g_BufferForBleCore[0]);
    pLE->serverIf       = cbData->serverIf;
    pLE->status         = cbData->status;
    pLE->isRegistered   = cbData->isRegistered;

    SignalEventForBleCore(EventFromLeServerStateChangedCallback);
}

void LeServerProfileChangedCallback(InfoFromLeServerProfileChangedCallback *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeServerProfileChangedCallback) <= BUFFER_SIZE_OF_BLE_OUT);

    InfoFromLeServerProfileChangedCallback *pLE = reinterpret_cast<InfoFromLeServerProfileChangedCallback*>(&g_BufferForBleCore[0]);
    pLE->status             = cbData->status;
    pLE->serverIf           = cbData->serverIf;
    pLE->uuid               = cbData->uuid;
    pLE->type               = cbData->type;
    pLE->serviceHandle      = cbData->serviceHandle;
    pLE->attributeHandle    = cbData->attributeHandle;
    pLE->isPrimaryService   = cbData->isPrimaryService;
    pLE->property           = cbData->property;

    SignalEventForBleCore(EventFromLeServerProfileChangedCallback);
}

void LeServerGattReqCallback(InfoFromLeServerGattReqCallback *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeServerGattReqCallback) <= BUFFER_SIZE_OF_BLE_OUT);
    NN_SDK_REQUIRES(cbData->length <= BUFFER_SIZE_OF_BLE_GATT_ATTRIBUTE);

    g_EventType = EventFromLeServerGattReqCallback;

    InfoFromLeServerGattReqCallback *pLE = reinterpret_cast<InfoFromLeServerGattReqCallback*>(&g_BufferForBle[0]);
    pLE->status             = cbData->status;
    pLE->responseStatus     = cbData->responseStatus;
    pLE->connId             = cbData->connId;
    pLE->requestType        = cbData->requestType;
    pLE->serviceHandle      = cbData->serviceHandle;
    pLE->attributeHandle    = cbData->attributeHandle;
    pLE->attrType           = cbData->attrType;
    memcpy(pLE->data, cbData->data, sizeof(pLE->data));
    pLE->length             = cbData->length;
    pLE->offset             = cbData->offset;

    SignalEventForBle(EventFromLeServerGattReqCallback);
}

void LeServerCoreGattReqCallback(InfoFromLeServerGattReqCallback *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeServerGattReqCallback) <= BUFFER_SIZE_OF_BLE_OUT);
    NN_SDK_REQUIRES(cbData->length <= BUFFER_SIZE_OF_BLE_GATT_ATTRIBUTE);

    g_EventType = EventFromLeServerGattReqCallback;

    InfoFromLeServerGattReqCallback *pLE = reinterpret_cast<InfoFromLeServerGattReqCallback*>(&g_BufferForBleCore[0]);
    pLE->status = cbData->status;
    pLE->responseStatus = cbData->responseStatus;
    pLE->connId = cbData->connId;
    pLE->requestType = cbData->requestType;
    pLE->serviceHandle = cbData->serviceHandle;
    pLE->attributeHandle = cbData->attributeHandle;
    pLE->attrType = cbData->attrType;
    memcpy(pLE->data, cbData->data, sizeof(pLE->data));
    pLE->length = cbData->length;
    pLE->offset = cbData->offset;

    SignalEventForBleCore(EventFromLeServerGattReqCallback);
}

void LeServerHidGattReqCallback(InfoFromLeServerGattReqCallback *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeServerGattReqCallback) <= BUFFER_SIZE_OF_BLE_OUT);
    NN_SDK_REQUIRES(cbData->length <= BUFFER_SIZE_OF_BLE_GATT_ATTRIBUTE);

    g_EventType = EventFromLeServerGattReqCallback;

    InfoFromLeServerGattReqCallback *pLE = reinterpret_cast<InfoFromLeServerGattReqCallback*>(&g_BufferForBleHid[0]);
    pLE->status = cbData->status;
    pLE->responseStatus = cbData->responseStatus;
    pLE->connId = cbData->connId;
    pLE->requestType = cbData->requestType;
    pLE->serviceHandle = cbData->serviceHandle;
    pLE->attributeHandle = cbData->attributeHandle;
    pLE->attrType = cbData->attrType;
    memcpy(pLE->data, cbData->data, sizeof(pLE->data));
    pLE->length = cbData->length;
    pLE->offset = cbData->offset;

    SignalEventForBleHid(EventFromLeServerGattReqCallback);
}

void LeScanStateChangedCallback(InfoFromLeScanStateChangedCallback *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeScanStateChangedCallback) <= BUFFER_SIZE_OF_BLE_OUT);
    NN_SDK_REQUIRES(cbData->adStructureNum <= BleAdStructureCountMax);

    InfoFromLeScanStateChangedCallback *pLE = reinterpret_cast<InfoFromLeScanStateChangedCallback*>(&g_BufferForBleCore[0]);
    pLE->status         = cbData->status;
    pLE->state          = cbData->state;
    pLE->pduType        = cbData->pduType;
    pLE->addressType    = cbData->addressType;
    pLE->address        = cbData->address;
    pLE->adStructureNum = cbData->adStructureNum;
    for (int i = 0; i < cbData->adStructureNum; ++i)
    {
        pLE->adStructures[i] = cbData->adStructures[i];
    }
    pLE->rssi           = cbData->rssi;

    SignalEventForBleCore(EventFromLeScanStateChangedCallback);
}

void LeScanFilterStateChangedCallback(InfoFromLeScanFilterStateChangedCallback *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeScanFilterStateChangedCallback) <= BUFFER_SIZE_OF_BLE_OUT);

    InfoFromLeScanFilterStateChangedCallback *pLE = reinterpret_cast<InfoFromLeScanFilterStateChangedCallback*>(&g_BufferForBleCore[0]);
    pLE->status     = cbData->status;
    pLE->operation  = cbData->operation;

    SignalEventForBleCore(EventFromLeScanFilterStateChangedCallback);
}

void LeConnParamUpdateCallback(InfoFromLeConnParamUpdateCallback *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeConnParamUpdateCallback) <= BUFFER_SIZE_OF_BLE_OUT);

    InfoFromLeConnParamUpdateCallback *pLE = reinterpret_cast<InfoFromLeConnParamUpdateCallback*>(&g_BufferForBleCore[0]);
    pLE->status             = cbData->status;
    pLE->connId             = cbData->connId;
    pLE->interval           = cbData->interval;
    pLE->slaveLatency       = cbData->slaveLatency;
    pLE->supervisionTimeout = cbData->supervisionTimeout;

    SignalEventForBleCore(EventFromLeConnParamUpdateCallback);
}

void LeConnParamUpdateReqCallback(InfoFromLeConnParamUpdateReqCallback* cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeConnParamUpdateReqCallback) <= BUFFER_SIZE_OF_BLE_OUT);

    InfoFromLeConnParamUpdateReqCallback *pLE = reinterpret_cast<InfoFromLeConnParamUpdateReqCallback*>(&g_BufferForBleCore[0]);
    pLE->connId             = cbData->connId;
    pLE->intervalMin        = cbData->intervalMin;
    pLE->intervalMax        = cbData->intervalMax;
    pLE->slaveLatency       = cbData->slaveLatency;
    pLE->supervisionTimeout = cbData->supervisionTimeout;

    SignalEventForBleCore(EventFromLeConnParamUpdateReqCallback);
}

void LeClientConnStateChangedCallback(InfoFromLeConnStateChangedCallback *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeConnStateChangedCallback) <= BUFFER_SIZE_OF_BLE_OUT);

    InfoFromLeConnStateChangedCallback *pLE = reinterpret_cast<InfoFromLeConnStateChangedCallback*>(&g_BufferForBleCore[0]);
    pLE->status     = cbData->status;
    pLE->clientIf   = cbData->clientIf;
    pLE->connId     = cbData->connId;
    pLE->connState  = cbData->connState;
    pLE->reason     = cbData->reason;
    pLE->address    = cbData->address;

    SignalEventForBleCore(EventFromLeClientConnStateChangedCallback);
}

void LeServerConnStateChangedCallback(InfoFromLeConnStateChangedCallback *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeConnStateChangedCallback) <= BUFFER_SIZE_OF_BLE_OUT);

    InfoFromLeConnStateChangedCallback *pLE = reinterpret_cast<InfoFromLeConnStateChangedCallback*>(&g_BufferForBleCore[0]);
    pLE->status     = cbData->status;
    pLE->serverIf   = cbData->serverIf;
    pLE->connId     = cbData->connId;
    pLE->connState  = cbData->connState;
    pLE->reason     = cbData->reason;
    pLE->address    = cbData->address;

    SignalEventForBleCore(EventFromLeServerConnStateChangedCallback);
}

void LeClientGattOperationCallback(InfoFromLeGattOperationCallback *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeGattOperationCallback) <= BUFFER_SIZE_OF_BLE_OUT);
    NN_SDK_REQUIRES(cbData->length <= BUFFER_SIZE_OF_BLE_GATT_ATTRIBUTE);

    InfoFromLeGattOperationCallback *pLE = reinterpret_cast<InfoFromLeGattOperationCallback*>(&g_BufferForBle[0]);
    pLE->status             = cbData->status;
    pLE->connId             = cbData->connId;
    pLE->operation          = cbData->operation;
    pLE->serviceUuid        = cbData->serviceUuid;
    pLE->charcteristicUuid  = cbData->charcteristicUuid;
    pLE->descriptorUuid     = cbData->descriptorUuid;
    pLE->length             = cbData->length;
    memcpy(pLE->value, cbData->value, cbData->length);

    SignalEventForBle(EventFromLeClientGattOpCallback);
}

void LeClientCoreGattOperationCallback(InfoFromLeGattOperationCallback *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeGattOperationCallback) <= BUFFER_SIZE_OF_BLE_OUT);
    NN_SDK_REQUIRES(cbData->length <= BUFFER_SIZE_OF_BLE_GATT_ATTRIBUTE);

    InfoFromLeGattOperationCallback *pLE = reinterpret_cast<InfoFromLeGattOperationCallback*>(&g_BufferForBleCore[0]);
    pLE->status = cbData->status;
    pLE->connId = cbData->connId;
    pLE->operation = cbData->operation;
    pLE->serviceUuid = cbData->serviceUuid;
    pLE->charcteristicUuid = cbData->charcteristicUuid;
    pLE->descriptorUuid = cbData->descriptorUuid;
    pLE->length = cbData->length;
    memcpy(pLE->value, cbData->value, cbData->length);

    SignalEventForBleCore(EventFromLeClientGattOpCallback);
}

void LeClientHidGattOperationCallback(InfoFromLeGattOperationCallback *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeGattOperationCallback) <= BUFFER_SIZE_OF_BLE_OUT);
    NN_SDK_REQUIRES(cbData->length <= BUFFER_SIZE_OF_BLE_GATT_ATTRIBUTE);

    InfoFromLeGattOperationCallback *pLE = reinterpret_cast<InfoFromLeGattOperationCallback*>(&g_BufferForBleHid[0]);
    pLE->status = cbData->status;
    pLE->connId = cbData->connId;
    pLE->operation = cbData->operation;
    pLE->serviceUuid = cbData->serviceUuid;
    pLE->charcteristicUuid = cbData->charcteristicUuid;
    pLE->descriptorUuid = cbData->descriptorUuid;
    pLE->length = cbData->length;
    memcpy(pLE->value, cbData->value, cbData->length);

    SignalEventForBleHid(EventFromLeClientGattOpCallback);
}

void LeClientServiceDiscoveryCallback(InfoFromLeClientGattServiceDiscoveryCallback *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeClientGattServiceDiscoveryCallback) <= BUFFER_SIZE_OF_BLE_OUT);

    InfoFromLeClientGattServiceDiscoveryCallback *pLE = reinterpret_cast<InfoFromLeClientGattServiceDiscoveryCallback*>(&g_BufferForBleCore[0]);
    pLE->status             = cbData->status;
    pLE->connId             = cbData->connId;
    pLE->numAttributes      = cbData->numAttributes;
    pLE->address            = cbData->address;
    for (int i = 0; i < cbData->numAttributes; ++i)
    {
        pLE->attributes[i] = cbData->attributes[i];
    }

    SignalEventForBleCore(EventFromLeClientServiceDiscoveryCallback);
}

void LeClientServiceDiscoveryStateChangedCallback(InfoFromLeClientGattServiceDiscoveryCallback *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeClientGattServiceDiscoveryCallback) <= BUFFER_SIZE_OF_BLE_OUT);

    InfoFromLeClientGattServiceDiscoveryCallback *pLE = reinterpret_cast<InfoFromLeClientGattServiceDiscoveryCallback*>(&g_BufferForBleCore[0]);
    pLE->status             = cbData->status;
    pLE->connId             = cbData->connId;
    pLE->numAttributes      = cbData->numAttributes;
    pLE->address            = cbData->address;
    for (int i = 0; i < cbData->numAttributes; ++i)
    {
        pLE->attributes[i] = cbData->attributes[i];
    }

    SignalEventForBleCore(EventFromLeClientServiceDiscoveryStateChangedCallback);
}

void LeClientConfigureMtuCallback(InfoFromLeClientMtuConfigurationCallback *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromLeClientMtuConfigurationCallback) <= BUFFER_SIZE_OF_BLE_OUT);

    InfoFromLeClientMtuConfigurationCallback *pLE = reinterpret_cast<InfoFromLeClientMtuConfigurationCallback*>(&g_BufferForBleCore[0]);
    pLE->status     = cbData->status;
    pLE->connId     = cbData->connId;
    pLE->mtu        = cbData->mtu;

    SignalEventForBleCore(EventFromLeClientConfigureMtuCallback);
}


//-----------------------------------------------------------------------------
// Hid callbacks from the HAL
//-----------------------------------------------------------------------------
//[Todo]to decide static or not.
/*******************************************************************************
 **
 ** Function    ConnectionStateCallback
 **
 ** Description
 **
 ** Parameters
 **
 ** Returns     None
 **
 *******************************************************************************/
static void ConnectionStateCallback(Btbdaddr *pBdAddr, BluetoothHhConnectionState state)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    NN_SDK_REQUIRES(sizeof(InfoFromConnectionStateCallback) <= BUFFER_SIZE_OF_HID_OUT);
    memcpy(reinterpret_cast<InfoFromConnectionStateCallback*>(&g_BufferForHid[0])->bluetoothAddress.address, pBdAddr->address, BD_ADDR_LEN);
    reinterpret_cast<InfoFromConnectionStateCallback*>(&g_BufferForHid[0])->state = state;

    //tick=nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetMilliSeconds();
    //g_EventCount++;
    //NN_SDK_LOG("[%d-1 Impl signal]tick = %d\n",g_EventCount, tick);
    SignalEventForHid(EventFromConnectionStateCallback);
}

/*******************************************************************************
 **
 ** Function    HidInfoCallback
 **
 ** Description
 **
 ** Parameters
 **
 ** Returns     None
 **
 *******************************************************************************/
static void HidInfoCallback(Btbdaddr *pBdAddr, BluetoothHhHidInfo HidInfo)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    SignalEventForHid(EventFromHidInfoCallback);
}

/*******************************************************************************
 **
 ** Function    ProtocolModeCallback
 **
 ** Description
 **
 ** Parameters
 **
 ** Returns     None
 **
 *******************************************************************************/
static void ProtocolModeCallback(Btbdaddr *pBdAddr, BluetoothHhStatus hhStatus, BluetoothHhProtocolMode mode)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    SignalEventForHid(EventFromProtocolModeCallback);
}

/*******************************************************************************
 **
 ** Function    IdleTimeCallback
 **
 ** Description
 **
 ** Parameters
 **
 ** Returns     None
 **
 *******************************************************************************/
static void IdleTimeCallback(Btbdaddr *pBdAddr, BluetoothHhStatus hhStatus, int IdleRate)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    SignalEventForHid(EventFromIdleTimeCallback);
}

/*******************************************************************************
 **
 ** Function        VirtualUnplugCallback
 **
 ** Description
 **
 ** Parameters
 **
 ** Returns     None
 **
 *******************************************************************************/
static void VirtualUnplugCallback(Btbdaddr *pBdAddr, BluetoothHhStatus hhStatus)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    SignalEventForHid(EventFromVirtualUnplugCallback);
}

/*******************************************************************************
 **
 ** Function        HandshakeCallback
 **
 ** Description
 **
 ** Parameters
 **
 ** Returns     None
 **
 *******************************************************************************/
static void HandshakeCallback(Btbdaddr *pBdAddr, BluetoothHhStatus hhStatus)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    SignalEventForHid(EventFromHandshakeCallback);
}


//-----------------------------------------------------------------------------
// Extension Callbacks from the HAL
// //[Todo] To separate from the HID.
//-----------------------------------------------------------------------------
static void SetTsiCallback(void *p_data, BluetoothHhStatus hhStatus)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    ExtEventType eventType;

    Btbdaddr bd_addr;

    tBSA_ROBSON_MSG *cbData = (tBSA_ROBSON_MSG *)p_data;

    eventType = EventFromTsiSetCallback;
    memcpy(bd_addr.address, cbData->set_tsi.peer_addr, BD_ADDR_LEN);

    InfoFromExtensionCallbacks *pExt = reinterpret_cast<InfoFromExtensionCallbacks*>(&g_BufferForHid[0]);
    pExt->eventType        = eventType;
    memcpy(pExt->bluetoothAddress.address, bd_addr.address, BD_ADDR_LEN);
    pExt->status           = hhStatus;
    //[Todo] To provide the data to the SF Client.

    SignalEventForHid(EventFromExtensionCallbacks);
}

static void ExitTsiCallback(void *p_data, BluetoothHhStatus hhStatus)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    ExtEventType eventType;

    Btbdaddr bd_addr;

    tBSA_ROBSON_MSG *cbData = (tBSA_ROBSON_MSG *)p_data;

    eventType = EventFromTsiExitCallback;
    memcpy(bd_addr.address, cbData->exit_tsi.peer_addr, BD_ADDR_LEN);

    InfoFromExtensionCallbacks *pExt = reinterpret_cast<InfoFromExtensionCallbacks*>(&g_BufferForHid[0]);
    pExt->eventType        = eventType;
    memcpy(pExt->bluetoothAddress.address, bd_addr.address, BD_ADDR_LEN);
    pExt->status           = hhStatus;
    //[Todo] To provide the data to the SF Client.

    SignalEventForHid(EventFromExtensionCallbacks);
}

static void SetBurstCallback(void *p_data, BluetoothHhStatus hhStatus)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    ExtEventType eventType;

    Btbdaddr bd_addr;

    tBSA_ROBSON_MSG *cbData = (tBSA_ROBSON_MSG *)p_data;

    eventType = EventFromBurstSetCallback;
    memcpy(bd_addr.address, cbData->set_burst.peer_addr, BD_ADDR_LEN);

    InfoFromExtensionCallbacks *pExt = reinterpret_cast<InfoFromExtensionCallbacks*>(&g_BufferForHid[0]);
    pExt->eventType        = eventType;
    memcpy(pExt->bluetoothAddress.address, bd_addr.address, BD_ADDR_LEN);
    pExt->status           = hhStatus;
    //[Todo] To provide the data to the SF Client.

    SignalEventForHid(EventFromExtensionCallbacks);
}

static void ExitBurstCallback(void *p_data, BluetoothHhStatus hhStatus)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    ExtEventType eventType;

    Btbdaddr bd_addr;

    tBSA_ROBSON_MSG *cbData = (tBSA_ROBSON_MSG *)p_data;

    eventType = EventFromBurstExitCallback;
    memcpy(bd_addr.address, cbData->exit_burst.peer_addr, BD_ADDR_LEN);

    InfoFromExtensionCallbacks *pExt = reinterpret_cast<InfoFromExtensionCallbacks*>(&g_BufferForHid[0]);
    pExt->eventType        = eventType;
    memcpy(pExt->bluetoothAddress.address, bd_addr.address, BD_ADDR_LEN);
    pExt->status           = hhStatus;
    //[Todo] To provide the data to the SF Client.

    SignalEventForHid(EventFromExtensionCallbacks);
}

static void SetZeroRetranCallback(void *p_data, BluetoothHhStatus hhStatus)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    ExtEventType eventType;

    Btbdaddr bd_addr;

    tBSA_ROBSON_MSG *cbData = (tBSA_ROBSON_MSG *)p_data;

    eventType = EventFromSetZeroRetranCallback;
    memcpy(bd_addr.address, cbData->set_0retran_mode.peer_addr, BD_ADDR_LEN);

    InfoFromExtensionCallbacks *pExt = reinterpret_cast<InfoFromExtensionCallbacks*>(&g_BufferForHid[0]);
    pExt->eventType         = eventType;
    memcpy(pExt->bluetoothAddress.address, bd_addr.address, BD_ADDR_LEN);
    pExt->status            = hhStatus;
    pExt->zeroRetran.enable = cbData->set_0retran_mode.enable;

    SignalEventForHid(EventFromExtensionCallbacks);
}

static void GetPendingConnectionsCallback(InfoFromExtensionCallbacks *cbData)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s \n",__FUNCTION__);
    InfoFromExtensionCallbacks *pExt = reinterpret_cast<InfoFromExtensionCallbacks*>(&g_BufferForHid[0]);
    pExt->eventType              = cbData->eventType;
    pExt->bluetoothAddress       = cbData->bluetoothAddress;
    pExt->status                 = cbData->status;
    pExt->connections.numPending = cbData->connections.numPending;

    SignalEventForHid(EventFromExtensionCallbacks);
}


/*******************************************************************************
 **
 ** Function    GetReportCallback
 **
 ** Description
 **
 ** Parameters
 **
 ** Returns     None
 **
 *******************************************************************************/
static void GetReportCallback(Btbdaddr *pBdAddr, BluetoothHhStatus hhStatus, uint8_t* RptData, int RptSize)
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s RptSize=%d RptData=%x\n",__FUNCTION__,RptSize,RptData);
    if (RptData != NULL) {

        tBSA_HH_UIPC_REPORT *p_uipc_report = (tBSA_HH_UIPC_REPORT *)RptData;

        NN_SDK_REQUIRES(sizeof(InfoFromGetReportCallback) <= BUFFER_SIZE_OF_HID_OUT);
        NN_SDK_REQUIRES(RptSize <= BUFFER_SIZE_OF_HID_DATA);
        InfoFromGetReportCallback *pGR = reinterpret_cast<InfoFromGetReportCallback*>(&g_BufferForHidReport[0]);

        memcpy(pGR->bluetoothAddress.address, pBdAddr->address, BD_ADDR_LEN);
        pGR->status = hhStatus;
        pGR->rptSize = p_uipc_report->report.report_len;

        //Don't copy the whole HidReport, only the used bytes
        memcpy(&pGR->rptData, &p_uipc_report->report, p_uipc_report->report.report_len);

        SignalEventForHidReport(EventFromGetReportCallback);
    }
}

static void SetReportStatusCallback(Btbdaddr *pBdAddr, BluetoothHhStatus status) NN_NOEXCEPT
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s state=%d\n",__FUNCTION__,status);
    InfoFromSetReportStatusCallback *pMsg = reinterpret_cast<InfoFromSetReportStatusCallback*>(g_BufferForHidReport);

    NN_SDK_REQUIRES(sizeof(InfoFromSetReportStatusCallback) <= BUFFER_SIZE_OF_HID_OUT);
    pMsg->status=status;
    memcpy(pMsg->bluetoothAddress.address, pBdAddr->address, BD_ADDR_LEN);

    SignalEventForHidReport(EventFromSetReportStatusCallback);
}

static void GetReportStatusCallback(Btbdaddr *pBdAddr, BluetoothHhStatus status, uint16_t length, uint8_t *p_data) NN_NOEXCEPT
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s state=%d length=%d p=%p d=%02x:%02x:%02x:%02x\n",__FUNCTION__,status,length,
    //        p_data, p_data? p_data[0] : 0, p_data? p_data[1] : 0, p_data? p_data[2] : 0, p_data? p_data[3] : 0);
    InfoFromGetReportStatusCallback *pMsg = reinterpret_cast<InfoFromGetReportStatusCallback*>(g_BufferForHidReport);

    NN_SDK_REQUIRES(sizeof(InfoFromGetReportStatusCallback) <= BUFFER_SIZE_OF_HID_OUT);
    pMsg->status=status;
    memcpy(pMsg->bluetoothAddress.address, pBdAddr->address, BD_ADDR_LEN);
    pMsg->length=length;

    if (length < sizeof(pMsg->data))
        memcpy(pMsg->data, p_data, length);
    else
        NN_SDK_LOG("%s: buffer too large: %d\n", __func__, length);

    SignalEventForHidReport(EventFromGetReportStatusCallback);
}


//-----------------------------------------------------------------------------
// Impl functions
//-----------------------------------------------------------------------------
//[Todo] to care a result.
nn::Result BluetoothDriverImpl::InitializeBluetoothImpl(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);

    //For EMC, to enable this return. HID process handles the error and works collect.
    //return ResultUnsupported();

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)

    //Avoiding initialization of A2 environment.----------------------
    nn::settings::factory::ConfigurationId1 boardType;
    nn::settings::factory::GetConfigurationId1(&boardType);
    //NN_SDK_LOG("%s\n",boardType.string);
    if(boardType.string[0]==0x00)//Try to check SDEV0.9
    {
        return ResultUnsupported();
    }
    else //Try to check SDEV1.0
    {
        const int COMPARISON_LENGTH=13;
        char string[COMPARISON_LENGTH + 1]="SDEV_00_01_00";
        bool isSame=true;
        for(int i=0; i<COMPARISON_LENGTH; i++)
        {
            if(boardType.string[i]!=string[i])
            {
                isSame=false;
                break;
            }
        }
        if(isSame)
        {
            return ResultUnsupported();
        }
    }
    //--------------------------------------------------------------

    nn::bluetooth::FirmwareDebugSettings::Initialize();
    if (nn::bluetooth::FirmwareDebugSettings::IsSkipBoot() == true)
    {
        //Initialize after boot should work even if SkipBoot enabled.
        //The explicit initialize(Second initialize) is called by a manufacture application. Not by system processes.
        if(g_isFirstInitialize)
        {
            auto result = nn::os::CreateSystemEvent(&g_DummySystemEvent, nn::os::EventClearMode_AutoClear, true);
            NN_SDK_REQUIRES(result.IsSuccess());
            NN_UNUSED(result);

            auto handle = nn::os::GetReadableHandleOfSystemEvent( &g_DummySystemEvent );
            pOutHandle.Set( nn::sf::NativeHandle( handle, false ) );

            g_isFirstInitialize = false;

            NN_RESULT_SUCCESS;
        }
        else
        {
            NN_SDK_LOG("[bluetooth]SkipBoot enabled. But, Second Initialize works.\n");
        }
    }

    g_BluetoothInterface.InitializeBluetoothInterface();

    BluetoothCallbacks bluetoothCallbacks = {
        sizeof(BluetoothCallbacks),
        AdapterStateChangedCallback,
        AdapterPropertiesCallback,
        RemoteDevicePropertiesCallback,
        DeviceFoundCallback,
        DiscoveryStateChangedCallback,
        PinRequestCallback,
        SspRequestCallback,
        BondStateChangedCallback,
        SetReportStatusCallback,
        GetReportStatusCallback,
        AclStateChangedCallback,
        CallbackThreadEventCallback,
        DutModeReceivedCallback,
        LEtestModeCallback,
        EnergyInfoCallback,
        FatalErrorCallback
    };

    auto result=ConvertResult(g_BluetoothInterface.GetBluetoothInterface().InitializeBluetooth(&bluetoothCallbacks),0);
    if(result.IsSuccess())
    {
        result = nn::os::CreateSystemEvent(&g_SystemEvent, nn::os::EventClearMode_AutoClear, true);
        NN_SDK_REQUIRES(result.IsSuccess());

        auto handle = nn::os::GetReadableHandleOfSystemEvent( &g_SystemEvent );
        pOutHandle.Set( nn::sf::NativeHandle( handle, false ) );
    }
    return result;
#else
    NN_UNUSED(g_isFirstInitialize);
    return ResultUnsupported();
#endif
}

nn::Result BluetoothDriverImpl::EnableBluetoothImpl() NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    int status;
    status = g_BluetoothInterface.GetBluetoothInterface().EnableBluetooth();
    if (status != 0)
    {
        NN_SDK_LOG("[bluetooth] %s: EnableBluetooth failed\n", __func__);
        return ResultFail();
    }

    nn::bluetooth::BluetoothAddress myAddress;
    BluetoothProperty bluetoothProperty;

    //Set BDaddr to the chip.
    //[Todo]To define what layer handle it at the product.
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    nn::settings::factory::BdAddress bdAddress;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::settings::factory::GetBluetoothBdAddress(&bdAddress));
    for(int i = 0; i < 6; i++)
    {
        myAddress.address[i] = bdAddress.octets[i];
    }
#else
    myAddress = {{0x40,0xD2,0x8A,0xAA,0xBB,0xCC}};
#endif
    bluetoothProperty.type = BT_PROPERTY_BDADDR;
    bluetoothProperty.len = 6;
    bluetoothProperty.pVal = &myAddress.address[0];

    NN_ABORT_UNLESS_RESULT_SUCCESS(ConvertResult(g_BluetoothInterface.GetBluetoothInterface().SetAdapterProperty(&bluetoothProperty),0));

    //Set secondary Super BdAddress equal to the BdAddress with the last two bytes inverted.
    myAddress.address[4] = ~myAddress.address[4];
    myAddress.address[5] = ~myAddress.address[5];

    bluetoothProperty.type = BT_PROPERTY_SUPER_BDADDR;
    bluetoothProperty.len = 6;
    bluetoothProperty.pVal = &myAddress.address[0];

    NN_ABORT_UNLESS_RESULT_SUCCESS(ConvertResult(g_BluetoothInterface.GetBluetoothInterface().SetAdapterProperty(&bluetoothProperty),0));

    NN_RESULT_SUCCESS;
}

nn::Result BluetoothDriverImpl::DisableBluetoothImpl() NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    return ConvertResult(g_BluetoothInterface.GetBluetoothInterface().DisableBluetooth(),0);
}

nn::Result BluetoothDriverImpl::CleanupBluetoothImpl() NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    g_BluetoothInterface.GetBluetoothInterface().CleanupBluetooth();//result is void.

    nn::bluetooth::FirmwareDebugSettings::Initialize();
    if (nn::bluetooth::FirmwareDebugSettings::IsSkipBoot() == true && g_isFirstInitialize)
    {
        nn::os::DestroySystemEvent(&g_DummySystemEvent);
    }
    else
    {
        nn::os::DestroySystemEvent(&g_SystemEvent);
    }
    NN_RESULT_SUCCESS;
}

nn::Result BluetoothDriverImpl::GetAdapterPropertiesImpl(nn::sf::Out<nn::bluetooth::AdapterProperty> pAdapterProperty) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    return ConvertResult(g_BluetoothInterface.GetBluetoothInterface().GetAdapterProperties(pAdapterProperty.GetPointer()),0);
}

nn::Result BluetoothDriverImpl::GetAdapterPropertyImpl(BluetoothPropertyType propertyType, const nn::sf::OutArray<std::uint8_t>& outArray) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    BluetoothProperty bluetoothProperty;
    bluetoothProperty.type=propertyType;
    bluetoothProperty.len=outArray.GetLength();
    bluetoothProperty.pVal=const_cast<void*>(reinterpret_cast<const void*>(&outArray[0]));
    return ConvertResult(g_BluetoothInterface.GetBluetoothInterface().GetAdapterProperty(&bluetoothProperty),0);
}

nn::Result BluetoothDriverImpl::SetAdapterPropertyImpl(BluetoothPropertyType propertyType, const nn::sf::InArray<std::uint8_t>& inArray) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    BluetoothProperty bluetoothProperty;
    bluetoothProperty.type=propertyType;
    bluetoothProperty.len=inArray.GetLength();
    bluetoothProperty.pVal=const_cast<void*>(reinterpret_cast<const void*>(&inArray[0]));
    return ConvertResult(g_BluetoothInterface.GetBluetoothInterface().SetAdapterProperty(&bluetoothProperty),0);
}

nn::Result BluetoothDriverImpl::StartDiscoveryImpl() NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    return ConvertResult(g_BluetoothInterface.GetBluetoothInterface().StartDiscovery(),0);
}

nn::Result BluetoothDriverImpl::CancelDiscoveryImpl() NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    return ConvertResult(g_BluetoothInterface.GetBluetoothInterface().CancelDiscovery(),0);
}

nn::Result BluetoothDriverImpl::CreateBondImpl(BluetoothAddress bluetoothAddress, BluetoothTransport transport) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    //PrintMacAddress(&bluetoothAddress);
    //NN_SDK_LOG("transport=%d\n",transport);
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);
    return ConvertResult(g_BluetoothInterface.GetBluetoothInterface().CreateBond(&address,transport),0);
}

nn::Result BluetoothDriverImpl::RemoveBondImpl(BluetoothAddress bluetoothAddress) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    //PrintMacAddress(&bluetoothAddress);
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);
    return ConvertResult(g_BluetoothInterface.GetBluetoothInterface().RemoveBond(&address),0);
}

nn::Result BluetoothDriverImpl::CancelBondImpl(BluetoothAddress bluetoothAddress) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    PrintMacAddress(&bluetoothAddress);
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);
    return ConvertResult(g_BluetoothInterface.GetBluetoothInterface().CancelBond(&address),0);
}

nn::Result BluetoothDriverImpl::PinReplyImpl(BluetoothAddress bluetoothAddress, uint8_t accept, uint8_t pinLength, BluetoothPinCode pinCode) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);
    return ConvertResult(g_BluetoothInterface.GetBluetoothInterface().PinReply(&address, accept, pinLength, &pinCode),0);
}
nn::Result BluetoothDriverImpl::SspReplyImpl(BluetoothAddress bluetoothAddress, uint8_t variant, uint8_t accept, uint32_t passkey) NN_NOEXCEPT
{
    //[Todo]to care a implicit enum cast.
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);
    return ConvertResult(g_BluetoothInterface.GetBluetoothInterface().SspReply(&address, static_cast<BluetoothSspVariant>(variant), accept, passkey),0);
}


nn::Result BluetoothDriverImpl::GetEventInfoImpl(nn::sf::Out<std::uint32_t> pEventType, const nn::sf::OutArray<std::uint8_t>& outArray) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);

    NN_SDK_REQUIRES(outArray.GetLength() >= BUFFER_SIZE_OF_CORE_OUT);
    memcpy(&outArray[0], &g_Buffer[0], BUFFER_SIZE_OF_CORE_OUT);
    *pEventType=g_EventType;
    nn::os::SignalEvent(&g_InternalEvent);
    NN_RESULT_SUCCESS;
}




//-----------------------------------------------------------------------------
// Impl functions for Hid
//-----------------------------------------------------------------------------
nn::Result BluetoothDriverImpl::InitializeHidImpl(nn::sf::Out<nn::sf::NativeHandle> pOutHandle, uint16_t interfaceVersion) NN_NOEXCEPT
{
    //NN_SDK_LOG("BLUETOOTH_HAL_IMPL: %s\n",__FUNCTION__);
    nn::bluetooth::FirmwareDebugSettings::Initialize();
    if (nn::bluetooth::FirmwareDebugSettings::IsSkipBoot() == true)
    {
        auto result = nn::os::CreateSystemEvent(&g_DummySystemEventForHid, nn::os::EventClearMode_AutoClear, true);
        NN_SDK_REQUIRES(result.IsSuccess());
        auto handle = nn::os::GetReadableHandleOfSystemEvent( &g_DummySystemEventForHid );
        pOutHandle.Set( nn::sf::NativeHandle( handle, false ) );

        result = nn::os::CreateSystemEvent( &g_DummySystemEventForHidReport, nn::os::EventClearMode_AutoClear, true);
        NN_SDK_REQUIRES(result.IsSuccess());

        NN_UNUSED(result);

        NN_RESULT_SUCCESS;
    }

    auto result = nn::os::CreateSystemEvent(&g_SystemEventForHid, nn::os::EventClearMode_AutoClear, true);
    NN_SDK_REQUIRES(result.IsSuccess());
    auto handle = nn::os::GetReadableHandleOfSystemEvent( &g_SystemEventForHid );
    pOutHandle.Set( nn::sf::NativeHandle( handle, false ) );

    result = nn::os::CreateSystemEvent(&g_SystemEventForHidReport, nn::os::EventClearMode_AutoClear, true);
    NN_SDK_REQUIRES(result.IsSuccess());

    NN_UNUSED(result);

    g_pBluetoothHidInterface = reinterpret_cast<const BluetoothHhInterface*>(g_BluetoothInterface.GetBluetoothInterface().GetProfileInterface(BT_PROFILE_HIDHOST_ID));
    NN_SDK_REQUIRES_NOT_NULL(g_pBluetoothHidInterface);

    BluetoothHhStatus status;
    static BluetoothHhCallbacks BluetoothHidCallbacks = {
        sizeof(BluetoothHhCallbacks),
        ConnectionStateCallback,
        HidInfoCallback,
        ProtocolModeCallback,
        IdleTimeCallback,
        GetReportCallback,
        VirtualUnplugCallback,
        HandshakeCallback,
        SetReportStatusCallback,
        GetReportStatusCallback
    };

    status = g_pBluetoothHidInterface->InitializeHid(&BluetoothHidCallbacks, interfaceVersion);

    NN_SDK_REQUIRES_EQUAL(status,BT_STATUS_SUCCESS);

    //Initialize the extension
    g_pBluetoothExtensionInterface = reinterpret_cast<const BluetoothExtension*>(g_BluetoothInterface.GetBluetoothInterface().GetProfileInterface(BT_PROFILE_EXTENSION));
    NN_SDK_REQUIRES_NOT_NULL(g_pBluetoothExtensionInterface);
    static BluetoothExtCallbacks BluetoothExtensionCallbacks = {
            sizeof(BluetoothExtensionCallbacks),
            SetTsiCallback,
            ExitTsiCallback,
            SetBurstCallback,
            ExitBurstCallback,
            SetZeroRetranCallback,
            GetPendingConnectionsCallback,
    };

    status = g_pBluetoothExtensionInterface->InitializeExtension(&BluetoothExtensionCallbacks);
    NN_SDK_REQUIRES_EQUAL(status,BT_STATUS_SUCCESS);


    NN_RESULT_SUCCESS;
}

nn::Result BluetoothDriverImpl::HidConnectImpl(BluetoothAddress bluetoothAddress) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    //PrintMacAddress(&bluetoothAddress);
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);
    return ConvertResult(g_pBluetoothHidInterface->connect(&address),1);
}

nn::Result BluetoothDriverImpl::HidDisconnectImpl(BluetoothAddress bluetoothAddress) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    //PrintMacAddress(&bluetoothAddress);
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);
    return ConvertResult(g_pBluetoothHidInterface->disconnect(&address),1);
}

nn::Result BluetoothDriverImpl::HidSendDataByStructureImpl(BluetoothAddress bluetoothAddress, HidData data) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);
    return ConvertResult(g_pBluetoothHidInterface->SendData(&address, reinterpret_cast<char*>(&data)),1);
}

nn::Result BluetoothDriverImpl::HidSendDataImpl(BluetoothAddress bluetoothAddress, const nn::sf::InArray<std::uint8_t>& inArray) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    //NN_SDK_LOG("[Impl]starting time = %d\n", nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetMilliSeconds());
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);
    return ConvertResult(g_pBluetoothHidInterface->SendData(&address, reinterpret_cast<char*>(const_cast<uint8_t*>(&inArray[0]))),1);
}

nn::Result BluetoothDriverImpl::HidSetReportImpl(BluetoothAddress bluetoothAddress, BluetoothHhReportType reportType, HidData data) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);
    return ConvertResult(g_pBluetoothHidInterface->SetReport(&address, reportType, reinterpret_cast<char*>(&data)),1);
}

nn::Result BluetoothDriverImpl::HidGetReportImpl(BluetoothAddress bluetoothAddress, BluetoothHhReportType reportType, uint8_t reportId) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);
    return ConvertResult(g_pBluetoothHidInterface->GetReport(&address, reportType, reportId, nn::bluetooth::BUFFER_SIZE_OF_HID_DATA),1);
}

nn::Result BluetoothDriverImpl::HidWakeControllerImpl(BluetoothAddress bluetoothAddress) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);
    return ConvertResult(g_pBluetoothHidInterface->WakeController(&address),1);
}

nn::Result BluetoothDriverImpl::HidAddPairedDeviceImpl(nn::settings::system::BluetoothDevicesSettings device) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    return ConvertResult(g_pBluetoothHidInterface->AddPairedDevice(&device),1);
}

nn::Result BluetoothDriverImpl::HidGetPairedDeviceImpl(BluetoothAddress bluetoothAddress, nn::sf::Out<nn::settings::system::BluetoothDevicesSettings> deviceOut) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);
    return ConvertResult(g_pBluetoothHidInterface->GetPairedDevice(&address, deviceOut.GetPointer()),1);
}

nn::Result BluetoothDriverImpl::CleanupHidImpl() NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    g_pBluetoothHidInterface->CleanupHid();//return value is void.

    nn::bluetooth::FirmwareDebugSettings::Initialize();
    if (nn::bluetooth::FirmwareDebugSettings::IsSkipBoot() == true)
    {
        nn::os::DestroySystemEvent(&g_DummySystemEventForHid);
        nn::os::DestroySystemEvent(&g_DummySystemEventForHidReport);
    }
    else
    {
        nn::os::DestroySystemEvent(&g_SystemEventForHid);
        nn::os::DestroySystemEvent(&g_SystemEventForHidReport);
    }
    NN_RESULT_SUCCESS;
}

nn::Result BluetoothDriverImpl::HidGetEventInfoImpl(nn::sf::Out<std::uint32_t> pEventType, const nn::sf::OutArray<std::uint8_t>& outArray) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);

    NN_SDK_REQUIRES(outArray.GetLength() >= BUFFER_SIZE_OF_HID_OUT);
    memcpy(&outArray[0], &g_BufferForHid[0], BUFFER_SIZE_OF_HID_OUT);
    *pEventType=g_EventTypeForHid;
    nn::os::SignalEvent(&g_InternalEventForHid);
    //tick=nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetMilliSeconds();
    //NN_SDK_LOG("[%d-3 Impl return]tick = %d\n",g_EventCount, tick);
    NN_RESULT_SUCCESS;
}

nn::Result BluetoothDriverImpl::RegisterHidReportEventImpl(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT
{
    nn::bluetooth::FirmwareDebugSettings::Initialize();
    if (nn::bluetooth::FirmwareDebugSettings::IsSkipBoot() == true)
    {
        auto handle = nn::os::GetReadableHandleOfSystemEvent( &g_DummySystemEventForHidReport );
        pOutHandle.Set( nn::sf::NativeHandle( handle, false ) );
    }
    else
    {
        auto handle = nn::os::GetReadableHandleOfSystemEvent( &g_SystemEventForHidReport );
        pOutHandle.Set( nn::sf::NativeHandle( handle, false ) );
    }

    NN_RESULT_SUCCESS;
}

nn::Result BluetoothDriverImpl::HidGetReportEventInfoImpl(nn::sf::Out<uint32_t> pEventType, const nn::sf::OutArray<std::uint8_t>& outArray) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(outArray.GetLength() >= BUFFER_SIZE_OF_HID_OUT);
    memcpy(&outArray[0], &g_BufferForHidReport[0], BUFFER_SIZE_OF_HID_OUT);
    *pEventType = g_EventTypeForHidReport;
    nn::os::SignalEvent(&g_InternalEventForHidReport);
    NN_RESULT_SUCCESS;
}
//-----------------------------------------------------------------------------
// Impl functions for Extension
//-----------------------------------------------------------------------------
nn::Result BluetoothDriverImpl::ExtSetTsiImpl(BluetoothAddress bluetoothAddress, uint8_t tsiMode) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s TsiNumber=%d\n",__FUNCTION__, tsiMode);
    BluetoothHhStatus status;
    Btbdaddr address;

    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);

    if(tsiMode==0xFF)
    {
        status=g_pBluetoothExtensionInterface->HalIfExitTSI(&address);
    }
    else
    {
        status=g_pBluetoothExtensionInterface->HalIfSetTSI(&address, tsiMode);
    }
    return ConvertResult(status,1);
}

nn::Result BluetoothDriverImpl::ExtSetBurstModeImpl(BluetoothAddress bluetoothAddress, bool isBurst) NN_NOEXCEPT
{
    //NN_SDK_LOG("%s isBurst=%d\n",__FUNCTION__, isBurst);
    BluetoothHhStatus status;
    Btbdaddr address;

    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);

    if(isBurst)
    {
        status=g_pBluetoothExtensionInterface->HalIfSetBurstMode(&address);
    }
    else
    {
        status=g_pBluetoothExtensionInterface->HalIfExitBurstMode(&address);
    }
    return ConvertResult(status,1);
}

nn::Result BluetoothDriverImpl::ExtSetZeroRetranImpl(BluetoothAddress bluetoothAddress, const nn::sf::InArray<std::uint8_t>& reportIds) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s, length=%d  ",__FUNCTION__, reportIds.GetLength());
    //for (int i=0; i<reportIds.GetLength(); i++) {
    //    NN_SDK_LOG("  id[%i]=%d", i, reportIds[i]);
    //}
    //NN_SDK_LOG("\n");
    BluetoothHhStatus status = BTHH_OK;
    Btbdaddr address;

    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);

    status=g_pBluetoothExtensionInterface->HalIfSetZeroRetran(&address, reportIds.GetLength(), &reportIds[0]);
    return ConvertResult(status,1);
}


nn::Result BluetoothDriverImpl::ExtSetMcModeImpl(bool isMcModeEnabled)
{
    //NN_SDK_LOG("[bluetooth] %s  enable=%d",__FUNCTION__, isMcModeEnabled);
    BluetoothHhStatus status = BTHH_OK;
    status=g_pBluetoothExtensionInterface->HalIfSetMcMode(isMcModeEnabled);
    return ConvertResult(status,1);
}

nn::Result BluetoothDriverImpl::ExtStartLlrModeImpl()
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    BluetoothHhStatus status = BTHH_OK;
    status=g_pBluetoothExtensionInterface->HalIfStartLlrMode();
    return ConvertResult(status,1);
}

nn::Result BluetoothDriverImpl::ExtExitLlrModeImpl()
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    BluetoothHhStatus status = BTHH_OK;
    status=g_pBluetoothExtensionInterface->HalIfExitLlrMode();
    return ConvertResult(status,1);
}

nn::Result BluetoothDriverImpl::ExtSetRadioImpl(bool enable)
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    BluetoothHhStatus status = BTHH_OK;
    status=g_pBluetoothExtensionInterface->HalIfSetRadio(enable);
    return ConvertResult(status,1);
}

nn::Result BluetoothDriverImpl::ExtSetVisibilityImpl(bool discoverable, bool connectable)
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    BluetoothHhStatus status = BTHH_OK;
    status=g_pBluetoothExtensionInterface->HalIfSetVisibility(discoverable, connectable);
    return ConvertResult(status,1);
}

nn::Result BluetoothDriverImpl::ExtSetTbfcScanImpl(bool enable)
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    BluetoothHhStatus status = BTHH_OK;
    status=g_pBluetoothExtensionInterface->HalIfSetTbfcScan(enable);
    return ConvertResult(status,1);
}

nn::Result BluetoothDriverImpl::ExtGetPendingConnectionsImpl()
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    BluetoothHhStatus status = BTHH_OK;
    status=g_pBluetoothExtensionInterface->HalIfGetPendingConnections();
    return ConvertResult(status,1);
}

nn::Result BluetoothDriverImpl::GetLatestPlrImpl(nn::sf::Out<nn::bluetooth::PlrStatistics> pOut)
{
    g_pBluetoothHidInterface->GetLatestPlr(pOut.GetPointer());
    NN_RESULT_SUCCESS;
}

nn::Result BluetoothDriverImpl::GetChannelMapImpl(nn::sf::Out<ChannelMap> pOutChannelMap)
{
    g_pBluetoothHidInterface->GetChannelMap(pOutChannelMap.GetPointer());
    NN_RESULT_SUCCESS;
}

nn::Result BluetoothDriverImpl::EnableBluetoothBoostSettingImpl(bool enable) NN_NOEXCEPT
{
    g_BluetoothInterface.GetBluetoothInterface().EnableBluetoothBoostSetting(enable);
    NN_RESULT_SUCCESS;
}

nn::Result BluetoothDriverImpl::IsBluetoothBoostEnabledImpl(nn::sf::Out<bool> pOut) NN_NOEXCEPT
{
    *pOut = g_BluetoothInterface.GetBluetoothInterface().IsBluetoothBoostEnabled();

    NN_RESULT_SUCCESS;
}

nn::Result BluetoothDriverImpl::EnableBluetoothAfhSettingImpl(bool enable) NN_NOEXCEPT
{
    g_BluetoothInterface.GetBluetoothInterface().EnableBluetoothAfhSetting(enable);
    NN_RESULT_SUCCESS;
}

nn::Result BluetoothDriverImpl::IsBluetoothAfhEnabledImpl(nn::sf::Out<bool> pOut) NN_NOEXCEPT
{
    *pOut = g_BluetoothInterface.GetBluetoothInterface().IsBluetoothAfhEnabled();

    NN_RESULT_SUCCESS;
}

NN_OS_EXTERN_C void Bluetooth_HAL_test_API(const char *api)
{
    //NN_SDK_LOG("Hello Bluetooth HAL..api=%s\n",api);

    // sample trackpad. 40:a6:d9:ff:bf:dd
    //Btbdaddr bd_addr = {{0x40,0xa6,0xd9,0xff,0xbf,0xdd}};
    // sample mouse. b8:f6:b1:2d:09:89
    //Btbdaddr bd_addr = {{0xB8,0xF6,0xB1,0x2D,0x09,0x89}};
    // sample keyboard. 78:ca:39:51:b2:10
    Btbdaddr bd_addr = {{0x78,0xCA,0x39,0x51,0xB2,0x10}};
    auto& bluetoothInterface = g_BluetoothInterface.GetBluetoothInterface();

    if (strcmp(api,"init")==0)
    {
        // Initialize the client
        //bcmApplication();
    }
    else if (strcmp(api,"StartDiscovery")==0)
    {
        bluetoothInterface.StartDiscovery();
        bluetoothInterface.CreateBond(&bd_addr,0);
    }
    else if (strcmp(api,"CancelDiscovery")==0)
    {
        bluetoothInterface.CancelDiscovery();
    }
    else if (strcmp(api,"CreateBond")==0)
    {
        bluetoothInterface.CreateBond(&bd_addr,0);
    }
    else if (strcmp(api,"GetRemoteServices")==0)
    {
        bluetoothInterface.GetRemoteServices(&bd_addr);
    }
    else if (strcmp(api,"SspReplyAccept")==0)
    {
        bluetoothInterface.SspReply(&bd_addr,BT_SSP_VARIANT_PASSKEY_ENTRY,true,0x1234);
    }
    else if (strcmp(api,"SspReplyDeny")==0)
    {
        bluetoothInterface.SspReply(&bd_addr,BT_SSP_VARIANT_PASSKEY_ENTRY,false,0x1234);
    }
#if 0
    else if (strcmp(api,"connect")==0)
    {
        pBluetoothHidInterface->connect(&bd_addr);
    }
    else if (strcmp(api,"disconnect")==0)
    {
        pBluetoothHidInterface->disconnect(&bd_addr);
    }
    else if (strcmp(api,"SetInfo")==0)
    {
        BluetoothHhHidInfo HidInfo;
        pBluetoothHidInterface->SetInfo(&bd_addr,HidInfo);
    }
    else if (strcmp(api,"GetProtocol")==0)
    {
        pBluetoothHidInterface->GetProtocol(&bd_addr,BTHH_REPORT_MODE);
    }
    else if (strcmp(api,"SetProtocol")==0)
    {
        pBluetoothHidInterface->SetProtocol(&bd_addr,BTHH_REPORT_MODE);
    }
    else if (strcmp(api,"GetReport")==0)
    {
        uint8_t GetReportId = 0x30; // example of a report id for Wii Remotes

        // bufferSize is not used
        pBluetoothHidInterface->GetReport(&bd_addr, BTHH_INPUT_REPORT,GetReportId,0);
    }
    else if (strcmp(api,"SetReport")==0)
    {
        pBluetoothHidInterface->SetReport(&bd_addr,BTHH_INPUT_REPORT, NULL);
    }
    else if (strcmp(api,"SendData")==0)
    {
        char data[] = {'c','o','d','e','\0'};

        pBluetoothHidInterface->SendData(&bd_addr,data);
    }
    else if (strcmp(api,"CleanupHid")==0)
    {
        pBluetoothHidInterface->CleanupHid();
    }
#endif
    else
    {
        NN_SDK_LOG("Unsupported command: %s\n", api);
    }
}


//-----------------------------------------------------------------------------
// Impl functions for BLE
//-----------------------------------------------------------------------------

nn::Result BluetoothDriverImpl::InitializeBleImpl(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    nn::bluetooth::FirmwareDebugSettings::Initialize();
    if (nn::bluetooth::FirmwareDebugSettings::IsSkipBoot() == true)
    {
        auto result = nn::os::CreateSystemEvent(&g_DummySystemEventForBleCore, nn::os::EventClearMode_AutoClear, true);
        NN_SDK_REQUIRES(result.IsSuccess());

        result = nn::os::CreateSystemEvent(&g_DummySystemEventForBleHid, nn::os::EventClearMode_AutoClear, true);
        NN_SDK_REQUIRES(result.IsSuccess());

        result = nn::os::CreateSystemEvent(&g_DummySystemEventForBle, nn::os::EventClearMode_AutoClear, true);
        NN_SDK_REQUIRES(result.IsSuccess());

        auto handle = nn::os::GetReadableHandleOfSystemEvent(&g_DummySystemEventForBleCore);
        pOutHandle.Set(nn::sf::NativeHandle(handle, false));

        NN_RESULT_SUCCESS;
    }

    g_pBluetoothLowEnergyInterface = reinterpret_cast<const BluetoothLeInterface*>(g_BluetoothInterface.GetBluetoothInterface().GetProfileInterface(BT_PROFILE_LOW_ENERGY));
    NN_SDK_REQUIRES_NOT_NULL(g_pBluetoothLowEnergyInterface);
    static BluetoothLeCallbacks BluetoothLowEnergyCallbacks = {
            sizeof(BluetoothLowEnergyCallbacks),
            LeConnParamUpdateCallback,
            LeConnParamUpdateReqCallback,
            LeScanStateChangedCallback,
            LeScanFilterStateChangedCallback,
            LeClientStateChangedCallback,
            LeServerStateChangedCallback,
            LeClientConnStateChangedCallback,
            LeServerConnStateChangedCallback,
            LeClientGattOperationCallback,
            LeClientCoreGattOperationCallback,
            LeClientHidGattOperationCallback,
            LeClientServiceDiscoveryCallback,
            LeClientServiceDiscoveryStateChangedCallback,
            LeClientConfigureMtuCallback,
            LeServerProfileChangedCallback,
            LeServerGattReqCallback,
            LeServerCoreGattReqCallback,
            LeServerHidGattReqCallback,
    };
    auto result = ConvertLeResult(g_pBluetoothLowEnergyInterface->InitializeBle(&BluetoothLowEnergyCallbacks));
    NN_SDK_REQUIRES(result.IsSuccess());

    result = nn::os::CreateSystemEvent(&g_SystemEventForBleCore, nn::os::EventClearMode_AutoClear, true);
    NN_SDK_REQUIRES(result.IsSuccess());

    result = nn::os::CreateSystemEvent(&g_SystemEventForBleHid, nn::os::EventClearMode_AutoClear, true);
    NN_SDK_REQUIRES(result.IsSuccess());

    result = nn::os::CreateSystemEvent(&g_SystemEventForBle, nn::os::EventClearMode_AutoClear, true);
    NN_SDK_REQUIRES(result.IsSuccess());

    auto handle = nn::os::GetReadableHandleOfSystemEvent(&g_SystemEventForBleCore);
    pOutHandle.Set(nn::sf::NativeHandle(handle, false));

    return result;
}

nn::Result BluetoothDriverImpl::CleanupBleImpl() NN_NOEXCEPT
{
    //NN_SDK_LOG("%s\n",__FUNCTION__);
    g_pBluetoothLowEnergyInterface->CleanupBle();  //result is void.

    nn::bluetooth::FirmwareDebugSettings::Initialize();
    if (nn::bluetooth::FirmwareDebugSettings::IsSkipBoot() == true)
    {
        nn::os::DestroySystemEvent(&g_DummySystemEventForBleCore);
        nn::os::DestroySystemEvent(&g_DummySystemEventForBleHid);
        nn::os::DestroySystemEvent(&g_DummySystemEventForBle);
    }
    else
    {
        nn::os::DestroySystemEvent(&g_SystemEventForBleCore);
        nn::os::DestroySystemEvent(&g_SystemEventForBleHid);
        nn::os::DestroySystemEvent(&g_SystemEventForBle);
    }
    NN_RESULT_SUCCESS;
}

nn::Result BluetoothDriverImpl::EnableBleImpl() NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s\n",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->EnableBle());
}

nn::Result BluetoothDriverImpl::DisableBleImpl() NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->DisableBle());
}

nn::Result BluetoothDriverImpl::SetLeVisibilityImpl(bool discoverable, bool connectable) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->SetLeVisibility(discoverable, connectable));
}

nn::Result BluetoothDriverImpl::SetLeConnectionParameterImpl(LeConnectionParams connectionParams) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->SetLeConnectionParameter(connectionParams));
}

nn::Result BluetoothDriverImpl::SetLeDefaultConnectionParameterImpl(LeConnectionParams connectionParams) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->SetLeDefaultConnectionParameter(connectionParams));
}

nn::Result BluetoothDriverImpl::SetLeAdvertiseDataImpl(LeAdvertiseData advertiseData) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->SetLeAdvertiseData(advertiseData));
}

nn::Result BluetoothDriverImpl::SetLeAdvertiseParameterImpl(BluetoothAddress bluetoothAddress, uint16_t minAdvInt, uint16_t maxAdvInt) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);

    return ConvertLeResult(g_pBluetoothLowEnergyInterface->SetLeAdvertiseParameter(&address, minAdvInt, maxAdvInt));
}

nn::Result BluetoothDriverImpl::StartLeScanImpl() NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->StartLeScan());
}

nn::Result BluetoothDriverImpl::StopLeScanImpl() NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->StopLeScan());
}

nn::Result BluetoothDriverImpl::SetLeScanParameterImpl(uint16_t scanInterval, uint16_t scanWindow) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s", __FUNCTION__");
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->SetLeScanParameter(scanInterval, scanWindow));
}

nn::Result BluetoothDriverImpl::AddLeScanFilterConditionImpl(BleAdvertiseFilter filter) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->AddLeScanFilterCondition(&filter));
}

nn::Result BluetoothDriverImpl::DeleteLeScanFilterConditionImpl(BleAdvertiseFilter filter) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->DeleteLeScanFilterCondition(&filter));
}

nn::Result BluetoothDriverImpl::DeleteLeScanFilterImpl(uint8_t filterIndex) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->DeleteLeScanFilter(filterIndex));
}

nn::Result BluetoothDriverImpl::ClearLeScanFiltersImpl() NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->ClearLeScanFilters());
}

nn::Result BluetoothDriverImpl::EnableLeScanFilterImpl(bool enable) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->EnableLeScanFilter(enable));
}

nn::Result BluetoothDriverImpl::RegisterLeClientImpl(GattAttributeUuid uuid) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->RegisterLeClient(uuid));
}

nn::Result BluetoothDriverImpl::UnregisterLeClientImpl(uint8_t clientIf) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->UnregisterLeClient(clientIf));
}

nn::Result BluetoothDriverImpl::UnregisterLeClientAllImpl() NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->UnregisterLeClientAll());
}

nn::Result BluetoothDriverImpl::LeClientConnectImpl(
    const nn::applet::AppletResourceUserId& aruid,
    uint8_t clientIf, BluetoothAddress bluetoothAddress, bool isDirect) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);

    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientConnect(clientIf, &address, isDirect));
}

nn::Result BluetoothDriverImpl::LeClientCancelConnectionImpl(uint8_t clientIf, BluetoothAddress bluetoothAddress, bool isDirect) NN_NOEXCEPT
{
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);

    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientCancelConnection(clientIf, &address, isDirect));
}

nn::Result BluetoothDriverImpl::LeClientDisconnectImpl(uint32_t connId) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientDisconnect(connId));
}

nn::Result BluetoothDriverImpl::LeClientGetAttributesImpl(uint32_t connId, BluetoothAddress bluetoothAddress) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);

    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientGetAttributes(connId, &address));
}

nn::Result BluetoothDriverImpl::LeClientDiscoverServiceImpl(uint32_t connId, GattAttributeUuid uuid) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientDiscoverService(connId, uuid));
}

nn::Result BluetoothDriverImpl::LeClientConfigureMtu(uint32_t connId, uint16_t mtu) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientConfigureMtu(connId, mtu));
}

nn::Result BluetoothDriverImpl::RegisterLeServerImpl(GattAttributeUuid uuid) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->RegisterLeServer(uuid));
}

nn::Result BluetoothDriverImpl::UnregisterLeServerImpl(uint8_t serverIf) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->UnregisterLeServer(serverIf));
}

nn::Result BluetoothDriverImpl::LeServerConnectImpl(uint8_t serverIf, BluetoothAddress bluetoothAddress, bool isDirect) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);

    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeServerConnect(serverIf, &address, isDirect));
}

nn::Result BluetoothDriverImpl::LeServerDisconnectImpl(uint8_t serverIf, BluetoothAddress bluetoothAddress) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    Btbdaddr address;
    memcpy(address.address, bluetoothAddress.address, BD_ADDR_LEN);

    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeServerDisconnect(serverIf, &address));
}

nn::Result BluetoothDriverImpl::CreateLeServiceImpl(uint8_t serverIf, GattAttributeUuid uuid, uint8_t handleNum, bool isPrimaryService) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->CreateLeService(serverIf, uuid, handleNum, isPrimaryService));
}

nn::Result BluetoothDriverImpl::StartLeServiceImpl(uint8_t serverIf, GattAttributeUuid uuid) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->StartLeService(serverIf, uuid));
}

nn::Result BluetoothDriverImpl::AddLeCharacteristicImpl(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid charUuid, uint16_t permissions, uint8_t properties) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->AddLeCharacteristic(serverIf, serviceUuid, charUuid, permissions, properties));
}

nn::Result BluetoothDriverImpl::AddLeDescriptorImpl(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid descUuid, uint16_t permissions) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->AddLeDescriptor(serverIf, serviceUuid, descUuid, permissions));
}

nn::Result BluetoothDriverImpl::GetLeCoreEventInfoImpl(nn::sf::Out<std::uint32_t> pEventType, const nn::sf::OutArray<std::uint8_t>& outArray) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(outArray.GetLength() >= BUFFER_SIZE_OF_BLE_OUT);
    memcpy(&outArray[0], &g_BufferForBleCore[0], BUFFER_SIZE_OF_BLE_OUT);
    *pEventType = g_EventTypeForBleCore;
    nn::os::SignalEvent(&g_InternalEventForBleCore);
    NN_RESULT_SUCCESS;
}

nn::Result BluetoothDriverImpl::LeGetFirstCharacteristicImpl(nn::sf::Out<nn::bluetooth::GattId> charOut, nn::sf::Out<std::uint8_t> property, uint32_t connId, GattId serviceId, bool isPrimary, GattAttributeUuid charFilter) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeGetFirstCharacteristic(charOut.GetPointer(), property.GetPointer(), connId, serviceId, isPrimary, charFilter));
}

nn::Result BluetoothDriverImpl::LeGetNextCharacteristicImpl(nn::sf::Out<nn::bluetooth::GattId> charOut, nn::sf::Out<std::uint8_t> property, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattAttributeUuid charFilter) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeGetNextCharacteristic(charOut.GetPointer(), property.GetPointer(), connId, serviceId, isPrimary, charIn, charFilter));
}

nn::Result BluetoothDriverImpl::LeGetFirstDescriptorImpl(nn::sf::Out<nn::bluetooth::GattId> descrOut, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattAttributeUuid descrFilter) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeGetFirstDescriptor(descrOut.GetPointer(), connId, serviceId, isPrimary, charIn, descrFilter));
}

nn::Result BluetoothDriverImpl::LeGetNextDescriptorImpl(nn::sf::Out<nn::bluetooth::GattId> descrOut, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattId descrIn, GattAttributeUuid descrFilter) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeGetNextDescriptor(descrOut.GetPointer(), connId, serviceId, isPrimary, charIn, descrIn, descrFilter));
}

nn::Result BluetoothDriverImpl::RegisterLeCoreDataPathImpl(GattAttributeUuid uuid) NN_NOEXCEPT
{
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->RegisterLeCoreDataPath(uuid));
}

nn::Result BluetoothDriverImpl::UnregisterLeCoreDataPathImpl(GattAttributeUuid uuid) NN_NOEXCEPT
{
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->UnregisterLeCoreDataPath(uuid));
}

nn::Result BluetoothDriverImpl::RegisterLeHidDataPathImpl(GattAttributeUuid uuid) NN_NOEXCEPT
{
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->RegisterLeHidDataPath(uuid));
}

nn::Result BluetoothDriverImpl::UnregisterLeHidDataPathImpl(GattAttributeUuid uuid) NN_NOEXCEPT
{
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->UnregisterLeHidDataPath(uuid));
}

nn::Result BluetoothDriverImpl::RegisterLeDataPathImpl(GattAttributeUuid uuid) NN_NOEXCEPT
{
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->RegisterLeDataPath(uuid));
}

nn::Result BluetoothDriverImpl::UnregisterLeDataPathImpl(GattAttributeUuid uuid) NN_NOEXCEPT
{
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->UnregisterLeDataPath(uuid));
}

nn::Result BluetoothDriverImpl::RegisterBleHidEventImpl(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT
{
    nn::bluetooth::FirmwareDebugSettings::Initialize();
    if (nn::bluetooth::FirmwareDebugSettings::IsSkipBoot() == true)
    {
        auto handle = nn::os::GetReadableHandleOfSystemEvent(&g_DummySystemEventForBleHid);
        pOutHandle.Set(nn::sf::NativeHandle(handle, false));
    }
    else
    {
        auto handle = nn::os::GetReadableHandleOfSystemEvent(&g_SystemEventForBleHid);
        pOutHandle.Set(nn::sf::NativeHandle(handle, false));
    }

    NN_RESULT_SUCCESS;
}

nn::Result BluetoothDriverImpl::GetLeHidEventInfoImpl(nn::sf::Out<std::uint32_t> pEventType, const nn::sf::OutArray<std::uint8_t>& outArray) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(outArray.GetLength() >= BUFFER_SIZE_OF_BLE_OUT);
    memcpy(&outArray[0], &g_BufferForBleHid[0], BUFFER_SIZE_OF_BLE_OUT);

    // Clear g_BufferForBleHid to indicate there is no operation result
    InfoFromLeGattOperationCallback *pLE = reinterpret_cast<InfoFromLeGattOperationCallback*>(&g_BufferForBleHid[0]);
    pLE->status     = BluetoothLeStatus::BT_GATT_ERR_NO_RESULT;
    pLE->connId     = BleInvalidConnectionHandle;
    pLE->operation  = GattOperationType_Unknown;
    pLE->length     = 0;
    memset(pLE->value, 0x00, sizeof(pLE->value));
    memset(&pLE->serviceUuid, 0x00, sizeof(GattAttributeUuid));
    memset(&pLE->charcteristicUuid, 0x00, sizeof(GattAttributeUuid));
    memset(&pLE->descriptorUuid, 0x00, sizeof(GattAttributeUuid));

    *pEventType = g_EventTypeForBleHid;
    nn::os::SignalEvent(&g_InternalEventForBleHid);
    NN_RESULT_SUCCESS;
}

nn::Result BluetoothDriverImpl::LeClientReadCharacteristicImpl(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, uint8_t authType) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientReadCharacteristic(connId, serviceId, isPrimary, charId, authType));
}

nn::Result BluetoothDriverImpl::LeClientReadDescriptorImpl(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, uint8_t authType) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientReadDescriptor(connId, serviceId, isPrimary, charId, descrId, authType));
}

nn::Result BluetoothDriverImpl::LeClientWriteCharacteristicImpl(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, const nn::sf::InArray<std::uint8_t>& pData, uint8_t authType, bool withResponse) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientWriteCharacteristic(connId, serviceId, isPrimary, charId, &pData[0], pData.GetLength(), authType, withResponse));
}

nn::Result BluetoothDriverImpl::LeClientWriteDescriptorImpl(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, const nn::sf::InArray<std::uint8_t>& pData, uint8_t authType) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientWriteDescriptor(connId, serviceId, isPrimary, charId, descrId, &pData[0], pData.GetLength(), authType));
}

nn::Result BluetoothDriverImpl::LeClientRegisterNotificationImpl(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientRegisterNotification(connId, serviceId, isPrimary, charId));
}

nn::Result BluetoothDriverImpl::LeClientDeregisterNotificationImpl(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId) NN_NOEXCEPT
{
    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientDeregisterNotification(connId, serviceId, isPrimary, charId));
}

bool BluetoothDriverImpl::GetIsManufacturingModeImpl() NN_NOEXCEPT
{
    nn::bluetooth::FirmwareDebugSettings::Initialize();

    return nn::bluetooth::FirmwareDebugSettings::IsSkipBoot();
}

nn::Result BluetoothUserImpl::RegisterBleEventImpl(
    nn::sf::Out<nn::sf::NativeHandle> pOutHandle,
    const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    nn::bluetooth::FirmwareDebugSettings::Initialize();
    if (nn::bluetooth::FirmwareDebugSettings::IsSkipBoot() == true)
    {
        auto handle = nn::os::GetReadableHandleOfSystemEvent(&g_DummySystemEventForBle);
        pOutHandle.Set(nn::sf::NativeHandle(handle, false));
    }
    else
    {
        auto handle = nn::os::GetReadableHandleOfSystemEvent(&g_SystemEventForBle);
        pOutHandle.Set(nn::sf::NativeHandle(handle, false));
    }

    NN_RESULT_SUCCESS;
}

nn::Result BluetoothUserImpl::LeClientReadCharacteristicImpl(
    const nn::applet::AppletResourceUserId& aruid,
    uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, uint8_t authType) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientReadCharacteristic(connId, serviceId, isPrimary, charId, authType));
}

nn::Result BluetoothUserImpl::LeClientReadDescriptorImpl(
    const nn::applet::AppletResourceUserId& aruid,
    uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, uint8_t authType) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientReadDescriptor(connId, serviceId, isPrimary, charId, descrId, authType));
}

nn::Result BluetoothUserImpl::LeClientWriteCharacteristicImpl(
    const nn::applet::AppletResourceUserId& aruid,
    uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, const nn::sf::InArray<std::uint8_t>& pData, uint8_t authType, bool withResponse) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientWriteCharacteristic(connId, serviceId, isPrimary, charId, &pData[0], pData.GetLength(), authType, withResponse));
}

nn::Result BluetoothUserImpl::LeClientWriteDescriptorImpl(
    const nn::applet::AppletResourceUserId& aruid,
    uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, const nn::sf::InArray<std::uint8_t>& pData, uint8_t authType) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientWriteDescriptor(connId, serviceId, isPrimary, charId, descrId, &pData[0], pData.GetLength(), authType));
}

nn::Result BluetoothUserImpl::LeClientRegisterNotificationImpl(
    const nn::applet::AppletResourceUserId& aruid,
    uint32_t connId, GattId serviceId, bool isPrimary, GattId charId) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientRegisterNotification(connId, serviceId, isPrimary, charId));
}

nn::Result BluetoothUserImpl::LeClientDeregisterNotificationImpl(
    const nn::applet::AppletResourceUserId& aruid,
    uint32_t connId, GattId serviceId, bool isPrimary, GattId charId) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeClientDeregisterNotification(connId, serviceId, isPrimary, charId));
}

nn::Result BluetoothUserImpl::SetLeResponseImpl(
    const nn::applet::AppletResourceUserId& aruid,
    uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid attrUuid, const nn::sf::InArray<std::uint8_t>& pData) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->SetLeResponse(serverIf, serviceUuid, attrUuid, &pData[0], pData.GetLength()));
}

nn::Result BluetoothUserImpl::LeSendIndicationImpl(
    const nn::applet::AppletResourceUserId& aruid,
    uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid charUuid, const nn::sf::InArray<std::uint8_t>& pData, bool confirm) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    //NN_SDK_LOG("[bluetooth] %s",__FUNCTION__);
    return ConvertLeResult(g_pBluetoothLowEnergyInterface->LeSendIndication(serverIf, serviceUuid, charUuid, pData.GetLength(), &pData[0], confirm));
}

nn::Result BluetoothUserImpl::GetLeEventInfoImpl(
    nn::sf::Out<std::uint32_t> pEventType, const nn::sf::OutArray<std::uint8_t>& outArray,
    const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    NN_UNUSED(aruid);

    NN_SDK_REQUIRES(outArray.GetLength() >= BUFFER_SIZE_OF_BLE_OUT);
    memcpy(&outArray[0], &g_BufferForBle[0], BUFFER_SIZE_OF_BLE_OUT);

    // Clear g_BufferForBle to indicate there is no operation result
    InfoFromLeGattOperationCallback *pLE = reinterpret_cast<InfoFromLeGattOperationCallback*>(&g_BufferForBle[0]);
    pLE->status = BluetoothLeStatus::BT_GATT_ERR_NO_RESULT;
    pLE->connId = BleInvalidConnectionHandle;
    pLE->operation = GattOperationType_Unknown;
    pLE->length = 0;
    memset(pLE->value, 0x00, sizeof(pLE->value));
    memset(&pLE->serviceUuid, 0x00, sizeof(GattAttributeUuid));
    memset(&pLE->charcteristicUuid, 0x00, sizeof(GattAttributeUuid));
    memset(&pLE->descriptorUuid, 0x00, sizeof(GattAttributeUuid));

    *pEventType = g_EventTypeForBle;
    nn::os::SignalEvent(&g_InternalEventForBle);
    NN_RESULT_SUCCESS;
}

}}  // namespace nn::bluetooth

