﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <nn/dd.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/applet/applet_FundamentalTypes.h>

#include <nn/bluetooth/bluetooth_Api.h>
#include <nn/bluetooth/bluetooth_Driver.h>

namespace nn { namespace bluetooth {

namespace {

//-----------------------------------------------------------------------------
// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
nn::sf::SharedPointer<IBluetoothDriver>  g_BluetoothDriver;

}   // namespace

//-----------------------------------------------------------------------------
//  Bluetoothドライバのサービスの使用を開始します。DFC 用です。
//
void InitializeBluetoothDriverByDfc() NN_NOEXCEPT
{
    /*
    NN_SDK_ASSERT(!g_BluetoothDriver);

    g_BluetoothDriver = CreateBluetoothDriverByDfc();

    auto result = g_BluetoothDriver->WaitServerReady();
    NN_SDK_ASSERT(result.IsSuccess());
    NN_UNUSED(result);
    */
}


//-----------------------------------------------------------------------------
//  Bluetoothドライバのサービスの使用を開始します。HIPC 用です。
//
void InitializeBluetoothDriver() NN_NOEXCEPT
{
    NN_SDK_ASSERT(!g_BluetoothDriver);

    g_BluetoothDriver = CreateBluetoothDriverByHipc();

    auto result = g_BluetoothDriver->WaitServerReady();
    NN_SDK_ASSERT(result.IsSuccess());
    NN_UNUSED(result);
}


//-----------------------------------------------------------------------------
//  Bluetoothドライバのサービスの使用を終了します。
//
void FinalizeBluetoothDriver() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_BluetoothDriver);
    g_BluetoothDriver = nullptr;
}

bool GetIsManufacturingMode() NN_NOEXCEPT
{
    return g_BluetoothDriver->GetIsManufacturingModeImpl();
}

nn::Result InitializeBluetooth(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    nn::sf::NativeHandle handle;
    NN_RESULT_DO(g_BluetoothDriver->InitializeBluetoothImpl(&handle));
    nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
    handle.Detach();
    NN_RESULT_SUCCESS;
}

nn::Result EnableBluetooth() NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->EnableBluetoothImpl();
    return result;
}

nn::Result DisableBluetooth() NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->DisableBluetoothImpl();
    return result;
}

nn::Result CleanupBluetooth() NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->CleanupBluetoothImpl();
    return result;
}

nn::Result GetAdapterProperties(AdapterProperty* pAdapterProperty) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAdapterProperty);
    auto result = g_BluetoothDriver->GetAdapterPropertiesImpl(pAdapterProperty);
    return result;
}

nn::Result GetAdapterProperty(BluetoothPropertyType propertyType, uint8_t value[] ,int16_t sizeOfValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(value);
    nn::sf::OutArray<uint8_t> outArray(&value[0], sizeOfValue);
    auto result = g_BluetoothDriver->GetAdapterPropertyImpl(propertyType, outArray);
    return result;
}

nn::Result SetAdapterProperty(BluetoothPropertyType propertyType, const uint8_t value[] ,int16_t sizeOfValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(value);
    nn::sf::InArray<uint8_t> inArray(&value[0], sizeOfValue);
    auto result = g_BluetoothDriver->SetAdapterPropertyImpl(propertyType, inArray);
    return result;
}

nn::Result StartDiscovery() NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->StartDiscoveryImpl();
    return result;
}

nn::Result CancelDiscovery() NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->CancelDiscoveryImpl();
    return result;
}

nn::Result CreateBond(const BluetoothAddress *pBluetoothAddress, BluetoothTransport transport) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    auto result = g_BluetoothDriver->CreateBondImpl(*pBluetoothAddress, transport);
    return result;
}

nn::Result RemoveBond(const BluetoothAddress *pBluetoothAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    auto result = g_BluetoothDriver->RemoveBondImpl(*pBluetoothAddress);
    return result;
}

nn::Result CancelBond(const BluetoothAddress *pBluetoothAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    auto result = g_BluetoothDriver->CancelBondImpl(*pBluetoothAddress);
    return result;
}

nn::Result PinReply(const BluetoothAddress *pBluetoothAddress, bool isAccept, const BluetoothPinCode *pPinCode, uint8_t pinLength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    NN_SDK_REQUIRES_NOT_NULL(pPinCode);
    uint8_t accept;
    if(isAccept)
    {
        accept=1;
    }
    else
    {
        accept=0;
    }
    auto result = g_BluetoothDriver->PinReplyImpl(*pBluetoothAddress, accept, pinLength, *pPinCode);
    return result;
}

nn::Result SspReply(const BluetoothAddress *pBluetoothAddress, BluetoothSspVariant variant, bool isAccept, uint32_t passkey) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    //[Todo]to care a sending enum.
    uint8_t accept;
    if(isAccept)
    {
        accept=1;
    }
    else
    {
        accept=0;
    }
    auto result = g_BluetoothDriver->SspReplyImpl(*pBluetoothAddress, static_cast<uint8_t>(variant), accept, passkey);
    return result;
}

nn::Result GetEventInfo(nn::bluetooth::EventType *pOutEventType, uint8_t outBuffer[], uint16_t sizeOfBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutEventType);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    uint32_t eventType;
    nn::sf::OutArray<uint8_t> outArray(&outBuffer[0], sizeOfBuffer);
    auto result = g_BluetoothDriver->GetEventInfoImpl(&eventType, outArray);
    *pOutEventType=static_cast<nn::bluetooth::EventType>(eventType);
    return result;
}

nn::Result InitializeHid(nn::os::SystemEventType* pSystemEvent, uint16_t interfaceVersion) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    nn::sf::NativeHandle handle;
    auto result = g_BluetoothDriver->InitializeHidImpl(&handle, interfaceVersion);
    if(result.IsSuccess())
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    return result;
}

nn::Result HidConnect(const BluetoothAddress *pBluetoothAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    auto result = g_BluetoothDriver->HidConnectImpl(*pBluetoothAddress);
    return result;
}

nn::Result HidDisconnect(const BluetoothAddress *pBluetoothAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    auto result = g_BluetoothDriver->HidDisconnectImpl(*pBluetoothAddress);
    return result;
}

nn::Result HidSendData(const BluetoothAddress *pBluetoothAddress, const HidData* pData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    NN_SDK_REQUIRES_NOT_NULL(pData);
    auto result = g_BluetoothDriver->HidSendDataByStructureImpl(*pBluetoothAddress, *pData);
    return result;
}

nn::Result HidSendData(const BluetoothAddress *pBluetoothAddress, const uint8_t buffer[], uint16_t sizeOfBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    nn::sf::InArray<uint8_t> inArray(&buffer[0], sizeOfBuffer);
    auto result = g_BluetoothDriver->HidSendDataImpl(*pBluetoothAddress, inArray);
    return result;
}

nn::Result HidSetReport(const BluetoothAddress *pBluetoothAddress, BluetoothHhReportType reportType, const HidData* pData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    NN_SDK_REQUIRES_NOT_NULL(pData);
    auto result = g_BluetoothDriver->HidSetReportImpl(*pBluetoothAddress, reportType, *pData);
    return result;
}

nn::Result HidGetReport(const BluetoothAddress *pBluetoothAddress, BluetoothHhReportType reportType, uint8_t reportId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    auto result = g_BluetoothDriver->HidGetReportImpl(*pBluetoothAddress, reportType, reportId);
    return result;
}

nn::Result HidWakeController(const BluetoothAddress *pBluetoothAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    auto result = g_BluetoothDriver->HidWakeControllerImpl(*pBluetoothAddress);
    return result;
}

nn::Result HidAddPairedDevice(const nn::settings::system::BluetoothDevicesSettings *pDevice) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDevice);
    auto result = g_BluetoothDriver->HidAddPairedDeviceImpl(*pDevice);
    return result;
}

nn::Result HidGetPairedDevice(const BluetoothAddress *pBluetoothAddress, nn::settings::system::BluetoothDevicesSettings *pDeviceOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    NN_SDK_REQUIRES_NOT_NULL(pDeviceOut);
    auto result = g_BluetoothDriver->HidGetPairedDeviceImpl(*pBluetoothAddress, pDeviceOut);
    return result;
}

nn::Result CleanupHid() NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->CleanupHidImpl();
    return result;
}

nn::Result HidGetEventInfo(nn::bluetooth::HidEventType *outEventType, uint8_t outBuffer[], uint16_t sizeOfBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outEventType);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    uint32_t eventType;
    nn::sf::OutArray<uint8_t> outArray(&outBuffer[0], sizeOfBuffer);
    auto result = g_BluetoothDriver->HidGetEventInfoImpl(&eventType, outArray);
    if(result.IsSuccess())
    {
        *outEventType=static_cast<nn::bluetooth::HidEventType>(eventType);
    }
    return result;
}

nn::Result RegisterHidReportEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    nn::sf::NativeHandle handle;
    auto result = g_BluetoothDriver->RegisterHidReportEventImpl(&handle);
    if(result.IsSuccess())
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    return result;
}

nn::Result HidGetReportEventInfo(nn::bluetooth::HidEventType *outEventType, uint8_t outBuffer[], uint16_t sizeOfBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outEventType);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    uint32_t eventType;
    nn::sf::OutArray<uint8_t> outArray(&outBuffer[0], sizeOfBuffer);
    auto result = g_BluetoothDriver->HidGetReportEventInfoImpl(&eventType, outArray);
    if(result.IsSuccess())
    {
        *outEventType=static_cast<nn::bluetooth::HidEventType>(eventType);
    }
    return result;
}

nn::Result ExtSetTsi(const BluetoothAddress *pBluetoothAddress, uint8_t tsiNumber) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    auto result = g_BluetoothDriver->ExtSetTsiImpl(*pBluetoothAddress, tsiNumber);
    return result;
}

nn::Result ExtSetBurstMode(const BluetoothAddress *pBluetoothAddress, bool isBurst) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    auto result = g_BluetoothDriver->ExtSetBurstModeImpl(*pBluetoothAddress, isBurst);
    return result;
}

nn::Result ExtSetZeroRetran(const BluetoothAddress *pBluetoothAddress, uint8_t numReportIds, uint8_t *pReportIds) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    NN_SDK_REQUIRES_NOT_NULL(pReportIds);
    nn::sf::InArray<uint8_t> inArray(&pReportIds[0], numReportIds);
    auto result = g_BluetoothDriver->ExtSetZeroRetranImpl(*pBluetoothAddress, inArray);
    return result;
}

nn::Result ExtSetMcMode(bool isMcModeEnabled) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->ExtSetMcModeImpl(isMcModeEnabled);
    return result;
}

nn::Result ExtStartLlrMode() NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->ExtStartLlrModeImpl();
    return result;
}

nn::Result ExtExitLlrMode() NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->ExtExitLlrModeImpl();
    return result;
}

nn::Result ExtSetRadio(bool enable) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->ExtSetRadioImpl(enable);
    return result;
}

nn::Result ExtSetVisibility(bool discoverable, bool connectable) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->ExtSetVisibilityImpl(discoverable, connectable);
    return result;
}

nn::Result ExtSetTbfcScan(bool enable) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->ExtSetTbfcScanImpl(enable);
    return result;
}

nn::Result ExtGetPendingConnections() NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->ExtGetPendingConnectionsImpl();
    return result;
}

nn::Result GetLatestPlr(nn::bluetooth::PlrStatistics *outPlrStatistics) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outPlrStatistics);
    auto result = g_BluetoothDriver->GetLatestPlrImpl(outPlrStatistics);
    return result;
}

nn::Result GetChannelMap(ChannelMap* pOutChannelMap) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutChannelMap);
    auto result = g_BluetoothDriver->GetChannelMapImpl(pOutChannelMap);
    return result;
}

void EnableBluetoothBoostSetting(bool enable) NN_NOEXCEPT
{
    g_BluetoothDriver->EnableBluetoothBoostSettingImpl(enable);
}

bool IsBluetoothBoostSettingEnabled() NN_NOEXCEPT
{
    bool enabled = true;

    g_BluetoothDriver->IsBluetoothBoostEnabledImpl(&enabled);

    return enabled;
}

void EnableBluetoothAfhSetting(bool enable) NN_NOEXCEPT
{
    g_BluetoothDriver->EnableBluetoothAfhSettingImpl(enable);
}

bool IsBluetoothAfhSettingEnabled() NN_NOEXCEPT
{
    bool enabled = true;

    g_BluetoothDriver->IsBluetoothAfhEnabledImpl(&enabled);

    return enabled;
}

nn::Result InitializeBluetoothLe(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    nn::sf::NativeHandle handle;
    auto result = g_BluetoothDriver->InitializeBleImpl(&handle);
    if (result.IsSuccess())
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    return result;
}

nn::Result CleanupBluetoothLe() NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->CleanupBleImpl();
    return result;
}


nn::Result EnableBluetoothLe() NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->EnableBleImpl();
    return result;
}

nn::Result DisableBluetoothLe() NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->DisableBleImpl();
    return result;
}

nn::Result SetLeVisibility(bool discoverable, bool connectable) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->SetLeVisibilityImpl(discoverable, connectable);
    return result;
}

nn::Result SetLeConnectionParameter(LeConnectionParams connectionParams) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->SetLeConnectionParameterImpl(connectionParams);
    return result;
}

nn::Result SetLeDefaultConnectionParameter(LeConnectionParams connectionParams) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->SetLeDefaultConnectionParameterImpl(connectionParams);
    return result;
}

nn::Result SetLeAdvertiseData(LeAdvertiseData advertiseData) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->SetLeAdvertiseDataImpl(advertiseData);
    return result;
}

nn::Result SetLeAdvertiseParameter(const BluetoothAddress *pBluetoothAddress, uint16_t minAdvInt, uint16_t maxAdvInt) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    auto result = g_BluetoothDriver->SetLeAdvertiseParameterImpl(*pBluetoothAddress, minAdvInt, maxAdvInt);
    return result;
}

nn::Result StartLeScan() NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->StartLeScanImpl();
    return result;
}

nn::Result StopLeScan() NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->StopLeScanImpl();
    return result;
}

nn::Result SetLeScanParameter(uint16_t scanInterval, uint16_t scanWindow) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->SetLeScanParameterImpl(scanInterval, scanWindow);
    return result;
}

nn::Result AddLeScanFilterCondition(const BleAdvertiseFilter* pFilter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFilter);

    auto result = g_BluetoothDriver->AddLeScanFilterConditionImpl(*pFilter);
    return result;
}

nn::Result DeleteLeScanFilterCondition(const BleAdvertiseFilter* pFilter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pFilter);

    auto result = g_BluetoothDriver->DeleteLeScanFilterConditionImpl(*pFilter);
    return result;
}

nn::Result DeleteLeScanFilter(uint8_t filterIndex) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->DeleteLeScanFilterImpl(filterIndex);
    return result;
}

nn::Result ClearLeScanFilters() NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->ClearLeScanFiltersImpl();
    return result;
}

nn::Result EnableLeScanFilter(bool enable) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->EnableLeScanFilterImpl(enable);
    return result;
}

nn::Result RegisterLeClient(GattAttributeUuid uuid) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->RegisterLeClientImpl(uuid);
    return result;
}

nn::Result UnregisterLeClient(uint8_t clientIf) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->UnregisterLeClientImpl(clientIf);
    return result;
}

nn::Result UnregisterLeClientAll() NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->UnregisterLeClientAllImpl();
    return result;
}

nn::Result LeClientConnect(const nn::applet::AppletResourceUserId& aruid, uint8_t clientIf, const BluetoothAddress *pBluetoothAddress, bool isDirect) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    auto result = g_BluetoothDriver->LeClientConnectImpl(aruid, clientIf, *pBluetoothAddress, isDirect);
    return result;
}

nn::Result LeClientCancelConnection(uint8_t clientIf, const BluetoothAddress *pBluetoothAddress, bool isDirect) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    auto result = g_BluetoothDriver->LeClientCancelConnectionImpl(clientIf, *pBluetoothAddress, isDirect);
    return result;
}

nn::Result LeClientDisconnect(uint32_t connId) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->LeClientDisconnectImpl(connId);
    return result;
}

nn::Result LeClientGetAttributes(uint32_t connId, const BluetoothAddress *pBluetoothAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    auto result = g_BluetoothDriver->LeClientGetAttributesImpl(connId, *pBluetoothAddress);
    return result;
}

nn::Result LeClientDiscoverService(uint32_t connId, GattAttributeUuid filterUuid) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->LeClientDiscoverServiceImpl(connId, filterUuid);
    return result;
}

nn::Result LeClientConfigureMtu(uint32_t connId, uint16_t mtu) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->LeClientConfigureMtu(connId, mtu);
    return result;
}

nn::Result RegisterLeServer(GattAttributeUuid uuid) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->RegisterLeServerImpl(uuid);
    return result;
}

nn::Result UnregisterLeServer(uint8_t serverIf) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->UnregisterLeServerImpl(serverIf);
    return result;
}

nn::Result LeServerConnect(uint8_t serverIf, const BluetoothAddress *pBluetoothAddress, bool isDirect) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    auto result = g_BluetoothDriver->LeServerConnectImpl(serverIf, *pBluetoothAddress, isDirect);
    return result;
}

nn::Result LeServerDisconnect(uint8_t serverIf, const BluetoothAddress *pBluetoothAddress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pBluetoothAddress);
    auto result = g_BluetoothDriver->LeServerDisconnectImpl(serverIf, *pBluetoothAddress);
    return result;
}

nn::Result CreateLeService(uint8_t serverIf, GattAttributeUuid uuid, uint8_t handleNum, bool isPrimaryService) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->CreateLeServiceImpl(serverIf, uuid, handleNum, isPrimaryService);
    return result;
}

nn::Result StartLeService(uint8_t serverIf, GattAttributeUuid uuid) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->StartLeServiceImpl(serverIf, uuid);
    return result;
}

nn::Result AddLeCharacteristic(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid charUuid, uint16_t permissions, uint8_t properties) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->AddLeCharacteristicImpl(serverIf, serviceUuid, charUuid, permissions, properties);
    return result;
}

nn::Result AddLeDescriptor(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid charUuid, uint16_t permissions) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->AddLeDescriptorImpl(serverIf, serviceUuid, charUuid, permissions);
    return result;
}

nn::Result GetLeCoreEventInfo(nn::bluetooth::BleEventType *pOutEventType, uint8_t outBuffer[], uint16_t sizeOfBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutEventType);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    uint32_t eventType;
    nn::sf::OutArray<uint8_t> outArray(&outBuffer[0], sizeOfBuffer);
    auto result = g_BluetoothDriver->GetLeCoreEventInfoImpl(&eventType, outArray);
    if (result.IsSuccess())
    {
        *pOutEventType = static_cast<nn::bluetooth::BleEventType>(eventType);
    }

    return result;
}

nn::Result LeGetFirstCharacteristic(GattId *charOut, uint8_t *property, uint32_t connId, GattId serviceId, bool isPrimary, GattAttributeUuid charFilter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(charOut);
    NN_SDK_REQUIRES_NOT_NULL(property);
    auto result = g_BluetoothDriver->LeGetFirstCharacteristicImpl(charOut, property, connId, serviceId, isPrimary, charFilter);
    return result;
}

nn::Result LeGetNextCharacteristic(GattId *charOut, uint8_t *property, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattAttributeUuid charFilter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(charOut);
    NN_SDK_REQUIRES_NOT_NULL(property);
    auto result = g_BluetoothDriver->LeGetNextCharacteristicImpl(charOut, property, connId, serviceId, isPrimary, charIn, charFilter);

    return result;
}

nn::Result LeGetFirstDescriptor(GattId *descOut, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattAttributeUuid descrFilter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(descOut);
    auto result = g_BluetoothDriver->LeGetFirstDescriptorImpl(descOut, connId, serviceId, isPrimary, charIn, descrFilter);
    return result;
}

nn::Result LeGetNextDescriptor(GattId *descOut, uint32_t connId, GattId serviceId, bool isPrimary, GattId charIn, GattId descrIn, GattAttributeUuid descrFilter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(descOut);
    auto result = g_BluetoothDriver->LeGetNextDescriptorImpl(descOut, connId, serviceId, isPrimary, charIn, descrIn, descrFilter);

    return result;
}

nn::Result RegisterLeCoreDataPath(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->RegisterLeCoreDataPathImpl(uuid);
    return result;
}

nn::Result UnregisterLeCoreDataPath(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->UnregisterLeCoreDataPathImpl(uuid);
    return result;
}

nn::Result RegisterLeHidDataPath(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->RegisterLeHidDataPathImpl(uuid);
    return result;
}

nn::Result UnregisterLeHidDataPath(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->UnregisterLeHidDataPathImpl(uuid);
    return result;
}

nn::Result RegisterLeDataPath(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->RegisterLeDataPathImpl(uuid);
    return result;
}

nn::Result UnregisterLeDataPath(const nn::bluetooth::GattAttributeUuid& uuid) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->UnregisterLeDataPathImpl(uuid);
    return result;
}

nn::Result RegisterBleHidEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    nn::sf::NativeHandle handle;
    auto result = g_BluetoothDriver->RegisterBleHidEventImpl(&handle);
    if (result.IsSuccess())
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    return result;
}

nn::Result GetLeHidEventInfo(nn::bluetooth::BleEventType *pOutEventType, uint8_t outBuffer[], uint16_t sizeOfBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutEventType);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    uint32_t eventType;
    nn::sf::OutArray<uint8_t> outArray(&outBuffer[0], sizeOfBuffer);
    auto result = g_BluetoothDriver->GetLeHidEventInfoImpl(&eventType, outArray);
    if (result.IsSuccess())
    {
        *pOutEventType = static_cast<nn::bluetooth::BleEventType>(eventType);
    }

    return result;
}

nn::Result LeClientReadCharacteristic(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, uint8_t authType) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->LeClientReadCharacteristicImpl(connId, serviceId, isPrimary, charId, authType);
    return result;
}

nn::Result LeClientReadDescriptor(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, uint8_t authType) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->LeClientReadDescriptorImpl(connId, serviceId, isPrimary, charId, descrId, authType);
    return result;
}

nn::Result LeClientWriteCharacteristic(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, const uint8_t* pData, uint16_t len, uint8_t authType, bool withResponse) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);
    nn::sf::InArray<uint8_t> inArray(&pData[0], len);
    auto result = g_BluetoothDriver->LeClientWriteCharacteristicImpl(connId, serviceId, isPrimary, charId, inArray, authType, withResponse);
    return result;
}

nn::Result LeClientWriteDescriptor(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, const uint8_t* pData, uint16_t len, uint8_t authType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);
    nn::sf::InArray<uint8_t> inArray(&pData[0], len);
    auto result = g_BluetoothDriver->LeClientWriteDescriptorImpl(connId, serviceId, isPrimary, charId, descrId, inArray, authType);
    return result;
}

nn::Result LeClientRegisterNotification(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->LeClientRegisterNotificationImpl(connId, serviceId, isPrimary, charId);
    return result;
}

nn::Result LeClientDeregisterNotification(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId) NN_NOEXCEPT
{
    auto result = g_BluetoothDriver->LeClientDeregisterNotificationImpl(connId, serviceId, isPrimary, charId);
    return result;
}
}} // namespace nn::bluetooth

