﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_ContextControl.h>

#include "bgtc_Log.h"
#include "bgtc_Instance.h"
#include "bgtc_CompositeEventHandlerThread.h"
#include "bgtc_TaskServiceImpl.h"
#include "bgtc_BatterySaver.h"

namespace nn { namespace bgtc {

namespace {
Result CheckPowerStateProcessingState()
{
    // PmState_SleepReady の通知を受けているということは依存の追加が漏れている
    if (g_Instance.GetCompositeEventHandlerThread().GetLastPowerState() == psc::PmState_SleepReady)
    {
        NN_BGTC_WARN("Calling ScheduleTask is too late. Please add psc::PmModuleId_Bgtc to dependencies of PmModule.\n");
        return ResultTooLate();
    }
    return ResultSuccess();
}
}

TaskServiceImpl::TaskServiceImpl()
    : m_ClientEntry(m_EventNodeForTrigger, m_SystemEventForSchedule)
    , m_EventNodeForTrigger(os::EventClearMode_AutoClear, true)
    , m_SystemEventForSchedule(os::EventClearMode_ManualClear, true)

{
    g_Instance.GetClientBroker().Register(m_ClientEntry);
    g_Instance.GetCompositeEventHandlerThread().AttachTaskRunnableEvent(&m_EventNodeForTrigger);
}

TaskServiceImpl::~TaskServiceImpl()
{
    g_Instance.GetCompositeEventHandlerThread().DetachTaskRunnableEvent(&m_EventNodeForTrigger);
    g_Instance.GetClientBroker().Unregister(m_ClientEntry);
}

Result TaskServiceImpl::NotifyTaskStarting() NN_NOEXCEPT
{
    NN_BGTC_TRACE("NotifyTaskStarting\n");
/*
    Result result = g_Instance.GetCompositeEventHandlerThread().CheckTaskProcessable();
    if (result.IsFailure())
    {
        NN_BGTC_WARN("Current SystemPowerState does not support running tasks.");
        return result;
    }
*/
    g_Instance.GetClientBroker().SetTaskProcessing(m_ClientEntry, true);
    g_Instance.GetCompositeEventHandlerThread().TriggerEvent(CompositeEventHandlerThread::EventCode_ChangedClientTaskState);
    return ResultSuccess();
}

Result TaskServiceImpl::NotifyTaskFinished() NN_NOEXCEPT
{
    NN_BGTC_TRACE("NotifyTaskFinished\n");

    g_Instance.GetClientBroker().SetTaskProcessing(m_ClientEntry, false);
    g_Instance.GetCompositeEventHandlerThread().TriggerEvent(CompositeEventHandlerThread::EventCode_ChangedClientTaskState);
    return ResultSuccess();
}

nn::Result TaskServiceImpl::GetTriggerEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle) NN_NOEXCEPT
{
    outHandle.Set(nn::sf::NativeHandle(m_EventNodeForTrigger.GetReadableHandle(), false));
    return ResultSuccess();
}

nn::Result TaskServiceImpl::IsInHalfAwake(nn::sf::Out<bool> bResult) NN_NOEXCEPT
{
    bResult.Set(g_Instance.GetCompositeEventHandlerThread().IsHalfAwake());
    return ResultSuccess();
}

nn::Result TaskServiceImpl::IsInFullAwake(nn::sf::Out<bool> bResult) NN_NOEXCEPT
{
    bResult.Set(g_Instance.GetCompositeEventHandlerThread().IsFullAwake());
    return ResultSuccess();
}

nn::Result TaskServiceImpl::NotifyClientName(const nn::sf::InArray<char>& name) NN_NOEXCEPT
{
    g_Instance.GetClientBroker().SetClientName(m_ClientEntry, name.GetData(), name.GetLength());

    return CheckPowerStateProcessingState();
}

nn::Result TaskServiceImpl::ScheduleTask(std::int32_t intervalSeconds) NN_NOEXCEPT
{
    NN_BGTC_TRACE("ScheduleTask %d on PmState: %d\n", intervalSeconds, g_Instance.GetCompositeEventHandlerThread().GetLastPowerState());

    g_Instance.GetClientBroker().ScheduleTask(m_ClientEntry, intervalSeconds);

    return CheckPowerStateProcessingState();
}

nn::Result TaskServiceImpl::SchedulePeriodicTask(std::int32_t intervalSecondsFirst, std::int32_t intervalSecondsPeriodic) NN_NOEXCEPT
{
    NN_BGTC_TRACE("SchedulePeriodicTask %d, %d on PmState: %d\n", intervalSecondsFirst, intervalSecondsPeriodic, g_Instance.GetCompositeEventHandlerThread().GetLastPowerState());

    g_Instance.GetClientBroker().SchedulePeriodicTask(m_ClientEntry, intervalSecondsFirst, intervalSecondsPeriodic);

    return CheckPowerStateProcessingState();
}

nn::Result TaskServiceImpl::GetScheduledTaskInterval(nn::sf::Out<std::int32_t> outSeconds) NN_NOEXCEPT
{
    if (!g_Instance.GetClientBroker().IsTaskScheduled(m_ClientEntry))
    {
        return ResultNotScheduled();
    }

    outSeconds.Set(g_Instance.GetClientBroker().GetScheduledTaskInterval(m_ClientEntry));
    return ResultSuccess();
}

Result TaskServiceImpl::UnscheduleTask() NN_NOEXCEPT
{
    NN_BGTC_TRACE("UnscheduleTask %d on PmState\n", g_Instance.GetCompositeEventHandlerThread().GetLastPowerState());

    g_Instance.GetClientBroker().UnscheduleTask(m_ClientEntry);

    return ResultSuccess();
}

nn::Result TaskServiceImpl::GetScheduleEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle) NN_NOEXCEPT
{
    outHandle.Set(nn::sf::NativeHandle(m_SystemEventForSchedule.GetReadableHandle(), false));
    return ResultSuccess();
}

nn::Result TaskServiceImpl::GetOperationMode(nn::sf::Out<nn::bgtc::OperationMode> outOperationMode) NN_NOEXCEPT
{
    outOperationMode.Set(g_Instance.GetCompositeEventHandlerThread().JudgeOperationMode());
    return ResultSuccess();
}

nn::Result TaskServiceImpl::WillDisconnectNetworkWhenEnteringSleep(nn::sf::Out<bool> bResult)
{
    bResult.Set(g_Instance.GetCompositeEventHandlerThread().WillDisconnectNetworkWhenEnteringSleep());
    return ResultSuccess();
}

nn::Result TaskServiceImpl::WillStayHalfAwakeInsteadSleep(nn::sf::Out<bool> bResult)
{
    bResult.Set(g_Instance.GetCompositeEventHandlerThread().WillStayHalfAwakeInsteadSleep());
    return ResultSuccess();
}

}}
