﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/psc.h>
#include <nn/psm/psm_SystemApi.h>
#include <mutex>

#include "bgtc_BackgroundStateEventHandler.h"
#include "bgtc_Log.h"
#include "bgtc_Instance.h"
#include "bgtc_ClientBroker.h"
#include "bgtc_Settings.h"
#include "bgtc_BatterySaver.h"

namespace nn{ namespace bgtc{

BackgroundStateEventHandler::BackgroundStateEventHandler(BatterySaver& bs)
    : m_TimerEvent(os::EventClearMode_ManualClear)
    , m_Mutex(true)
    , m_LeavingHalfAwakeMargin(LeavingHalfAwakeMarginDefault)
    , m_BatterySaver(bs)
    , m_State(State_Idle)
{
    int32_t value;
    if (TryGetSettingsFwdbgValue<int32_t>(&value, "leaving_halfawake_margin"))
    {
        m_LeavingHalfAwakeMargin = TimeSpan::FromSeconds(value);
    }
}

BackgroundStateEventHandler::~BackgroundStateEventHandler()
{
}

void BackgroundStateEventHandler::TriggerEvent(EventCode eventCode)
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    NN_BGTC_TRACE("Caught event %s on %s\n", GetEventCodeString(eventCode), GetStateString(m_State));
    switch (m_State)
    {
    case State_Idle:
        switch(eventCode)
        {
        case EventCode_EnteringHalfAwake:
            SetState(State_Waiting);
            return;

        case EventCode_LeavingHalfAwake:
            return;

        case EventCode_EteringSleep:
            return;

        case EventCode_ChangedClientTaskState:
            if (IsProcessingClientExists())
            {
                SetState(State_Processing);
            }
            return;
        case EventCode_StopByBatterySaver:
            return;

        case EventCode_ChangedDeviceState:
            if (m_BatterySaver.ShouldCareAboutSleepUnsupportedDevices())
            {
                SetState(State_Waiting);
            }
            break;

        default:
            break;
        }
        break;

    case State_Processing:
        switch(eventCode)
        {
        case EventCode_EnteringHalfAwake:
            return;

        case EventCode_LeavingHalfAwake:
            return;

        case EventCode_EteringSleep:
            SetState(State_Idle);
            return;

        case EventCode_ChangedClientTaskState:
            if (!IsProcessingClientExists())
            {
                NN_BGTC_INFO("Waiting %d seconds for possibility of new tasks.\n", m_LeavingHalfAwakeMargin.GetSeconds());
                SetState(State_Waiting);
            }
            return;

        case EventCode_StopByBatterySaver:
            SetState(State_Idle);
            return;

        case EventCode_ChangedDeviceState:
            break;

        default:
            break;
        }
        break;

    case State_Waiting:
        switch (eventCode)
        {
        case EventCode_EnteringHalfAwake:
            SetState(State_Processing);
            return;

        case EventCode_LeavingHalfAwake:
            return;

        case EventCode_EteringSleep:
            SetState(State_Idle);
            return;

        case EventCode_ChangedClientTaskState:
            if (IsProcessingClientExists())
            {
                SetState(State_Processing);
            }
            return;

        case EventCode_SignaledTimer:
            if (!m_BatterySaver.ShouldCareAboutSleepUnsupportedDevices())
            {
                SetState(State_Idle);
            }
            else
            {
                NN_BGTC_INFO("Sleep unsupported devices(eg. USB-Ether) are exist. Keeping state as State_Waiting.\n");
            }
            return;

        case EventCode_StopByBatterySaver:
            SetState(State_Idle);
            return;

        case EventCode_ChangedDeviceState:
            if (!m_BatterySaver.ShouldCareAboutSleepUnsupportedDevices())
            {
                StartTimer();
            }
            return;

        default:
            break;
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
    NN_BGTC_WARN("Unhandled event %s on %s\n", GetEventCodeString(eventCode), GetStateString(m_State));
} // NOLINT[impl/function_size]

os::TimerEvent& BackgroundStateEventHandler::GetTimerEvent()
{
    return m_TimerEvent;
}

void BackgroundStateEventHandler::StartTimer()
{
    m_TimerEvent.StartOneShot(m_LeavingHalfAwakeMargin);
}

void BackgroundStateEventHandler::StopTimer()
{
    m_TimerEvent.Stop();
}

bool BackgroundStateEventHandler::IsProcessingClientExists() const
{
    return g_Instance.GetClientBroker().IsProcessingClientExists();
}

void nn::bgtc::BackgroundStateEventHandler::SetState(State stateTo)
{
    switch (stateTo)
    {
    case State_Idle:
        g_Instance.GetClientBroker().ResetTaskProcessingState();
        StopTimer();
        break;

    case State_Processing:
        StopTimer();
        break;

    case State_Waiting:
        StartTimer();
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
    NN_BGTC_INFO("Changing state [%s] to [%s]\n", GetStateString(m_State), GetStateString(stateTo));
    m_State = stateTo;
    m_StateChangeEvent.Signal();
}

const char * BackgroundStateEventHandler::GetEventCodeString(EventCode eventCode)
{
    const char* ppStrings[] =
    {
        "EventCode_EnteringHalfAwake",
        "EventCode_LeavingHalfAwake",
        "EventCode_EteringSleep",
        "EventCode_ChangedClientTaskState",
        "EventCode_SignaledTimer",
        "EventCode_StopByBatterySaver",
        "EventCode_ChangedDeviceState",
    };
    int index = static_cast<int>(eventCode);
    const int count = sizeof(ppStrings) / sizeof(ppStrings[0]);
    if (index < 0 || index >= count)
    {
        return "unknown";
    }
    return ppStrings[index];
}

const char * BackgroundStateEventHandler::GetStateString(State state)
{
    const char* ppStrings[] =
    {
        "State_Idle",
        "State_Processing",
        "State_Waiting",
    };
    int index = static_cast<int>(state);
    const int count = sizeof(ppStrings) / sizeof(ppStrings[0]);
    if (index < 0 || index >= count)
    {
        return "unknown";
    }
    return ppStrings[index];
}

void BackgroundStateEventHandler::AttachStateChangeEvent(bgsu::MultiClientSystemEvent::Node * pEventNode)
{
    m_StateChangeEvent.Attach(pEventNode);
}

void BackgroundStateEventHandler::DetachStateChangeEvent(bgsu::MultiClientSystemEvent::Node * pEventNode)
{
    m_StateChangeEvent.Detach(pEventNode);
}

void BackgroundStateEventHandler::SetIsUsingSleepUnsupportedDevices(bool bIsUsingSleepUnsupportedDevices)
{
#if NN_BGTC_ENABLE_WOWLAN
    if (m_BatterySaver.SetIsUsingSleepUnsupportedDevices(bIsUsingSleepUnsupportedDevices))
    {
        TriggerEvent(EventCode_ChangedDeviceState);
    }
#else
    NN_UNUSED(bIsUsingSleepUnsupportedDevices);
#endif
}

bool BackgroundStateEventHandler::WillDisconnectNetworkWhenEnteringSleep() const
{
    return m_BatterySaver.IsUsingSleepUnsupportedDevices();
}

bool BackgroundStateEventHandler::WillStayHalfAwakeInsteadSleep() const
{
    return m_BatterySaver.ShouldCareAboutSleepUnsupportedDevices();
}

}
}
