﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/service/bcat_ArchiveDecryptionKeyGenerator.h>
#include <nn/bcat/detail/service/core/bcat_ProductionMode.h>
#include <nn/bcat/detail/bcat_Log.h>
#include <nn/crypto/crypto_PasswordBasedKeyGenerator.h>
#include <nn/util/util_FormatString.h>
#include <nn/nn_StaticAssert.h>

namespace nn { namespace bcat { namespace service {

namespace
{
#if defined (NN_BUILD_CONFIG_OS_HORIZON)

    // 一般公開ライブラリにソルトが入らないように注意する。
    const char* SaltForProd[ArchiveDecryptionKeyGenerator::SaltIndexMax + 1] =
    {
        // 第一世代
        "a3e20c5c1cd7b720",
        "7f4c637432c8d420",
        "188d087d92a0c087",
        "8e7d23fa7fafe60f",
        "5252ae57c026d3cb",
        "2650f5e53554f01d",
        "b213a1e986307c9f",
        "875d8b01e3df5d7c",
        "c1b9a5ce866e00b1",
        "6a48ae69161e0138",
        "3f7b0401928b1f46",
        "0e9db55903a10f0e",
        "a8914bcbe7b888f9",
        "b15ef3ed6ce0e4cc",
        "f3b9d9f43dedf569",
        "bda4f7a0508c7462",
        // 第二世代
        "f5dc3586b1b2a8af",
        "7f6828b6f33dd118",
        "860de88547dcbf70",
        "ccbacacb70d11fb5",
        "b1475e5ea18151b9",
        "5f857ca15cf3374c",
        "cfa747c1d09d4f05",
        "30e7d70cb6f98101",
        "c8b3c78772bdcf43",
        "533dfc0702ed9874",
        "a29301cac5219e5c",
        "5776f5bec1b0df06",
        "1d4ab85a07ac4251",
        "7c1bd512b1cf5092",
        "2691cb8b3f76b411",
        "4400abee651c9eb9"
    };

#else

    // bcatSrv.lib(static link) をリリースすることになった場合に備えて、Windows 版は開発用のものと同じにしておく。
    const char* SaltForProd[ArchiveDecryptionKeyGenerator::SaltIndexMax + 1] =
    {
        // 第一世代
        "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "",
        // 第二世代
        "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", ""
    };

#endif

    const char* SaltForDev[ArchiveDecryptionKeyGenerator::SaltIndexMax + 1] =
    {
        // 第一世代
        "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "",
        // 第二世代
        "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", ""
    };
}

void ArchiveDecryptionKeyGenerator::Generate(ArchiveDecryptionKey* outKey,
    const char* passphrase, const nn::ApplicationId& appId, int saltIndex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(saltIndex >= SaltIndexMin && saltIndex <= SaltIndexMax);

    char salt[48] = {};

    if (detail::service::core::ProductionMode::IsProduction())
    {
        // NN_DETAIL_BCAT_INFO("[bcat] Generate archive decryption key (production).\n");
        nn::util::SNPrintf(salt, sizeof (salt), "%016llx%s", appId.value, SaltForProd[saltIndex]);
    }
    else
    {
        // NN_DETAIL_BCAT_INFO("[bcat] Generate archive decryption key (develop).\n");
        nn::util::SNPrintf(salt, sizeof (salt), "%016llx%s", appId.value, SaltForDev[saltIndex]);
    }

    Generate(outKey, passphrase, salt);
}

void ArchiveDecryptionKeyGenerator::Generate(ArchiveDecryptionKey* outKey,
    const char* passphrase, const char* salt) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outKey);
    NN_SDK_REQUIRES_NOT_NULL(passphrase);
    NN_SDK_REQUIRES_NOT_NULL(salt);

    nn::crypto::PasswordBasedKeyGenerator<nn::crypto::Sha256Generator> generator;

    generator.Initialize(reinterpret_cast<const nn::Bit8*>(passphrase), strlen(passphrase),
        reinterpret_cast<const nn::Bit8*>(salt), strlen(salt), 4096);

    generator.GetBytes(outKey->value, ArchiveKeySizeMax);
}

}}}
