﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/core/bcat_SessionManager.h>
#include <nn/bcat/detail/service/core/bcat_ExecutionQueue.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

SessionManager::SessionManager() NN_NOEXCEPT :
    m_Mutex(true)
{
    std::memset(m_Sessions, 0, sizeof (m_Sessions));
}

void SessionManager::Register(SessionId* outSessionId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outSessionId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    SessionId sessionId = IdIssuer::GetInstance().IssueSessionId();

    for (auto& session : m_Sessions)
    {
        if (session.id == InvalidSessionId)
        {
            session.id = sessionId;
            std::memset(session.tasks, 0, sizeof (session.tasks));

            *outSessionId = sessionId;
            return;
        }
    }

    NN_ABORT("SessionManager::Register failed.");
}

void SessionManager::Unregister(SessionId sessionId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(sessionId, InvalidSessionId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = Search(sessionId);

    if (index == -1)
    {
        return;
    }

    for (int i = 0; i < NN_ARRAY_SIZE(m_Sessions[index].tasks); i++)
    {
        if (m_Sessions[index].tasks[i].taskId != InvalidTaskId)
        {
            ExecutionQueue::GetInstance().Remove(m_Sessions[index].tasks[i].taskId);
        }
    }

    m_Sessions[index].id = InvalidSessionId;
}

nn::Result SessionManager::AddForegroundTask(TaskId* outTaskId, SessionId sessionId, nn::ApplicationId appId, uint32_t appVersion,
    const char* dirName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outTaskId);
    NN_SDK_REQUIRES_NOT_EQUAL(sessionId, InvalidSessionId);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(AddTask(outTaskId, sessionId, appId, appVersion, dirName, true));

    NN_RESULT_SUCCESS;
}

nn::Result SessionManager::AddBackgroundTask(TaskId* outTaskId, SessionId sessionId, nn::ApplicationId appId, uint32_t appVersion,
    const char* dirName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outTaskId);
    NN_SDK_REQUIRES_NOT_EQUAL(sessionId, InvalidSessionId);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_DO(AddTask(outTaskId, sessionId, appId, appVersion, dirName, false));

    NN_RESULT_SUCCESS;
}

void SessionManager::RemoveTask(SessionId sessionId, TaskId taskId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(sessionId, InvalidSessionId);
    NN_SDK_REQUIRES_NOT_EQUAL(taskId, InvalidTaskId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int sessionIndex = Search(sessionId);

    if (sessionIndex == -1)
    {
        return;
    }

    int taskIndex = SearchTask(sessionIndex, taskId);

    if (taskIndex == -1)
    {
        return;
    }

    m_Sessions[sessionIndex].tasks[taskIndex].taskId = InvalidTaskId;

    ExecutionQueue::GetInstance().Remove(taskId);
}

void SessionManager::CancelTask(SessionId sessionId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(sessionId, InvalidSessionId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int sessionIndex = Search(sessionId);

    if (sessionIndex == -1)
    {
        return;
    }

    for (int i = 0; i < NN_ARRAY_SIZE(m_Sessions[sessionIndex].tasks); i++)
    {
        if (m_Sessions[sessionIndex].tasks[i].taskId != InvalidTaskId)
        {
            ExecutionQueue::GetInstance().CancelAndRemove(m_Sessions[sessionIndex].tasks[i].taskId);

            m_Sessions[sessionIndex].tasks[i].taskId = InvalidTaskId;
        }
    }
}

int SessionManager::Search(SessionId sessionId) const NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_Sessions); i++)
    {
        if (sessionId == m_Sessions[i].id)
        {
            return i;
        }
    }

    return -1;
}

int SessionManager::SearchTask(int sessionIndex, TaskId taskId) const NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_Sessions[sessionIndex].tasks); i++)
    {
        if (taskId == m_Sessions[sessionIndex].tasks[i].taskId)
        {
            return i;
        }
    }

    return -1;
}

int SessionManager::SearchEmptyTask(int sessionIndex) const NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_Sessions[sessionIndex].tasks); i++)
    {
        if (m_Sessions[sessionIndex].tasks[i].taskId == InvalidTaskId)
        {
            return i;
        }
    }

    return -1;
}

nn::Result SessionManager::AddTask(TaskId* outTaskId, SessionId sessionId,
    nn::ApplicationId appId, uint32_t appVersion, const char* dirName, bool isForeground) NN_NOEXCEPT
{
    int sessionIndex = Search(sessionId);

    NN_SDK_ASSERT(sessionIndex != -1);

    int taskIndex = SearchEmptyTask(sessionIndex);

    if (taskIndex == -1)
    {
        taskIndex = SearchEmptyTask(sessionIndex);
    }

    NN_RESULT_THROW_UNLESS(taskIndex != -1, ResultTooManySyncRequests());

    if (isForeground)
    {
        NN_RESULT_THROW_UNLESS(ExecutionQueue::GetInstance().AddToHead(outTaskId, sessionId, appId, appVersion, dirName),
            ResultTaskLimitReached());
    }
    else
    {
        NN_RESULT_THROW_UNLESS(ExecutionQueue::GetInstance().AddToTail(outTaskId, sessionId, appId, appVersion, dirName),
            ResultTaskLimitReached());
    }

    Task task = {*outTaskId};

    m_Sessions[sessionIndex].tasks[taskIndex] = task;

    NN_RESULT_SUCCESS;
}

}}}}}
