﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/core/bcat_DeliveryCacheDownloader.h>
#include <nn/bcat/detail/service/core/bcat_DeliveryCacheProgressManager.h>
#include <nn/bcat/detail/service/core/bcat_DeliveryListDownloader.h>
#include <nn/bcat/detail/service/core/bcat_DeliveryListSubtractor.h>
#include <nn/bcat/detail/service/core/bcat_DirectoryDownloader.h>
#include <nn/bcat/detail/service/core/bcat_DirectorySynchronizer.h>
#include <nn/bcat/detail/service/core/bcat_PassphraseManager.h>
#include <nn/bcat/detail/service/msgpack/bcat_FileInputStream.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

namespace
{
    void MakeDownloadDirectoryPath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT
    {
        int length = nn::util::SNPrintf(path, size, "bcat-dl:/applications/%016llx/directories", appId.value);

        NN_UNUSED(length);
        NN_SDK_ASSERT(length < static_cast<int>(size));
    }

    void MakeDownloadDirectoryMetaPath(char* path, size_t size, nn::ApplicationId appId) NN_NOEXCEPT
    {
        int length = nn::util::SNPrintf(path, size, "bcat-dl:/applications/%016llx/directories/download.meta", appId.value);

        NN_UNUSED(length);
        NN_SDK_ASSERT(length < static_cast<int>(size));
    }
}

DeliveryCacheDownloader::DeliveryCacheDownloader() NN_NOEXCEPT :
    m_Mutex(true),
    m_TaskId(InvalidTaskId),
    m_AppId(nn::ApplicationId::GetInvalidId()),
    m_AppVersion(0),
    m_pDownloadCancelEvent(nullptr),
    m_pSyncCancelEvent(nullptr),
    m_TargetDirectory(nullptr),
    m_IncreasingDirCount(0),
    m_ReducingDirCount(0),
    m_IsDownloadDirectoryMetaExists(false),
    m_ResumeDownloadDirectoryIndex(-1),
    m_DownloadTargetDirectoryIndex(-1),
    m_Progress(),
    m_WorkBuffer(nullptr),
    m_WorkBufferSize(0)
{
    NN_STATIC_ASSERT(sizeof (DeliveryListDownloader) < sizeof (m_ObjectBuffer));
    NN_STATIC_ASSERT(sizeof (DeliveryListSubtractor) < sizeof (m_ObjectBuffer));
    NN_STATIC_ASSERT(sizeof (DirectoryDownloader) < sizeof (m_ObjectBuffer));
    NN_STATIC_ASSERT(sizeof (DirectorySynchronizer) < sizeof (m_ObjectBuffer));

    m_Passphrase[0] = '\0';
}

void DeliveryCacheDownloader::SetDownloadCancelEvent(nn::os::Event* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_pDownloadCancelEvent = pEvent;
}

void DeliveryCacheDownloader::SetSyncCancelEvent(nn::os::Event* pEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEvent);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_pSyncCancelEvent = pEvent;
}

void DeliveryCacheDownloader::SetTargetDirectory(const char* dirName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(dirName);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_TargetDirectory = dirName;
}

nn::Result DeliveryCacheDownloader::Download(TaskId taskId, nn::ApplicationId appId, uint32_t appVersion,
    bool ignoreNetworkUseRestriction, bool enableProviderSpecificDelivery, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(taskId, InvalidTaskId);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0u);

    NN_DETAIL_BCAT_INFO("[bcat] DeliveryCacheDownloader::Download(%016llx) ...\n", appId.value);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_TaskId = taskId;
    m_AppId = appId;
    m_AppVersion = appVersion;

    m_WorkBuffer = reinterpret_cast<Bit8*>(buffer);
    m_WorkBufferSize = bufferSize;

    m_Progress.Clear();

    NN_DETAIL_BCAT_DOWNLOAD_STORAGE_SCOPED_MOUNT();

    NN_RESULT_DO(HandleResult(DeliveryCacheStorageManager::GetInstance().Mount(appId)));

    NN_UTIL_SCOPE_EXIT
    {
        DeliveryCacheStorageManager::GetInstance().Unmount(appId);
    };

    NN_RESULT_DO(HandleResult(StepSetup()));

    NN_RESULT_DO(HandleResult(StepConnect(ignoreNetworkUseRestriction)));

    NN_RESULT_DO(HandleResult(StepProcessList(enableProviderSpecificDelivery)));

    if (m_TargetDirectory)
    {
        m_DownloadTargetDirectoryIndex = SearchDirectoryIndex(m_TargetDirectory);

        if (m_DownloadTargetDirectoryIndex == -1)
        {
            NN_RESULT_DO(HandleResult(ResultNotFound()));
        }

        if (m_DownloadTargetDirectoryIndex == m_ResumeDownloadDirectoryIndex)
        {
            NN_RESULT_DO(HandleResult(StepResumeDownloadDirectory()));
        }
        else
        {
            NN_RESULT_DO(HandleResult(StepDownloadTargetDirectory()));
        }
    }
    else
    {
        // 容量を確保するために、サイズが減る処理を先に実行すること。
        NN_RESULT_DO(HandleResult(StepRemoveDirectories()));

        if (m_ResumeDownloadDirectoryIndex != -1)
        {
            NN_RESULT_DO(HandleResult(StepResumeDownloadDirectory()));
        }

        NN_RESULT_DO(HandleResult(StepDownloadDirectories()));
    }

    m_Progress.NotifyDone(nn::ResultSuccess());
    DeliveryCacheProgressManager::GetInstance().SetProgress(m_TaskId, m_Progress);

    NN_RESULT_SUCCESS;
}

const DeliveryCacheProgressImpl& DeliveryCacheDownloader::GetProgress() const NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    return m_Progress;
}

nn::Result DeliveryCacheDownloader::StepSetup() NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DeliveryCacheDownloader::StepSetup ...\n");

    NN_RESULT_TRY(PassphraseManager::GetInstance().SaveTemporary(m_AppId))
        NN_RESULT_CATCH(ResultNotFound)
        {
        }
    NN_RESULT_END_TRY;

    NN_RESULT_DO(PassphraseManager::GetInstance().Get(m_Passphrase, sizeof (m_Passphrase), m_AppId));

    const char* root = "bcat-dl:/applications";

    char expectDir[64];
    nn::util::SNPrintf(expectDir, sizeof (expectDir), "%s/%016llx", root, m_AppId.value);

    const char* expectDirs[] = {expectDir};

    NN_RESULT_DO(CleanupDownloadDirectoryIfNoneMatch(root, expectDirs, NN_ARRAY_SIZE(expectDirs)));

    m_IsDownloadDirectoryMetaExists = LoadDownloadDirectoryMeta(&m_DownloadDirectoryMeta).IsSuccess();

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDownloader::StepConnect(bool ignoreNetworkUseRestriction) NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DeliveryCacheDownloader::StepConnect ...\n");

    m_Progress.NotifyStartConnect();
    DeliveryCacheProgressManager::GetInstance().SetProgress(m_TaskId, m_Progress);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::SetRequestRequirementPreset(m_Connection.GetRequestHandle(),
        ignoreNetworkUseRestriction ?
        nn::nifm::RequirementPreset_InternetForSystemProcessSharable : nn::nifm::RequirementPreset_InternetForSystemProcess));

    m_Connection.SubmitRequest();

    int index = -1;

    if (m_pDownloadCancelEvent)
    {
        index = nn::os::WaitAny(m_Connection.GetSystemEvent().GetBase(), m_pDownloadCancelEvent->GetBase());
    }
    else
    {
        index = nn::os::WaitAny(m_Connection.GetSystemEvent().GetBase());
    }

    NN_RESULT_THROW_UNLESS(index == 0, ResultCanceledByUser());

    NN_SDK_ASSERT(!m_Connection.IsRequestOnHold());

    NN_RESULT_THROW_UNLESS(m_Connection.IsAvailable(), ResultInternetRequestNotAccepted());

    NN_DETAIL_BCAT_INFO("[bcat] The network connection was established!\n");

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDownloader::StepProcessList(bool enableProviderSpecificDelivery) NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DeliveryCacheDownloader::StepProcessList ...\n");

    m_Progress.NotifyStartProcessList();
    DeliveryCacheProgressManager::GetInstance().SetProgress(m_TaskId, m_Progress);

    // ダウンロード処理
    {
        DeliveryListDownloader* pDownloader = new (m_ObjectBuffer) DeliveryListDownloader();

        NN_UTIL_SCOPE_EXIT
        {
            pDownloader->~DeliveryListDownloader();
        };

        if (m_pDownloadCancelEvent)
        {
            pDownloader->SetCancelEvent(m_pDownloadCancelEvent);
        }

        NN_RESULT_DO(pDownloader->Download(m_AppId, m_Passphrase, &m_Connection,
            enableProviderSpecificDelivery, m_WorkBuffer, m_WorkBufferSize));
    }

    // 差分検出処理
    {
        DeliveryListSubtractor* pSubtractor = new (m_ObjectBuffer) DeliveryListSubtractor();

        NN_UTIL_SCOPE_EXIT
        {
            pSubtractor->~DeliveryListSubtractor();
        };

        if (m_TargetDirectory)
        {
            pSubtractor->SetTargetDirectory(m_TargetDirectory);
        }

        NN_RESULT_DO(pSubtractor->Analyze(m_AppId, m_AppVersion, m_WorkBuffer, m_WorkBufferSize));

        m_Progress.SetWholeDownloadSize(pSubtractor->GetWholeDownloadSize());
        DeliveryCacheProgressManager::GetInstance().SetProgress(m_TaskId, m_Progress);

        pSubtractor->GetDirectoryView(&m_DirectoryMetaView, &m_DirectoryDiffView);

        pSubtractor->GetDirectoryList(&m_DirCount, m_Dirs, NN_ARRAY_SIZE(m_Dirs));

        pSubtractor->GetIncreasingDirectoryIndexList(&m_IncreasingDirCount, m_IncreasingDirIndexes, NN_ARRAY_SIZE(m_IncreasingDirIndexes));
        pSubtractor->GetReducingDirectoryIndexList(&m_ReducingDirCount, m_ReducingDirIndexes, NN_ARRAY_SIZE(m_ReducingDirIndexes));

        NN_SDK_ASSERT(m_IncreasingDirCount + m_ReducingDirCount <= m_DirCount);
    }

    // レジューム対象検知
    NN_RESULT_TRY(GetResumeDownloadDirectoryIndex(&m_ResumeDownloadDirectoryIndex))
        NN_RESULT_CATCH(ResultNotFound)
        {
            m_ResumeDownloadDirectoryIndex = -1;
        }
        NN_RESULT_CATCH(ResultInvalidState)
        {
            m_ResumeDownloadDirectoryIndex = -1;

            NN_DETAIL_BCAT_INFO("[bcat] Last download directory(%s) is updated.\n", m_DownloadDirectoryMeta.name.value);
        }
    NN_RESULT_END_TRY;

    if (m_ResumeDownloadDirectoryIndex != -1)
    {
        NN_DETAIL_BCAT_INFO("[bcat] Last download directory(%s) is resumable.\n", m_DownloadDirectoryMeta.name.value);
    }

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDownloader::StepRemoveDirectories() NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DeliveryCacheDownloader::StepRemoveDirectories ...\n");

    int indexes[DeliveryCacheDirectoryCountMax] = {};
    int count = 0;

    m_DirectoryDiffView.GetIndexListFromLocal(&count, indexes, NN_ARRAY_SIZE(indexes), DiffType_Remove);

    if (count == 0)
    {
        NN_DETAIL_BCAT_INFO("[bcat] Do nothing.\n");
        NN_RESULT_SUCCESS;
    }

    for (int i = 0; i < count; i++)
    {
        DirectoryMeta entry = {};
        NN_RESULT_DO(m_DirectoryMetaView.GetEntry(&entry, indexes[i]));

        NN_DETAIL_BCAT_INFO("[bcat] Remove directory(%s) ...\n", entry.name.value);

        char dirPath[160] = {};
        DeliveryCacheStorageManager::GetInstance().MakeDirectoryPath(dirPath, sizeof (dirPath), m_AppId, entry.name.value);

        NN_RESULT_TRY(FileSystem::DeleteDirectoryRecursively(dirPath))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
            }
        NN_RESULT_END_TRY;
    }

    // 後ろから削除することで、削除するインデックスがずれることを回避する。
    for (int i = count - 1; i >= 0; i--)
    {
        NN_RESULT_DO(m_DirectoryMetaView.Remove(indexes[i]));
    }

    NN_RESULT_DO(DirectoryMetaView::SaveToDeliveryCacheStorage(m_DirectoryMetaView, m_AppId, false));

    NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::Commit(NN_DETAIL_BCAT_DOWNLOAD_MOUNT_NAME));

    NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceledByUser());

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDownloader::StepResumeDownloadDirectory() NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DeliveryCacheDownloader::StepResumeDownloadDirectory ...\n");

    NN_SDK_ASSERT_NOT_EQUAL(m_ResumeDownloadDirectoryIndex, -1);

    NN_RESULT_DO(ResumeDownloadDirectory(m_Dirs[m_ResumeDownloadDirectoryIndex]));
    NN_RESULT_DO(SyncDirectory(m_Dirs[m_ResumeDownloadDirectoryIndex]));

    for (int i = 0; i < m_IncreasingDirCount; i++)
    {
        if (m_ResumeDownloadDirectoryIndex == m_IncreasingDirIndexes[i])
        {
            int moveCount = m_IncreasingDirCount - 1 - i;

            if (moveCount > 0)
            {
                std::memmove(&m_IncreasingDirIndexes[i], &m_IncreasingDirIndexes[i + 1], sizeof (int) * moveCount);
            }

            m_IncreasingDirCount--;
            NN_RESULT_SUCCESS;
        }
    }
    for (int i = 0; i < m_ReducingDirCount; i++)
    {
        if (m_ResumeDownloadDirectoryIndex == m_ReducingDirIndexes[i])
        {
            int moveCount = m_ReducingDirCount - 1 - i;

            if (moveCount > 0)
            {
                std::memmove(&m_ReducingDirIndexes[i], &m_ReducingDirIndexes[i + 1], sizeof (int) * moveCount);
            }

            m_ReducingDirCount--;
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDownloader::StepDownloadTargetDirectory() NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DeliveryCacheDownloader::StepDownloadTargetDirectory ...\n");

    NN_SDK_ASSERT_NOT_EQUAL(m_DownloadTargetDirectoryIndex, -1);

    NN_RESULT_DO(CleanupDownloadDirectory());
    NN_RESULT_DO(DownloadDirectory(m_Dirs[m_DownloadTargetDirectoryIndex]));
    NN_RESULT_DO(SyncDirectory(m_Dirs[m_DownloadTargetDirectoryIndex]));

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDownloader::StepDownloadDirectories() NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DeliveryCacheDownloader::StepDownloadDirectories ...\n");

    if (m_IncreasingDirCount == 0 && m_ReducingDirCount == 0)
    {
        NN_DETAIL_BCAT_INFO("[bcat] Do nothing.\n");
        NN_RESULT_SUCCESS;
    }

    for (int i = 0; i < m_ReducingDirCount; i++)
    {
        NN_RESULT_DO(CleanupDownloadDirectory());
        NN_RESULT_DO(DownloadDirectory(m_Dirs[m_ReducingDirIndexes[i]]));
        NN_RESULT_DO(SyncDirectory(m_Dirs[m_ReducingDirIndexes[i]]));
    }
    for (int i = 0; i < m_IncreasingDirCount; i++)
    {
        NN_RESULT_DO(CleanupDownloadDirectory());
        NN_RESULT_DO(DownloadDirectory(m_Dirs[m_IncreasingDirIndexes[i]]));
        NN_RESULT_DO(SyncDirectory(m_Dirs[m_IncreasingDirIndexes[i]]));
    }

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDownloader::ResumeDownloadDirectory(const ListDirectory& dir) NN_NOEXCEPT
{
    DirectoryDownloader* pDownloader = new (m_ObjectBuffer) DirectoryDownloader();

    NN_UTIL_SCOPE_EXIT
    {
        pDownloader->~DirectoryDownloader();
    };

    if (m_pDownloadCancelEvent)
    {
        pDownloader->SetCancelEvent(m_pDownloadCancelEvent);
    }

    pDownloader->SetLastProgressCallback(LastProgressCallback, this);
    pDownloader->SetStartDownloadFileCallback(StartDownloadFileCallback, this);
    pDownloader->SetDownloadFileCallback(DownloadFileCallback, this);

    NN_RESULT_DO(pDownloader->Download(m_AppId, dir.name, m_Passphrase, &m_Connection, m_WorkBuffer, m_WorkBufferSize));

    // ダウンロードが正常終了したら削除する。
    NN_RESULT_DO(DeleteDownloadDirectoryMetaFile());

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDownloader::DownloadDirectory(const ListDirectory& dir) NN_NOEXCEPT
{
    DirectoryMeta meta = {};

    meta.name = dir.name;
    meta.digest = dir.digest;

    NN_RESULT_DO(CreateDownloadDirectoryMetaFile(meta));

    DirectoryDownloader* pDownloader = new (m_ObjectBuffer) DirectoryDownloader();

    NN_UTIL_SCOPE_EXIT
    {
        pDownloader->~DirectoryDownloader();
    };

    if (m_pDownloadCancelEvent)
    {
        pDownloader->SetCancelEvent(m_pDownloadCancelEvent);
    }

    pDownloader->SetStartDownloadFileCallback(StartDownloadFileCallback, this);
    pDownloader->SetDownloadFileCallback(DownloadFileCallback, this);

    NN_RESULT_DO(pDownloader->Download(m_AppId, dir.name, m_Passphrase, &m_Connection, m_WorkBuffer, m_WorkBufferSize));

    // ダウンロードが正常終了したら削除する。
    NN_RESULT_DO(DeleteDownloadDirectoryMetaFile());

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDownloader::SyncDirectory(const ListDirectory& dir) NN_NOEXCEPT
{
    m_Progress.NotifyStartCommitDirectory(dir.name);
    DeliveryCacheProgressManager::GetInstance().SetProgress(m_TaskId, m_Progress);

    DirectorySynchronizer* pSynchronizer = new (m_ObjectBuffer) DirectorySynchronizer();

    NN_UTIL_SCOPE_EXIT
    {
        pSynchronizer->~DirectorySynchronizer();
    };

    if (m_pSyncCancelEvent)
    {
        pSynchronizer->SetCancelEvent(m_pSyncCancelEvent);
    }

    DirectoryMeta dirMeta = {};

    dirMeta.name = dir.name;
    dirMeta.digest = dir.digest;

    NN_RESULT_DO(pSynchronizer->Sync(m_AppId, dirMeta, m_WorkBuffer, m_WorkBufferSize));

    NN_RESULT_THROW_UNLESS(!IsCanceled(), ResultCanceledByUser());

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDownloader::LoadDownloadDirectoryMeta(DirectoryMeta* outMeta) NN_NOEXCEPT
{
    char path[80] = {};
    MakeDownloadDirectoryMetaPath(path, sizeof (path), m_AppId);

    nn::fs::FileHandle handle = {};

    bool deleteOnReturn = false;

    NN_RESULT_TRY(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_THROW(ResultNotFound());
        }
    NN_RESULT_END_TRY;

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);

        if (deleteOnReturn)
        {
            nn::fs::DeleteFile(path);
        }
    };

    NN_RESULT_TRY(nn::fs::ReadFile(handle, 0, outMeta, sizeof (DirectoryMeta)))
        NN_RESULT_CATCH_ALL
        {
            deleteOnReturn = true;
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDownloader::CreateDownloadDirectoryMetaFile(const DirectoryMeta& meta) NN_NOEXCEPT
{
    {
        char path[80] = {};
        MakeDownloadDirectoryMetaPath(path, sizeof (path), m_AppId);

        nn::fs::FileHandle handle = {};

        NN_RESULT_TRY(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                NN_RESULT_DO(FileSystem::CreateFile(path, sizeof (meta)));
                NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write));
            }
        NN_RESULT_END_TRY;

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_RESULT_DO(nn::fs::WriteFile(handle, 0, &meta, sizeof (meta),
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::Commit(NN_DETAIL_BCAT_DOWNLOAD_MOUNT_NAME));

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDownloader::DeleteDownloadDirectoryMetaFile() NN_NOEXCEPT
{
    char path[80] = {};
    MakeDownloadDirectoryMetaPath(path, sizeof (path), m_AppId);

    NN_RESULT_TRY(nn::fs::DeleteFile(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
            NN_RESULT_SUCCESS;
        }
    NN_RESULT_END_TRY;

    NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::Commit(NN_DETAIL_BCAT_DOWNLOAD_MOUNT_NAME));

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDownloader::CleanupDownloadDirectory() NN_NOEXCEPT
{
    char path[64] = {};
    MakeDownloadDirectoryPath(path, sizeof (path), m_AppId);

    NN_RESULT_TRY(FileSystem::DeleteDirectoryRecursively(path))
        NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
        {
        }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDownloader::CleanupDownloadDirectoryIfNoneMatch(const char* root, const char* expectDirs[], int expectDirCount) NN_NOEXCEPT
{
    {
        nn::fs::DirectoryHandle handle = {};

        NN_RESULT_TRY(nn::fs::OpenDirectory(&handle, root, nn::fs::OpenDirectoryMode_Directory))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory(handle);
        };

        int64_t count = 0;
        NN_RESULT_DO(nn::fs::GetDirectoryEntryCount(&count, handle));

        if (count == 0)
        {
            NN_RESULT_SUCCESS;
        }
        if (count == expectDirCount)
        {
            bool isMatch = true;

            for (int i = 0; i < count; i++)
            {
                if (!FileSystem::Exists(expectDirs[i]))
                {
                    isMatch = false;
                    break;
                }
            }
            if (isMatch)
            {
                NN_RESULT_SUCCESS;
            }
        }
    }

    NN_RESULT_DO(FileSystem::DeleteDirectoryRecursively("bcat-dl:/applications"));

    NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::Commit(NN_DETAIL_BCAT_DOWNLOAD_MOUNT_NAME));

    NN_DETAIL_BCAT_INFO("[bcat] Cleanup application download directory.\n");

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDownloader::GetResumeDownloadDirectoryIndex(int* outIndex) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_IsDownloadDirectoryMetaExists, ResultNotFound());

    int index = SearchDirectoryIndex(m_DownloadDirectoryMeta.name.value);

    NN_RESULT_THROW_UNLESS(index != -1, ResultNotFound());

    // ダイジェスト値が一致する場合、更新は発生していないためレジュームダウンロードが可能。
    if (std::memcmp(m_DownloadDirectoryMeta.digest.value, m_Dirs[index].digest.value, sizeof (m_Dirs[index].digest.value)) == 0)
    {
        // OK
    }
    else
    {
        NN_RESULT_THROW(ResultInvalidState());
    }

    *outIndex = index;

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDownloader::HandleResult(nn::Result result) NN_NOEXCEPT
{
    if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
    {
        result = ResultUsableSpaceNotEnough();
    }

    if (result.IsFailure())
    {
        m_Progress.NotifyDone(result);
        DeliveryCacheProgressManager::GetInstance().SetProgress(m_TaskId, m_Progress);
    }

    return result;
}

int DeliveryCacheDownloader::SearchDirectoryIndex(const char* dirName) NN_NOEXCEPT
{
    for (int i = 0; i < m_DirCount; i++)
    {
        if (nn::util::Strnicmp(dirName, m_Dirs[i].name.value, sizeof (m_Dirs[i].name.value)) == 0)
        {
            return i;
        }
    }

    return -1;
}

bool DeliveryCacheDownloader::IsCanceled() NN_NOEXCEPT
{
    return (m_pDownloadCancelEvent && m_pDownloadCancelEvent->TryWait());
}

void DeliveryCacheDownloader::LastProgressCallback(int64_t downloaded, const DirectoryName& dirName, void* param) NN_NOEXCEPT
{
    DeliveryCacheDownloader* pDownloader = reinterpret_cast<DeliveryCacheDownloader*>(param);

    pDownloader->m_Progress.SetDownloadProgress(downloaded, dirName);
    DeliveryCacheProgressManager::GetInstance().SetProgress(pDownloader->m_TaskId, pDownloader->m_Progress);
}

void DeliveryCacheDownloader::StartDownloadFileCallback(const DirectoryName& dirName, const FileName& fileName, int64_t size, void* param) NN_NOEXCEPT
{
    DeliveryCacheDownloader* pDownloader = reinterpret_cast<DeliveryCacheDownloader*>(param);

    pDownloader->m_Progress.NotifyStartDownloadFile(dirName, fileName, size);
    DeliveryCacheProgressManager::GetInstance().SetProgress(pDownloader->m_TaskId, pDownloader->m_Progress);
}

void DeliveryCacheDownloader::DownloadFileCallback(int64_t downloaded, void* param) NN_NOEXCEPT
{
    DeliveryCacheDownloader* pDownloader = reinterpret_cast<DeliveryCacheDownloader*>(param);

    if (pDownloader->m_Progress.UpdateDownloadFileProgress(downloaded))
    {
        DeliveryCacheProgressManager::GetInstance().SetProgress(pDownloader->m_TaskId, pDownloader->m_Progress);
    }
}

}}}}}
