﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/core/bcat_BackgroundWorkerThread.h>
#include <nn/bcat/detail/service/core/bcat_TaskManager.h>
#include <nn/bcat/detail/service/core/bcat_DeliveryCacheDownloader.h>
#include <nn/bcat/detail/service/core/bcat_DeliveryCacheProgressManager.h>
#include <nn/bcat/detail/service/core/bcat_DirectorySynchronizer.h>
#include <nn/bcat/detail/service/core/bcat_ExecutionQueue.h>
#include <nn/bcat/detail/service/core/bcat_PassphraseManager.h>
#include <nn/bcat/detail/service/core/bcat_SessionManager.h>
#include <nn/npns/npns_Api.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/bgtc.h>
#include <nn/bgtc/bgtc_Task.h>
#include <nn/fs/fs_PriorityPrivate.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

namespace
{
    const char* RunningTaskPath = "bcat-sys:/dc/running.app";

    const size_t ObjectBufferSize = 128 * 1024;

    const size_t WorkBufferSize = 512 * 1024;
}

namespace
{
    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK Bit8 g_ThreadStack[48 * 1024];

    nn::os::Event g_StopEvent(nn::os::EventClearMode_ManualClear);
}

namespace
{
    nn::os::SdkMutexType g_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();

    Bit64 g_ObjectBuffer[ObjectBufferSize / sizeof (Bit64)];

    NN_STATIC_ASSERT(sizeof (DeliveryCacheDownloader) < sizeof (g_ObjectBuffer));
    NN_STATIC_ASSERT(sizeof (DirectorySynchronizer) < sizeof (g_ObjectBuffer));

    Bit8 g_WorkBuffer[WorkBufferSize];

    SessionId g_BackgroundSessionId = InvalidSessionId;

    TaskId g_BackgroundTaskId = InvalidTaskId;

    nn::ApplicationId g_BackgroundTaskApplicationId = nn::ApplicationId::GetInvalidId();
    RevisionHash g_RevisionHash = {};

    nn::ApplicationId g_RunTaskBlockList[detail::ipc::SessionCountMax] = {};
    int g_RunTaskBlockListCount = 0;

    nn::os::Event g_NpnsConnectedEvent(nn::os::EventClearMode_ManualClear);

    ExecutionQueue::Item g_CurrentQueueItem = {};

    nn::os::Event g_DownloadCancelEvent(nn::os::EventClearMode_ManualClear);
    nn::os::Event g_SyncCancelEvent(nn::os::EventClearMode_ManualClear);

    nn::os::Event* g_pFinishEvent = nullptr;

    nn::os::Event g_SystemRunnableEvent(nn::os::EventClearMode_ManualClear);
    std::atomic<bool> g_IsSystemRunnable(true);
}

namespace
{
    void EnqueueBackgroundTaskIfEmpty() NN_NOEXCEPT
    {
        std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

        if (ExecutionQueue::GetInstance().IsEmpty())
        {
            TaskManager::Record record = {};

            if (TaskManager::GetInstance().GetRunnableTaskRecord(&record))
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(SessionManager::GetInstance().AddBackgroundTask(&g_BackgroundTaskId,
                    g_BackgroundSessionId, record.appId, record.appVersion));

                g_BackgroundTaskApplicationId = record.appId;
                g_RevisionHash = record.revisionHash;
            }
        }
    }

    nn::Result StartTask() NN_NOEXCEPT
    {
        std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

        if (!ExecutionQueue::GetInstance().Pop(&g_CurrentQueueItem))
        {
            NN_RESULT_THROW(ResultNotFound());
        }

        for (int i = 0; i < g_RunTaskBlockListCount; i++)
        {
            if (g_CurrentQueueItem.appId == g_RunTaskBlockList[i])
            {
                DeliveryCacheProgressImpl progress = {};

                progress.status = DeliveryCacheProgressStatusImpl_Done;
                progress.result = ParameterConverter::Convert(ResultRequestBlocked());

                DeliveryCacheProgressManager::GetInstance().SetProgress(g_CurrentQueueItem.taskId, progress);

                NN_RESULT_THROW(ResultRequestBlocked());
            }
        }

        g_DownloadCancelEvent.Clear();
        g_SyncCancelEvent.Clear();

        NN_RESULT_SUCCESS;
    }

    void FinishCurrentTask(nn::Result result) NN_NOEXCEPT
    {
        std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

        if (g_CurrentQueueItem.taskId == g_BackgroundTaskId)
        {
            TaskManager::GetInstance().NotifyDone(g_CurrentQueueItem.appId, result, g_RevisionHash);

            DeliveryCacheProgressManager::GetInstance().Unregister(g_BackgroundTaskId);

            SessionManager::GetInstance().RemoveTask(g_BackgroundSessionId, g_BackgroundTaskId);

            g_BackgroundTaskId = InvalidTaskId;
            g_BackgroundTaskApplicationId = nn::ApplicationId::GetInvalidId();
        }

        g_CurrentQueueItem.taskId = InvalidTaskId;
        g_CurrentQueueItem.appId = nn::ApplicationId::GetInvalidId();

        if (g_pFinishEvent)
        {
            g_pFinishEvent->Signal();
            g_pFinishEvent = nullptr;
        }
    }

    nn::Result CreateRunningTaskFile() NN_NOEXCEPT
    {
        NN_DETAIL_BCAT_SYSTEM_STORAGE_SCOPED_MOUNT();

        {
            nn::fs::FileHandle handle = {};

            NN_RESULT_TRY(nn::fs::OpenFile(&handle, RunningTaskPath, nn::fs::OpenMode_Write))
                NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::CreateFile(RunningTaskPath, sizeof (nn::ApplicationId)));
                    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, RunningTaskPath, nn::fs::OpenMode_Write));
                }
                NN_RESULT_CATCH_ALL
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
                }
            NN_RESULT_END_TRY;

            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handle);
            };

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::WriteFile(handle, 0, &g_CurrentQueueItem.appId, sizeof (nn::ApplicationId),
                nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::Commit(NN_DETAIL_BCAT_SYSTEM_MOUNT_NAME));

        NN_RESULT_SUCCESS;
    }

    nn::Result DeleteRunningTaskFile() NN_NOEXCEPT
    {
        NN_DETAIL_BCAT_SYSTEM_STORAGE_SCOPED_MOUNT();

        NN_RESULT_TRY(nn::fs::DeleteFile(RunningTaskPath))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
            }
            NN_RESULT_CATCH_ALL
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
            }
        NN_RESULT_END_TRY;

        NN_ABORT_UNLESS_RESULT_SUCCESS(FileSystem::Commit(NN_DETAIL_BCAT_SYSTEM_MOUNT_NAME));

        NN_RESULT_SUCCESS;
    }

    nn::Result GetLastRunningTask(nn::ApplicationId* outAppId) NN_NOEXCEPT
    {
        NN_DETAIL_BCAT_SYSTEM_STORAGE_SCOPED_MOUNT_READ_ONLY();

        nn::fs::FileHandle handle = {};

        NN_RESULT_TRY(nn::fs::OpenFile(&handle, RunningTaskPath, nn::fs::OpenMode_Read))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                NN_RESULT_THROW(ResultNotFound());
            }
            NN_RESULT_CATCH_ALL
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(NN_RESULT_CURRENT_RESULT);
            }
        NN_RESULT_END_TRY;

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, outAppId, sizeof (nn::ApplicationId)));

        NN_RESULT_SUCCESS;
    }

    nn::Result RecoverTransaction() NN_NOEXCEPT
    {
        nn::ApplicationId appId;

        NN_RESULT_TRY(GetLastRunningTask(&appId))
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY;

        NN_DETAIL_BCAT_INFO("[bcat] The '%s' was found. appId = %016llx\n", RunningTaskPath, appId.value);

        nn::Result result = DeliveryCacheStorageManager::GetInstance().Mount(appId);

        // アプリケーションがロックしていた場合、解放されるまで待つ。
        while (ResultLocked::Includes(result))
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));

            result = DeliveryCacheStorageManager::GetInstance().Mount(appId);
        }
        if (result.IsFailure())
        {
            NN_RESULT_DO(DeleteRunningTaskFile());
            return result;
        }

        NN_UTIL_SCOPE_EXIT
        {
            DeliveryCacheStorageManager::GetInstance().Unmount(appId);
        };

        if (DirectorySynchronizer::IsTransactionCreated(appId))
        {
            NN_DETAIL_BCAT_DOWNLOAD_STORAGE_SCOPED_MOUNT();

            DirectorySynchronizer* pSynchronizer = new (g_ObjectBuffer) DirectorySynchronizer();

            NN_UTIL_SCOPE_EXIT
            {
                pSynchronizer->~DirectorySynchronizer();
            };

            NN_ABORT_UNLESS_RESULT_SUCCESS(pSynchronizer->Recover(appId, g_WorkBuffer, sizeof (g_WorkBuffer)));
        }

        NN_RESULT_DO(DeleteRunningTaskFile());

        NN_RESULT_SUCCESS;
    }

    nn::Result Download() NN_NOEXCEPT
    {
        bool isBackground = (g_CurrentQueueItem.taskId == g_BackgroundTaskId);

        if (isBackground)
        {
            nn::fs::SetPriorityRawOnCurrentThread(nn::fs::PriorityRaw_Background);
        }

        NN_UTIL_SCOPE_EXIT
        {
            if (isBackground)
            {
                nn::fs::SetPriorityRawOnCurrentThread(nn::fs::PriorityRaw_Normal);
            }
        };

        CreateRunningTaskFile();

        NN_UTIL_SCOPE_EXIT
        {
            DeleteRunningTaskFile();
        };

        // アプリケーションからの同期要求の場合、ネットワークの利用制限は無視する。
        bool ignoreNetworkUseRestriction = !isBackground;
        // アプリケーションからの同期要求の場合、事業者別配信を有効にする。
        bool enableProviderSpecificDelivery = !isBackground;

        DeliveryCacheDownloader* pDownloader = new (g_ObjectBuffer) DeliveryCacheDownloader();

        NN_UTIL_SCOPE_EXIT
        {
            pDownloader->~DeliveryCacheDownloader();
        };

        pDownloader->SetDownloadCancelEvent(&g_DownloadCancelEvent);
        pDownloader->SetSyncCancelEvent(&g_SyncCancelEvent);

        if (g_CurrentQueueItem.dirName.value[0])
        {
            pDownloader->SetTargetDirectory(g_CurrentQueueItem.dirName.value);
        }

        nn::Result result = pDownloader->Download(g_CurrentQueueItem.taskId,
            g_CurrentQueueItem.appId, g_CurrentQueueItem.appVersion,
            ignoreNetworkUseRestriction, enableProviderSpecificDelivery, g_WorkBuffer, sizeof (g_WorkBuffer));

        if (result.IsFailure())
        {
            NN_DETAIL_BCAT_INFO("[bcat] DeliveryCacheDownloader::Download failed. code = %03d-%04d\n",
                result.GetModule(), result.GetDescription());
        }

        return result;
    }

    void ProcessRunTask() NN_NOEXCEPT
    {
        while (NN_STATIC_CONDITION(true))
        {
            nn::Result result = StartTask();

            if (result.IsSuccess())
            {
                NN_DETAIL_BCAT_INFO("[bcat] Task(%016llx) ...\n", g_CurrentQueueItem.appId.value);

                result = Download();

                NN_DETAIL_BCAT_INFO("[bcat] Task(%016llx) done!\n", g_CurrentQueueItem.appId.value);

                FinishCurrentTask(result);
            }
            else if (ResultRequestBlocked::Includes(result))
            {
                FinishCurrentTask(result);
            }
            else
            {
                NN_SDK_ASSERT(ResultNotFound::Includes(result));
                break;
            }
        }
    }

    void ProcessNpnsSubscription() NN_NOEXCEPT
    {
        TopicId topicId = {};

        while (TaskManager::GetInstance().GetNpnsTopicToBeUnsubscribed(&topicId))
        {
            nn::npns::State state = nn::npns::GetState();

            if (!(state == nn::npns::State_Connected || state == nn::npns::State_ConnectedOnHalfAwake))
            {
                break;
            }

            NN_DETAIL_BCAT_INFO("[bcat] nn::npns::UnsubscribeTopic(%s) ...\n", topicId.value);

            nn::Result result = nn::npns::UnsubscribeTopic(topicId.value);

            if (result.IsSuccess())
            {
                NN_DETAIL_BCAT_INFO("[bcat] nn::npns::UnsubscribeTopic(%s) ... done!\n", topicId.value);
            }
            else
            {
                NN_DETAIL_BCAT_INFO("[bcat] nn::npns::UnsubscribeTopic(%s) ... done! code = %03d-%04d\n",
                    topicId.value, result.GetModule(), result.GetDescription());

                if (nn::npns::ResultTopicNotSubscribed::Includes(result) || nn::npns::ResultTopicNotFound::Includes(result))
                {
                    // すでに未購読の状態だったら成功扱いとする。
                    result = nn::ResultSuccess();
                }
            }

            TaskManager::GetInstance().NotifyNpnsTopicUnsubscriptionProcessed(topicId, result);

            if (nn::npns::ResultCommandTimeOut::Includes(result))
            {
                // 以降の購読解除処理がタイムアウト待ちしないよう、速やかに終了する。
                return;
            }
        }

        // エラー状態も含めた (includesErrorStatus = true) 購読解除すべきトピックがすべて購読解除されるまでは購読処理は行わない。
        if (TaskManager::GetInstance().HasNpnsTopicToBeUnsubscribed(true))
        {
            return;
        }

        while (TaskManager::GetInstance().GetNpnsTopicToBeSubscribed(&topicId))
        {
            nn::npns::State state = nn::npns::GetState();

            if (!(state == nn::npns::State_Connected || state == nn::npns::State_ConnectedOnHalfAwake))
            {
                break;
            }

            NN_DETAIL_BCAT_INFO("[bcat] nn::npns::SubscribeTopic(%s) ...\n", topicId.value);

            nn::Result result = nn::npns::SubscribeTopic(topicId.value);

            if (result.IsSuccess())
            {
                NN_DETAIL_BCAT_INFO("[bcat] nn::npns::SubscribeTopic(%s) ... done!\n", topicId.value);
            }
            else
            {
                NN_DETAIL_BCAT_INFO("[bcat] nn::npns::SubscribeTopic(%s) ... done! code = %03d-%04d\n",
                    topicId.value, result.GetModule(), result.GetDescription());
            }

            TaskManager::GetInstance().NotifyNpnsTopicSubscriptionProcessed(topicId, result);

            if (nn::npns::ResultCommandTimeOut::Includes(result))
            {
                // 以降の購読処理がタイムアウト待ちしないよう、速やかに終了する。
                return;
            }
        }
    }

    void WorkerThread(void*) NN_NOEXCEPT
    {
        NN_DETAIL_BCAT_HANDLING_DATA_CORRUPTION_SCOPED_DECLARE();

        SessionManager::GetInstance().Register(&g_BackgroundSessionId);

        NN_UTIL_SCOPE_EXIT
        {
            SessionManager::GetInstance().Unregister(g_BackgroundSessionId);
        };

        if (detail::service::util::Account::IsNetworkServiceAccountAvailable())
        {
            TaskManager::GetInstance().NotifyNintendoAccountLinked();
        }

        RecoverTransaction();

        nn::bgtc::Task bgtcTask;
        NN_ABORT_UNLESS_RESULT_SUCCESS(bgtcTask.Initialize());

        NN_UTIL_SCOPE_EXIT
        {
            bgtcTask.Finalize();
        };

        while (NN_STATIC_CONDITION(true))
        {
            while (!g_IsSystemRunnable)
            {
                nn::os::WaitAny(g_StopEvent.GetBase(), g_SystemRunnableEvent.GetBase());

                if (g_StopEvent.TryWait())
                {
                    return;
                }

                g_SystemRunnableEvent.Clear();
            }

            nn::os::WaitAny(g_StopEvent.GetBase(),
                ExecutionQueue::GetInstance().GetEvent().GetBase(),
                TaskManager::GetInstance().GetScheduleEvent().GetBase(),
                TaskManager::GetInstance().GetSubscriptionEvent().GetBase(),
                g_NpnsConnectedEvent.GetBase());

            if (g_StopEvent.TryWait())
            {
                return;
            }
            if (!g_IsSystemRunnable)
            {
                continue;
            }

            nn::Result bgtcResult = bgtcTask.NotifyStarting();

            if (bgtcResult.IsFailure())
            {
                NN_DETAIL_BCAT_INFO("[bcat] nn::bgtc::Task::NotifyStarting failed. code = %03d-%04d\n",
                    bgtcResult.GetModule(), bgtcResult.GetDescription());
            }

            NN_UTIL_SCOPE_EXIT
            {
                if (bgtcResult.IsSuccess())
                {
                    bgtcTask.NotifyFinished();
                }
            };

            if (TaskManager::GetInstance().GetScheduleEvent().TryWait())
            {
                NN_DETAIL_BCAT_INFO("[bcat] (!) TaskManager::ScheduleEvent was signaled.\n");
                TaskManager::GetInstance().GetScheduleEvent().Clear();

                TaskManager::GetInstance().Schedule();

                EnqueueBackgroundTaskIfEmpty();
            }
            if (ExecutionQueue::GetInstance().GetEvent().TryWait())
            {
                NN_DETAIL_BCAT_INFO("[bcat] (!) ExecutionQueue::Event was signaled.\n");
                ExecutionQueue::GetInstance().GetEvent().Clear();

                ProcessRunTask();
                EnqueueBackgroundTaskIfEmpty();
            }
            if (TaskManager::GetInstance().GetSubscriptionEvent().TryWait() || g_NpnsConnectedEvent.TryWait())
            {
                if (TaskManager::GetInstance().GetSubscriptionEvent().TryWait())
                {
                    NN_DETAIL_BCAT_INFO("[bcat] (!) TaskManager::SubscriptionEvent was signaled.\n");
                    TaskManager::GetInstance().GetSubscriptionEvent().Clear();
                }
                if (g_NpnsConnectedEvent.TryWait())
                {
                    NN_DETAIL_BCAT_INFO("[bcat] (!) NpnsConnectedEvent was signaled.\n");
                    g_NpnsConnectedEvent.Clear();
                }

                ProcessNpnsSubscription();
            }
        }
    } // NOLINT(impl/function_size)
}

void BackgroundWorkerThread::Start() NN_NOEXCEPT
{
    g_StopEvent.Clear();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, WorkerThread, nullptr,
        g_ThreadStack, sizeof (g_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(bcat, Downloader)));

    nn::os::SetThreadNamePointer(&g_Thread, NN_SYSTEM_THREAD_NAME(bcat, Downloader));
    nn::os::StartThread(&g_Thread);
}

void BackgroundWorkerThread::Stop() NN_NOEXCEPT
{
    g_StopEvent.Signal();

    nn::os::DestroyThread(&g_Thread);
}

void BackgroundWorkerThread::GetProgress(DeliveryCacheProgressImpl* outProgress, TaskId taskId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outProgress);
    NN_SDK_REQUIRES_NOT_EQUAL(taskId, InvalidTaskId);

    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    DeliveryCacheProgressImpl progress = {};

    if (detail::service::core::ExecutionQueue::GetInstance().Exists(taskId))
    {
        progress.status = DeliveryCacheProgressStatusImpl_Queued;
    }
    else
    {
        detail::service::core::DeliveryCacheProgressManager::GetInstance().GetProgress(&progress, taskId);
    }

    *outProgress = progress;
}

bool BackgroundWorkerThread::GetCurrentBackgroundTaskApplicationId(nn::ApplicationId* outAppId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAppId);

    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    if (g_CurrentQueueItem.taskId != InvalidTaskId && g_CurrentQueueItem.taskId == g_BackgroundTaskId)
    {
        *outAppId = g_CurrentQueueItem.appId;
        return true;
    }

    return false;
}

void BackgroundWorkerThread::RegisterBackgroundTask(nn::ApplicationId appId, uint32_t appVersion) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    NN_ABORT_UNLESS_RESULT_SUCCESS(TaskManager::GetInstance().AddOrUpdate(appId, appVersion));
}

void BackgroundWorkerThread::UnregisterBackgroundTask(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    NN_ABORT_UNLESS_RESULT_SUCCESS(TaskManager::GetInstance().Remove(appId));
}

void BackgroundWorkerThread::SuspendBackgroundTask(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    nn::Result result = TaskManager::GetInstance().Suspend(appId);

    if (result.IsFailure())
    {
        NN_DETAIL_BCAT_INFO("[bcat] TaskManager::Suspend(%016llx) failed. code = %03d-%04d\n",
            appId.value, result.GetModule(), result.GetDescription());
    }

    if (g_BackgroundTaskApplicationId == appId)
    {
        if (g_CurrentQueueItem.taskId == g_BackgroundTaskId)
        {
            g_DownloadCancelEvent.Signal();

            // SyncCancelEvent を Signal するとコミット処理中のディレクトリが不定な状態になってしまうため何もしない。
        }
        else
        {
            SessionManager::GetInstance().RemoveTask(g_BackgroundSessionId, g_BackgroundTaskId);
        }
    }
}

void BackgroundWorkerThread::ResumeBackgroundTask(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    nn::Result result = TaskManager::GetInstance().Resume(appId);

    if (result.IsFailure())
    {
        NN_DETAIL_BCAT_INFO("[bcat] TaskManager::Resume(%016llx) failed. code = %03d-%04d\n",
            appId.value, result.GetModule(), result.GetDescription());
    }
}

void BackgroundWorkerThread::BlockTask(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    nn::os::Event finishEvent(nn::os::EventClearMode_ManualClear);
    bool wait = false;

    {
        std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

        NN_ABORT_UNLESS(g_RunTaskBlockListCount < NN_ARRAY_SIZE(g_RunTaskBlockList));

        g_RunTaskBlockList[g_RunTaskBlockListCount++] = appId;

        nn::Result result = TaskManager::GetInstance().Suspend(appId);

        if (result.IsFailure())
        {
            NN_DETAIL_BCAT_INFO("[bcat] TaskManager::Suspend(%016llx) failed. code = %03d-%04d\n",
                appId.value, result.GetModule(), result.GetDescription());
        }

        if (g_CurrentQueueItem.taskId != InvalidTaskId && g_CurrentQueueItem.appId == appId)
        {
            g_DownloadCancelEvent.Signal();
            g_SyncCancelEvent.Signal();

            NN_SDK_ASSERT(g_pFinishEvent == nullptr);
            g_pFinishEvent = &finishEvent;

            wait = true;
        }
    }

    if (wait)
    {
        finishEvent.Wait();
    }
}

void BackgroundWorkerThread::UnblockTask(nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    nn::Result result = TaskManager::GetInstance().Resume(appId);

    if (result.IsFailure() && !ResultNotFound::Includes(result))
    {
        NN_DETAIL_BCAT_INFO("[bcat] TaskManager::Resume(%016llx) failed. code = %03d-%04d\n",
            appId.value, result.GetModule(), result.GetDescription());
    }

    for (int i = 0; i < g_RunTaskBlockListCount; i++)
    {
        if (appId == g_RunTaskBlockList[i])
        {
            int moveCount = g_RunTaskBlockListCount - 1 - i;

            if (moveCount > 0)
            {
                std::memmove(&g_RunTaskBlockList[i], &g_RunTaskBlockList[i + 1], sizeof (nn::ApplicationId) * moveCount);
            }
            g_RunTaskBlockListCount--;

            return;
        }
    }
}

void BackgroundWorkerThread::CancelCurrentBackgroundTask() NN_NOEXCEPT
{
    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    if (g_CurrentQueueItem.taskId != InvalidTaskId && g_CurrentQueueItem.taskId == g_BackgroundTaskId)
    {
        g_DownloadCancelEvent.Signal();

        // SyncCancelEvent を Signal するとコミット処理中のディレクトリが不定な状態になってしまうため何もしない。
    }
}

void BackgroundWorkerThread::CancelTask(SessionId sessionId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(sessionId, InvalidSessionId);

    std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

    if (g_CurrentQueueItem.sessionId == sessionId)
    {
        g_DownloadCancelEvent.Signal();

        // SyncCancelEvent を Signal するとコミット処理中のディレクトリが不定な状態になってしまうため何もしない。
    }
}

bool BackgroundWorkerThread::GetNextScheduleInterval(nn::TimeSpan* outInterval) NN_NOEXCEPT
{
    TaskManager::Record record = {};

    // 実行可能なタスクが存在すれば、即座に起きるようにする。
    if (TaskManager::GetInstance().GetRunnableTaskRecord(&record))
    {
        *outInterval = nn::TimeSpan(0);
        return true;
    }

    return TaskManager::GetInstance().GetNextScheduleInterval(outInterval);
}

void BackgroundWorkerThread::NotifyNpnsConnected() NN_NOEXCEPT
{
    g_NpnsConnectedEvent.Signal();
}

void BackgroundWorkerThread::NotifySystemRunnableStateChanged(bool isRunnable) NN_NOEXCEPT
{
    g_IsSystemRunnable = isRunnable;

    g_SystemRunnableEvent.Signal();
}

}}}}}
