﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/bcat_DeliveryCacheFileService.h>
#include <nn/bcat/detail/service/bcat_DeliveryCacheStorageService.h>
#include <nn/bcat/detail/service/core/bcat_DeliveryCacheStorageManager.h>
#include <nn/bcat/detail/service/core/bcat_MetaView.h>

#define CHECK_IPC_STRING(value, maxSize) \
    do                                                              \
    {                                                               \
        size_t size = value.GetLength();                            \
        if (size == 0 || size > maxSize || value[size - 1] != '\0') \
        {                                                           \
            NN_RESULT_THROW(ResultInvalidArgument());               \
        }                                                           \
    }                                                               \
    while (NN_STATIC_CONDITION(false))

namespace nn { namespace bcat { namespace detail { namespace service {

DeliveryCacheFileService::DeliveryCacheFileService(nn::sf::SharedPointer<DeliveryCacheStorageService>&& parent,
    nn::ApplicationId appId, const Capability& capability) NN_NOEXCEPT :
    m_Mutex(true),
    m_Parent(std::move(parent)),
    m_Capability(capability),
    m_AppId(appId),
    m_IsOpened(false)
{
}

DeliveryCacheFileService::~DeliveryCacheFileService() NN_NOEXCEPT
{
    if (m_IsOpened)
    {
        nn::fs::CloseFile(m_Handle);
    }

    m_Parent->NotifyFileServiceDeleted();
}

nn::Result DeliveryCacheFileService::Open(const nn::bcat::DirectoryName& dirName, const nn::bcat::FileName& fileName) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(dirName.IsValid(), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(fileName.IsValid(), ResultInvalidArgument());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(!m_IsOpened, ResultAlreadyOpened());

    detail::service::core::FileMetaView view;

    NN_RESULT_DO(detail::service::core::FileMetaView::LoadFromDeliveryCacheStorage(&view, m_AppId, dirName.value));

    FileMeta meta = {};
    bool isFound = false;

    for (int i = 0; i < view.GetCount(); i++)
    {
        NN_RESULT_DO(view.GetEntry(&meta, i));

        if (nn::util::Strnicmp(meta.name.value, fileName.value, sizeof (fileName.value)) == 0)
        {
            isFound = true;
            break;
        }
    }

    NN_RESULT_THROW_UNLESS(isFound, ResultNotFound());

    char path[128] = {};
    detail::service::core::DeliveryCacheStorageManager::GetInstance().MakeFilePath(path, sizeof (path),
        m_AppId, dirName.value, fileName.value);

    NN_RESULT_DO(nn::fs::OpenFile(&m_Handle, path, nn::fs::OpenMode_Read));

    m_Meta = meta;
    m_IsOpened = true;

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheFileService::Read(nn::sf::Out<std::uint64_t> outSize, std::int64_t offset, const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_IsOpened, ResultNotOpened());

    size_t actualSize = 0;

    NN_RESULT_DO(nn::fs::ReadFile(&actualSize, m_Handle, offset, buffer.GetPointerUnsafe(), buffer.GetSize()));

    *outSize = static_cast<uint64_t>(actualSize);

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheFileService::GetSize(nn::sf::Out<std::int64_t> outSize) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_IsOpened, ResultNotOpened());

    NN_RESULT_DO(nn::fs::GetFileSize(outSize.GetPointer(), m_Handle));

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheFileService::GetDigest(nn::sf::Out<nn::bcat::Digest> outDigest) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_IsOpened, ResultNotOpened());

    *outDigest = m_Meta.digest;

    NN_RESULT_SUCCESS;
}

}}}}
