﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/bcat_DeliveryCacheDirectoryService.h>
#include <nn/bcat/detail/service/bcat_DeliveryCacheStorageService.h>
#include <nn/bcat/detail/service/core/bcat_MetaView.h>

#define CHECK_IPC_STRING(value, maxSize) \
    do                                                              \
    {                                                               \
        size_t size = value.GetLength();                            \
        if (size == 0 || size > maxSize || value[size - 1] != '\0') \
        {                                                           \
            NN_RESULT_THROW(ResultInvalidArgument());               \
        }                                                           \
    }                                                               \
    while (NN_STATIC_CONDITION(false))

namespace nn { namespace bcat { namespace detail { namespace service {

DeliveryCacheDirectoryService::DeliveryCacheDirectoryService(nn::sf::SharedPointer<DeliveryCacheStorageService>&& parent,
    nn::ApplicationId appId, const Capability& capability) NN_NOEXCEPT :
    m_Mutex(true),
    m_Parent(std::move(parent)),
    m_Capability(capability),
    m_AppId(appId),
    m_IsOpened(false),
    m_EntryCount(0)
{
}

DeliveryCacheDirectoryService::~DeliveryCacheDirectoryService() NN_NOEXCEPT
{
    m_Parent->NotifyDirectoryServiceDeleted();
}

nn::Result DeliveryCacheDirectoryService::Open(const nn::bcat::DirectoryName& dirName) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(dirName.IsValid(), ResultInvalidArgument());

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(!m_IsOpened, ResultAlreadyOpened());

    detail::service::core::FileMetaView view;

    NN_RESULT_DO(detail::service::core::FileMetaView::LoadFromDeliveryCacheStorage(&view, m_AppId, dirName.value, false));

    m_EntryCount = view.GetCount();

    m_DirName = dirName;
    m_IsOpened = true;

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDirectoryService::Read(nn::sf::Out<std::int32_t> outCount, const nn::sf::OutArray<nn::bcat::DeliveryCacheDirectoryEntry>& outEntries) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_IsOpened, ResultNotOpened());

    detail::service::core::FileMetaView view;

    NN_RESULT_DO(detail::service::core::FileMetaView::LoadFromDeliveryCacheStorage(&view, m_AppId, m_DirName.value));

    int actualCount = 0;

    while (actualCount < static_cast<int>(outEntries.GetLength()))
    {
        FileMeta meta = {};

        NN_RESULT_TRY(view.GetEntry(&meta, actualCount))
            NN_RESULT_CATCH(ResultNotFound)
            {
                break;
            }
        NN_RESULT_END_TRY;

        DeliveryCacheDirectoryEntry& entry = outEntries[actualCount++];

        entry.name = meta.name;
        entry.size = meta.size;
        entry.digest = meta.digest;
    }

    *outCount = static_cast<int32_t>(actualCount);

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDirectoryService::GetCount(nn::sf::Out<std::int32_t> outCount) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_RESULT_THROW_UNLESS(m_IsOpened, ResultNotOpened());

    *outCount = static_cast<int32_t>(m_EntryCount);

    NN_RESULT_SUCCESS;
}

}}}}
