﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gpio/gpio.h>
#include <nn/os.h>
#include "audioctrl_GpioInterruptHandler.h"


namespace nn { namespace audioctrl { namespace server { namespace detail {

GpioInterruptHandler::GpioInterruptHandler() NN_NOEXCEPT
{
}

GpioInterruptHandler::GpioInterruptHandler(nn::gpio::GpioPadName name, nn::gpio::Direction direction, nn::gpio::InterruptMode mode) NN_NOEXCEPT
{
    Initialize(name, direction, mode);
}

void GpioInterruptHandler::Initialize(nn::gpio::GpioPadName name, nn::gpio::Direction direction, nn::gpio::InterruptMode mode) NN_NOEXCEPT
{
    nn::gpio::OpenSession(&m_Gpio, name);
    nn::gpio::SetDirection(&m_Gpio, direction);
    nn::gpio::SetInterruptMode(&m_Gpio, mode);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::gpio::BindInterrupt(&m_Event, &m_Gpio));
    nn::gpio::ClearInterruptStatus(&m_Gpio);
    nn::os::ClearSystemEvent(&m_Event);
    nn::gpio::SetInterruptEnable(&m_Gpio, true);
    nn::os::InitializeMultiWaitHolder(&m_EventHolder, &m_Event);
    nn::os::SetMultiWaitHolderUserData(&m_EventHolder, reinterpret_cast<uintptr_t>(this));

    // チャタリング防止機能の設定
    nn::gpio::SetDebounceEnabled(&m_Gpio, true);
    // TODO: 暫定的にチャタリング防止の秒数をここで設定する
    // (秒数はピンのグループ単位で設定されるので、ここで設定するのは本来は適切ではない)
    nn::gpio::SetDebounceTime(&m_Gpio, 16);
}

void GpioInterruptHandler::Finalize() NN_NOEXCEPT
{
    nn::os::UnlinkMultiWaitHolder(&m_EventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_EventHolder);
    nn::gpio::UnbindInterrupt(&m_Gpio);
    nn::gpio::CloseSession(&m_Gpio);
}

void GpioInterruptHandler::LinkMultiWait(nn::os::MultiWaitType* pWaiter) NN_NOEXCEPT
{
    nn::os::LinkMultiWaitHolder(pWaiter, &m_EventHolder);
}

void GpioInterruptHandler::Clear() NN_NOEXCEPT
{
    nn::gpio::ClearInterruptStatus(&m_Gpio);
    nn::os::ClearSystemEvent(&m_Event);
    nn::gpio::SetInterruptEnable(&m_Gpio, true);
}

bool GpioInterruptHandler::IsHigh() NN_NOEXCEPT
{
    return (nn::gpio::GetValue(&m_Gpio) == nn::gpio::GpioValue_High);
}

bool GpioInterruptHandler::IsLow() NN_NOEXCEPT
{
    return (nn::gpio::GetValue(&m_Gpio) == nn::gpio::GpioValue_Low);
}

nn::os::SystemEventType* GpioInterruptHandler::GetSystemEvent() NN_NOEXCEPT
{
    return &m_Event;
}

}}}}  // namespace nn::audioctrl::server::detail
