﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audioctrl_AudioControllerServiceImpl-os.win32.h"

#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>
#include <nn/settings/system/settings_Audio.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/audioctrl/audioctrl_PlayReport.h>
#include <nn/audioctrl/detail/audioctrl_Log.h>

namespace nn { namespace audioctrl { namespace server { namespace detail {

AudioControllerServiceImpl::AudioControllerServiceImpl() NN_NOEXCEPT
    : m_Mutex(true),
      m_AudioVolumeUpdateEventForPlayReport(nn::os::EventClearMode_AutoClear, true),
      m_AudioOutputDeviceUpdateEventForPlayReport(nn::os::EventClearMode_AutoClear, true),
      m_CurrentAudioOutputTarget(AudioTarget_Invalid),
      m_HeadphoneVolumeWarningDisplayedEventCount(0),
      m_DefaultOutput(AudioTarget_Speaker), // OMM が存在しなく Initialize() が呼ばれない環境なのでここで Speaker を設定しておく
      m_IsInitialized(false),
      m_SystemOutputMasterVolume(0.f)
{
    // VS2013 で警告が出るので初期化リストを使用しない
    for(auto& isTargetConnected : m_IsTargetsConnected)
    {
        isTargetConnected = false;
    }

    for(auto& audioOutputMode : m_AudioOutputMode)
    {
        audioOutputMode = AudioOutputMode_Invalid;
    }

    for(auto& audioVolume : m_AudioVolumes)
    {
        audioVolume.flags.Reset();
        audioVolume.volume = 0;
    }
}

nn::settings::system::AudioVolumeTarget AudioControllerServiceImpl::ConvertToAudioVolumeTarget(AudioTarget target) NN_NOEXCEPT
{
    switch (target)
    {
        case nn::audioctrl::AudioTarget_Speaker:
        {
            return nn::settings::system::AudioVolumeTarget_Speaker;
        }
        case nn::audioctrl::AudioTarget_Headphone:
        {
            return nn::settings::system::AudioVolumeTarget_Headphone;
        }
        case nn::audioctrl::AudioTarget_UsbOutputDevice:
        {
            return nn::settings::system::AudioVolumeTarget_Usb;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

nn::settings::system::AudioOutputModeTarget AudioControllerServiceImpl::ConvertToAudioOutputModeTarget(AudioTarget target) NN_NOEXCEPT
{
    switch (target)
    {
        case nn::audioctrl::AudioTarget_Speaker:
        {
            return nn::settings::system::AudioOutputModeTarget_Speaker;
        }
        case nn::audioctrl::AudioTarget_Headphone:
        {
            return nn::settings::system::AudioOutputModeTarget_Headphone;
        }
        case nn::audioctrl::AudioTarget_UsbOutputDevice:
        {
            return nn::settings::system::AudioOutputModeTarget_Usb;
        }
        case nn::audioctrl::AudioTarget_Tv:
        {
            return nn::settings::system::AudioOutputModeTarget_Hdmi;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

PlayReportAudioOutputTarget AudioControllerServiceImpl::ConvertToPlayReportAudioOutputTarget(AudioTarget target) NN_NOEXCEPT
{
    switch (target)
    {
        case nn::audioctrl::AudioTarget_Invalid:
        {
            return PlayReportAudioOutputTarget_Invalid;
        }
        case nn::audioctrl::AudioTarget_Speaker:
        {
            return PlayReportAudioOutputTarget_Speaker;
        }
        case nn::audioctrl::AudioTarget_Headphone:
        {
            return PlayReportAudioOutputTarget_Headphone;
        }
        case nn::audioctrl::AudioTarget_UsbOutputDevice:
        {
            return PlayReportAudioOutputTarget_UsbOutputDevice;
        }
        case nn::audioctrl::AudioTarget_Tv:
        {
            return PlayReportAudioOutputTarget_Tv;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

/*
 * -------------------------------------------------------------------------------------------------------------------------------------|
 * |                                              出力チャンネル数の列挙型の対応                                                        |
 * -------------------------------------------------------------------------------------------------------------------------------------|
 * |         nn::audioctrl の列挙型         |                             nn::settings の列挙型                                         |
 * -----------------------------------------|-------------------------------------------------------------------------------------------|
 * | nn::audioctrl::AudioOutputMode_Pcm1ch  | nn::settings::system::AudioOutputMode_1ch                                                 |
 * | nn::audioctrl::AudioOutputMode_Pcm2ch  | nn::settings::system::AudioOutputMode_2ch                                                 |
 * | nn::audioctrl::AudioOutputMode_Pcm6ch  | nn::settings::system::AudioOutputMode_5_1ch_Forced (接続先が対応してなくても 5.1 ch 出力) |
 * | nn::audioctrl::AudioOutputMode_PcmAuto | nn::settings::system::AudioOutputMode_5_1ch        (接続先が対応していれば 5.1 ch 出力)   |
 * -------------------------------------------------------------------------------------------------------------------------------------|
 */
nn::settings::system::AudioOutputMode AudioControllerServiceImpl::ConvertToAudioOutputMode(AudioOutputMode mode) NN_NOEXCEPT
{
    switch (mode)
    {
        case nn::audioctrl::AudioOutputMode_Pcm1ch:
        {
            return nn::settings::system::AudioOutputMode_1ch;
        }
        case nn::audioctrl::AudioOutputMode_Pcm2ch:
        {
            return nn::settings::system::AudioOutputMode_2ch;
        }
        case nn::audioctrl::AudioOutputMode_Pcm6ch:
        {
            return nn::settings::system::AudioOutputMode_5_1ch_Forced;
        }
        case nn::audioctrl::AudioOutputMode_PcmAuto:
        {
            return nn::settings::system::AudioOutputMode_5_1ch;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

AudioOutputMode AudioControllerServiceImpl::ConvertToAudioOutputMode(nn::settings::system::AudioOutputMode mode) NN_NOEXCEPT
{
    switch (mode)
    {
        case nn::settings::system::AudioOutputMode_1ch:
        {
            return nn::audioctrl::AudioOutputMode_Pcm1ch;
        }
        case nn::settings::system::AudioOutputMode_2ch:
        {
            return nn::audioctrl::AudioOutputMode_Pcm2ch;
        }
        case nn::settings::system::AudioOutputMode_5_1ch_Forced:
        {
            return nn::audioctrl::AudioOutputMode_Pcm6ch;
        }
        case nn::settings::system::AudioOutputMode_5_1ch:
        {
            return nn::audioctrl::AudioOutputMode_PcmAuto;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

bool AudioControllerServiceImpl::IsValidAudioTargetForOutputMode(AudioTarget target) NN_NOEXCEPT
{
    return (target == AudioTarget_Speaker || target == AudioTarget_Tv || target == AudioTarget_UsbOutputDevice);
}

bool AudioControllerServiceImpl::IsValidTargetForOutputTarget(AudioTarget target) NN_NOEXCEPT
{
    return target == AudioTarget_Invalid || target == AudioTarget_Speaker || target == AudioTarget_Headphone || target == AudioTarget_Tv || target == AudioTarget_UsbOutputDevice;
}

bool AudioControllerServiceImpl::IsValidOutputMode(AudioOutputMode mode) NN_NOEXCEPT
{
    return mode == AudioOutputMode_Pcm1ch || mode == AudioOutputMode_Pcm2ch || mode == AudioOutputMode_Pcm6ch || mode == AudioOutputMode_PcmAuto;
}

void AudioControllerServiceImpl::Initialize(bool hasCodecIc, bool hasHda, bool isUacEnabled) NN_NOEXCEPT
{
    NN_UNUSED(hasCodecIc);
    NN_UNUSED(hasHda);
    NN_UNUSED(isUacEnabled);

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // 初期化済みフラグを有効にする
    m_IsInitialized = true;

    // システム出力マスターボリュームの初期値を設定
    m_SystemOutputMasterVolume = 1.0f;
}

void AudioControllerServiceImpl::Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // 初期化済みフラグを無効にする
    m_IsInitialized = false;
}

void AudioControllerServiceImpl::Sleep() NN_NOEXCEPT
{

}

void AudioControllerServiceImpl::Wake() NN_NOEXCEPT
{

}

void AudioControllerServiceImpl::SetTargetVolume(AudioTarget target, int8_t volume) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    auto& audioVolume = m_AudioVolumes[target];
    audioVolume.volume = volume;
}

int8_t AudioControllerServiceImpl::GetTargetVolume(AudioTarget target) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return m_AudioVolumes[target].volume;
}

void AudioControllerServiceImpl::SetTargetMute(AudioTarget target, bool isMute) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    auto& audioVolume = m_AudioVolumes[target];
    audioVolume.flags.Set<nn::settings::system::AudioVolumeFlag::Mute>(isMute);
}

bool AudioControllerServiceImpl::IsTargetMute(AudioTarget target) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return m_AudioVolumes[target].flags.Test<nn::settings::system::AudioVolumeFlag::Mute>();
}

void AudioControllerServiceImpl::SaveTargetVolumeSetting(AudioTarget target) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    nn::settings::system::SetAudioVolume(m_AudioVolumes[target], ConvertToAudioVolumeTarget(target));
}

int AudioControllerServiceImpl::GetTargetVolumeMin() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return VolumeMin;
}

int AudioControllerServiceImpl::GetTargetVolumeMax() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return VolumeMax;
}

void AudioControllerServiceImpl::SetTargetConnected(AudioTarget target, bool isConnected) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_IsTargetsConnected[target] = isConnected;
}

bool AudioControllerServiceImpl::IsTargetConnected(AudioTarget target) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return m_IsTargetsConnected[target];
}

bool AudioControllerServiceImpl::SetDefaultTarget(AudioTarget target, nn::TimeSpanType fadeOutTime, nn::TimeSpanType fadeInTime) NN_NOEXCEPT
{
    NN_UNUSED(fadeOutTime);
    NN_UNUSED(fadeInTime);

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_DefaultOutput = target;
    return true;
}

AudioTarget AudioControllerServiceImpl::GetDefaultTarget() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return m_DefaultOutput;
}

AudioOutputMode AudioControllerServiceImpl::GetOutputModeSetting(AudioTarget target) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!IsValidAudioTargetForOutputMode(target))
    {
        return AudioOutputMode_Invalid;
    }
    return ConvertToAudioOutputMode(nn::settings::system::GetAudioOutputMode(ConvertToAudioOutputModeTarget(target)));
}

void AudioControllerServiceImpl::SetOutputModeSetting(AudioTarget target, AudioOutputMode mode) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!IsValidAudioTargetForOutputMode(target))
    {
        return;
    }

    if (!IsValidOutputMode(mode))
    {
        return;
    }

    return nn::settings::system::SetAudioOutputMode(ConvertToAudioOutputMode(mode), ConvertToAudioOutputModeTarget(target));
}

AudioOutputMode AudioControllerServiceImpl::GetAudioOutputMode(AudioTarget target) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!IsValidAudioTargetForOutputMode(target))
    {
        return AudioOutputMode_Invalid;
    }
    return m_AudioOutputMode[target];
}

void AudioControllerServiceImpl::SetAudioOutputMode(AudioTarget target, AudioOutputMode mode) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!IsValidAudioTargetForOutputMode(target))
    {
        return;
    }

    if (!IsValidOutputMode(mode))
    {
        return;
    }

    m_AudioOutputMode[target] = mode;
}

void AudioControllerServiceImpl::SetForceMutePolicy(ForceMutePolicy policy) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if(policy == ForceMutePolicy_SpeakerMuteOnHeadphoneUnplugged)
    {
        nn::settings::system::SetForceMuteOnHeadphoneRemoved(true);
    }
    else if(policy == ForceMutePolicy_Disable)
    {
        nn::settings::system::SetForceMuteOnHeadphoneRemoved(false);
    }
    else
    {
        NN_ABORT("Unexpected policy");
    }
}

ForceMutePolicy AudioControllerServiceImpl::GetForceMutePolicy() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if(nn::settings::system::IsForceMuteOnHeadphoneRemoved())
    {
        return ForceMutePolicy_SpeakerMuteOnHeadphoneUnplugged;
    }
    else
    {
        return ForceMutePolicy_Disable;
    }
}

void AudioControllerServiceImpl::SetOutputTarget(AudioTarget target) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (!IsValidTargetForOutputTarget(target))
    {
        return;
    }

    if(target == AudioTarget_Speaker || target == AudioTarget_Headphone)
    {
        SetTargetVolume(target, m_AudioVolumes[target].volume);
    }
}

void AudioControllerServiceImpl::SetInputTargetForceEnabled(bool isEnabled) NN_NOEXCEPT
{
    NN_UNUSED(isEnabled);
}

void AudioControllerServiceImpl::SetHeadphoneOutputLevelMode(HeadphoneOutputLevelMode mode) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_REQUIRES(mode == HeadphoneOutputLevelMode_Normal || mode == HeadphoneOutputLevelMode_HighPower);
    auto& audioVolume = m_AudioVolumes[AudioTarget_Headphone];

    audioVolume.flags.Set<nn::settings::system::AudioVolumeFlag::HighPower>(mode == HeadphoneOutputLevelMode_HighPower);
    SaveTargetVolumeSetting(AudioTarget_Headphone);
}

HeadphoneOutputLevelMode AudioControllerServiceImpl::GetHeadphoneOutputLevelMode() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if(m_AudioVolumes[AudioTarget_Headphone].flags.Test<nn::settings::system::AudioVolumeFlag::HighPower>())
    {
        return HeadphoneOutputLevelMode_HighPower;
    }
    else
    {
        return HeadphoneOutputLevelMode_Normal;
    }
}

void AudioControllerServiceImpl::AcquireAudioVolumeUpdateEventForPlayReport(nn::os::NativeHandle* outHandle) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if(outHandle == nullptr)
    {
        return;
    }

    m_AudioVolumeUpdateEventForPlayReport.Signal();
    *outHandle = m_AudioVolumeUpdateEventForPlayReport.GetReadableHandle();
}

void AudioControllerServiceImpl::AcquireAudioOutputDeviceUpdateEventForPlayReport(nn::os::NativeHandle* outHandle) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if(outHandle == nullptr)
    {
        return;
    }

    m_AudioOutputDeviceUpdateEventForPlayReport.Signal();
    *outHandle = m_AudioOutputDeviceUpdateEventForPlayReport.GetReadableHandle();
}

void AudioControllerServiceImpl::NotifyAudioOutputTarget(AudioTarget target) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_CurrentAudioOutputTarget = target;
}

void AudioControllerServiceImpl::NotifyUnsupportedUsbOutputDeviceAttached() NN_NOEXCEPT
{

}

PlayReportAudioOutputTarget AudioControllerServiceImpl::GetAudioOutputTargetForPlayReport() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return ConvertToPlayReportAudioOutputTarget(m_CurrentAudioOutputTarget);
}

void AudioControllerServiceImpl::GetAudioVolumeDataForPlayReport(PlayReportAudioVolumeData *pOutData) NN_NOEXCEPT
{
    *pOutData = {};
}

void AudioControllerServiceImpl::NotifyHeadphoneVolumeWarningDisplayedEvent() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    ++m_HeadphoneVolumeWarningDisplayedEventCount;
    nn::settings::system::SetHeadphoneVolumeWarningDisplayedEventCount(m_HeadphoneVolumeWarningDisplayedEventCount);
}

int AudioControllerServiceImpl::GetHeadphoneVolumeWarningDisplayedEventCount() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return m_HeadphoneVolumeWarningDisplayedEventCount;
}

int AudioControllerServiceImpl::GetHeadphoneRs1VolumeMax(HeadphoneOutputLevelMode mode) NN_NOEXCEPT
{
    NN_UNUSED(mode);

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return 1;
}

OverlaySenderController& AudioControllerServiceImpl::GetOverlaySenderController() NN_NOEXCEPT
{
    return m_OverlaySenderController;
}

bool AudioControllerServiceImpl::IsHeadphoneVolumeWarningTransmissionNeeded() NN_NOEXCEPT
{
    return false;
}

void AudioControllerServiceImpl::SetSystemOutputMasterVolume(float volume) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_SystemOutputMasterVolume = volume;
}

float AudioControllerServiceImpl::GetSystemOutputMasterVolume() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    return m_SystemOutputMasterVolume;
}

bool AudioControllerServiceImpl::NeedsToUpdateHeadphoneVolume(bool* pOutIsHighPowerAvailable, int8_t* pOutVolume, bool isRestricted) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    return nn::settings::system::NeedsToUpdateHeadphoneVolume(pOutIsHighPowerAvailable, pOutVolume, isRestricted);
}

}}}}  // namespace nn::audioctrl::server::detail
