﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audioctrl_AudioControllerServiceImpl-os.horizon.h"

#include <cmath> // pow
#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_Thread.h>
#include <nn/settings/system/settings_Audio.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/audioctrl/audioctrl_PlayReport.h>
#include <nn/audioctrl/detail/audioctrl_Log.h>

#include "audioctrl_UserInputHandler.h"

// TODO: remove this decl.
namespace nn { namespace audio {

// Defined in "audio_AudioHardware.cpp"
void NotifyOutputDeviceEvent() NN_NOEXCEPT;
void NotifyOutputDeviceEvent(bool isHda, nn::TimeSpanType fadeOutTime, nn::TimeSpanType fadeInTime) NN_NOEXCEPT;
void NotifyHeadphoneOutSelect(bool isEnable) NN_NOEXCEPT;
void NotifyOutputDeviceForceSwitchEvent(nn::audioctrl::AudioTarget target) NN_NOEXCEPT;
void NotifyInputDeviceForceEnable(bool isEnable) NN_NOEXCEPT;
void NotifyHeadphoneInEnabled(bool isEnable) NN_NOEXCEPT;
void SetHeadphoneAmplifierGain(int32_t gain) NN_NOEXCEPT;
void SetSpeakerVolume(int32_t volume) NN_NOEXCEPT;
void SetHeadphoneVolume(int32_t volume) NN_NOEXCEPT;
bool IsJackPlugged() NN_NOEXCEPT;
void SetOutputMasterVolume(float volume, uint32_t curveDurationInMicroseconds) NN_NOEXCEPT;
void SetUacOutputDeviceVolume(float volume) NN_NOEXCEPT;
int GetNumUacSpeakers() NN_NOEXCEPT;
void NotifyUacSpeakerSwitch(bool isEnable) NN_NOEXCEPT;
void SetOutputMode(nn::audioctrl::AudioTarget target, nn::audioctrl::AudioOutputMode mode) NN_NOEXCEPT;
}}

namespace  {

class VolumeMapper
{
public:
    static const int VolumeLevelCount = 16;
    // HighPower 時に欧州安全規制の RS1レベル (セーフガードが不要なレベル) 以下の最大音量設定値
    // 値が決定された経緯は以下の URL の課題を参照
    // http://spdlybra.nintendo.co.jp/jira/browse/SWHUID-10403?focusedCommentId=1867238&page=com.atlassian.jira.plugin.system.issuetabpanels:comment-tabpanel#comment-1867238
    static const uint8_t Rs1VolumeMaxForHeadphoneNormalMode = VolumeLevelCount - 1;
    static const uint8_t Rs1VolumeMaxForHeadphoneHighPowerMode = 9;
private:
    static const uint8_t VolumeMapForSpeaker[VolumeLevelCount];
    static const uint8_t VolumeMapForHeadphone[VolumeLevelCount];
    static const uint8_t VolumeMapForUsbOutputDevice[VolumeLevelCount];

public:
    static uint32_t GetHardwareVolume(int32_t volume, nn::audioctrl::AudioTarget target) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_RANGE(volume, 0, VolumeLevelCount);

        uint32_t val = 0;
        switch (target)
        {
        case nn::audioctrl::AudioTarget_Speaker:
            val = VolumeMapper::VolumeMapForSpeaker[volume];
            return val;
        case nn::audioctrl::AudioTarget_Headphone:
            val = VolumeMapper::VolumeMapForHeadphone[volume];
            return val;
        case nn::audioctrl::AudioTarget_UsbOutputDevice:
            val = VolumeMapper::VolumeMapForUsbOutputDevice[volume];
            return val;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

private:
    VolumeMapper() NN_NOEXCEPT;
};
const uint8_t VolumeMapper::VolumeMapForSpeaker[VolumeLevelCount] =
{
    0, 21, 39, 55, 69, 81, 91, 100, 108, 115, 121, 127, 133, 138, 143, 148
};
const uint8_t VolumeMapper::VolumeMapForHeadphone[VolumeLevelCount] =
{
    0, 32, 46, 61, 76, 91, 104, 115, 124, 131, 138, 145, 152, 158, 164, 170
};

const uint8_t VolumeMapper::VolumeMapForUsbOutputDevice[VolumeLevelCount] =
{
    0, 132, 158, 171, 182, 192, 200, 207, 213, 219, 225, 231, 237, 243, 249, 255
};
NN_DEFINE_STATIC_CONSTANT(const int VolumeMapper::VolumeLevelCount);
NN_DEFINE_STATIC_CONSTANT(const uint8_t VolumeMapper::Rs1VolumeMaxForHeadphoneHighPowerMode);
NN_DEFINE_STATIC_CONSTANT(const uint8_t VolumeMapper::Rs1VolumeMaxForHeadphoneNormalMode);

} // namespace

namespace nn { namespace audioctrl { namespace server { namespace detail {

AudioControllerServiceImpl::AudioControllerServiceImpl() NN_NOEXCEPT
    : m_AsyncTaskController(m_UserInputHandler, m_PlayReportController),
      m_Mutex(true),
      m_IsTargetsConnected{},
      m_AudioOutputMode{},
      m_AudioVolumes{},
      m_CurrentAudioOutputTarget(AudioTarget_Invalid),
      m_HeadphoneVolumeWarningDisplayedEventCount(0),
      m_DefaultOutput(AudioTarget_Invalid),
      m_IsInitialized(false),
      m_SystemOutputMasterVolume(0.f)
{

}

uint32_t AudioControllerServiceImpl::GetHardwareHeadphoneAmplifierGain(nn::audioctrl::HeadphoneOutputLevelMode mode) NN_NOEXCEPT
{
    // ALC5639 Datasheet Rev 0.944 8.4. MX-02h: Headphone Output Control 参照
    const int HeadphoneOutputAmplifierGainForHighPower = 0x8; //  -0.0 dB
    const int HeadphoneOutputAmplifierGainForNoraml = 0x11;   // -13.5 dB

    switch (mode)
    {
    case nn::audioctrl::HeadphoneOutputLevelMode_Normal:
        return HeadphoneOutputAmplifierGainForNoraml;
    case nn::audioctrl::HeadphoneOutputLevelMode_HighPower:
        return HeadphoneOutputAmplifierGainForHighPower;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

nn::settings::system::AudioVolumeTarget AudioControllerServiceImpl::ConvertToAudioVolumeTarget(AudioTarget target) NN_NOEXCEPT
{
    switch (target)
    {
        case nn::audioctrl::AudioTarget_Speaker:
        {
            return nn::settings::system::AudioVolumeTarget_Speaker;
        }
        case nn::audioctrl::AudioTarget_Headphone:
        {
            return nn::settings::system::AudioVolumeTarget_Headphone;
        }
        case nn::audioctrl::AudioTarget_UsbOutputDevice:
        {
            return nn::settings::system::AudioVolumeTarget_Usb;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

nn::settings::system::AudioOutputModeTarget AudioControllerServiceImpl::ConvertToAudioOutputModeTarget(AudioTarget target) NN_NOEXCEPT
{
    switch (target)
    {
        case nn::audioctrl::AudioTarget_Speaker:
        {
            return nn::settings::system::AudioOutputModeTarget_Speaker;
        }
        case nn::audioctrl::AudioTarget_Headphone:
        {
            return nn::settings::system::AudioOutputModeTarget_Headphone;
        }
        case nn::audioctrl::AudioTarget_UsbOutputDevice:
        {
            return nn::settings::system::AudioOutputModeTarget_Usb;
        }
        case nn::audioctrl::AudioTarget_Tv:
        {
            return nn::settings::system::AudioOutputModeTarget_Hdmi;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

PlayReportAudioOutputTarget AudioControllerServiceImpl::ConvertToPlayReportAudioOutputTarget(AudioTarget target) NN_NOEXCEPT
{
    switch (target)
    {
        case nn::audioctrl::AudioTarget_Invalid:
        {
            return PlayReportAudioOutputTarget_Invalid;
        }
        case nn::audioctrl::AudioTarget_Speaker:
        {
            return PlayReportAudioOutputTarget_Speaker;
        }
        case nn::audioctrl::AudioTarget_Headphone:
        {
            return PlayReportAudioOutputTarget_Headphone;
        }
        case nn::audioctrl::AudioTarget_UsbOutputDevice:
        {
            return PlayReportAudioOutputTarget_UsbOutputDevice;
        }
        case nn::audioctrl::AudioTarget_Tv:
        {
            return PlayReportAudioOutputTarget_Tv;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

HeadphoneOutputLevelMode AudioControllerServiceImpl::ConvertToHeadphoneOutputLevelMode(nn::settings::system::AudioVolume volume) NN_NOEXCEPT
{
    if(volume.flags.Test<nn::settings::system::AudioVolumeFlag::HighPower>())
    {
        return HeadphoneOutputLevelMode_HighPower;
    }
    else
    {
        return HeadphoneOutputLevelMode_Normal;
    }
}

/*
 * -------------------------------------------------------------------------------------------------------------------------------------|
 * |                                              出力チャンネル数の列挙型の対応                                                        |
 * -------------------------------------------------------------------------------------------------------------------------------------|
 * |         nn::audioctrl の列挙型         |                             nn::settings の列挙型                                         |
 * -----------------------------------------|-------------------------------------------------------------------------------------------|
 * | nn::audioctrl::AudioOutputMode_Pcm1ch  | nn::settings::system::AudioOutputMode_1ch                                                 |
 * | nn::audioctrl::AudioOutputMode_Pcm2ch  | nn::settings::system::AudioOutputMode_2ch                                                 |
 * | nn::audioctrl::AudioOutputMode_Pcm6ch  | nn::settings::system::AudioOutputMode_5_1ch_Forced (接続先が対応してなくても 5.1 ch 出力) |
 * | nn::audioctrl::AudioOutputMode_PcmAuto | nn::settings::system::AudioOutputMode_5_1ch        (接続先が対応していれば 5.1 ch 出力)   |
 * -------------------------------------------------------------------------------------------------------------------------------------|
 */
nn::settings::system::AudioOutputMode AudioControllerServiceImpl::ConvertToAudioOutputMode(AudioOutputMode mode) NN_NOEXCEPT
{
    switch (mode)
    {
        case nn::audioctrl::AudioOutputMode_Pcm1ch:
        {
            return nn::settings::system::AudioOutputMode_1ch;
        }
        case nn::audioctrl::AudioOutputMode_Pcm2ch:
        {
            return nn::settings::system::AudioOutputMode_2ch;
        }
        case nn::audioctrl::AudioOutputMode_Pcm6ch:
        {
            return nn::settings::system::AudioOutputMode_5_1ch_Forced;
        }
        case nn::audioctrl::AudioOutputMode_PcmAuto:
        {
            return nn::settings::system::AudioOutputMode_5_1ch;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

AudioOutputMode AudioControllerServiceImpl::ConvertToAudioOutputMode(nn::settings::system::AudioOutputMode mode) NN_NOEXCEPT
{
    switch (mode)
    {
        case nn::settings::system::AudioOutputMode_1ch:
        {
            return nn::audioctrl::AudioOutputMode_Pcm1ch;
        }
        case nn::settings::system::AudioOutputMode_2ch:
        {
            return nn::audioctrl::AudioOutputMode_Pcm2ch;
        }
        case nn::settings::system::AudioOutputMode_5_1ch_Forced:
        {
            return nn::audioctrl::AudioOutputMode_Pcm6ch;
        }
        case nn::settings::system::AudioOutputMode_5_1ch:
        {
            return nn::audioctrl::AudioOutputMode_PcmAuto;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

bool AudioControllerServiceImpl::IsValidAudioTargetForOutputMode(AudioTarget target) NN_NOEXCEPT
{
    return (target == AudioTarget_Speaker || target == AudioTarget_Tv || target == AudioTarget_UsbOutputDevice);
}

bool AudioControllerServiceImpl::IsValidTargetForOutputTarget(AudioTarget target) NN_NOEXCEPT
{
    return target == AudioTarget_Invalid || target == AudioTarget_Speaker || target == AudioTarget_Headphone || target == AudioTarget_Tv;
}

bool AudioControllerServiceImpl::IsValidOutputMode(AudioOutputMode mode) NN_NOEXCEPT
{
    return mode == AudioOutputMode_Pcm1ch || mode == AudioOutputMode_Pcm2ch || mode == AudioOutputMode_Pcm6ch || mode == AudioOutputMode_PcmAuto;
}

bool AudioControllerServiceImpl::IsTargetCodecDevice(AudioTarget target) NN_NOEXCEPT
{
    return target == AudioTarget_Speaker || target == AudioTarget_Headphone;
}

void AudioControllerServiceImpl::Initialize(bool hasCodecIc, bool hasHda, bool isUacEnabled) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    // 初期化済みフラグを有効にする
    m_IsInitialized = true;

    m_HasCodecIc = hasCodecIc;
    m_HasHda = hasHda;
    m_IsUacEnabled = isUacEnabled;

    // オーディオ出力設定をロードし AudioHardware に設定
    const nn::audioctrl::AudioTarget TargetsForAudioOutputMode[] = {
        AudioTarget_Speaker,
        AudioTarget_Tv,
        AudioTarget_UsbOutputDevice,
    };

    for(const auto Target : TargetsForAudioOutputMode)
    {
        const auto OutputMode = GetOutputModeSetting(Target);
        m_AudioOutputMode[Target] = OutputMode;
        nn::audio::SetOutputMode(Target, OutputMode);
    }

    m_DefaultOutput = AudioTarget_Invalid;

    // TODO: Workaround for SIGLO-61827
    // Initialize() が呼ばれるより前に AudioHardware に API を呼ばれてしまう不具合 (SIGLO-61827) があるため、
    // ここで AudioHardware が書いた値をタイミング依存で上書きしてしまうことがあるため無効化
    // コンストラクタで値は設定されており、Initialize / Finalize が実機上で起動中に呼ばれることもないので、実質的には設定しなくても振舞は変わらない
    // m_CurrentAudioOutputTarget = AudioTarget_Invalid;

    // ヘッドホン音量警告の表示回数をロード
    m_HeadphoneVolumeWarningDisplayedEventCount = nn::settings::system::GetHeadphoneVolumeWarningDisplayedEventCount();

    // システム出力マスターボリュームの初期値を設定
    m_SystemOutputMasterVolume = 1.0f;

    // オーバレイ通知送信コントローラを初期化
    m_OverlaySenderController.Initialize();

    // コーデック IC を経由した音声入出力機能の初期化
    if (m_HasCodecIc)
    {
        // 挿抜状態に応じて初期化
        m_IsTargetsConnected[AudioTarget_Speaker]   = true;
        m_IsTargetsConnected[AudioTarget_Headphone] = nn::audio::IsJackPlugged();
        // TODO: USB OutputDevice?

        // 音量設定をロード
        nn::settings::system::GetAudioVolume(&m_AudioVolumes[AudioTarget_Speaker], ConvertToAudioVolumeTarget(AudioTarget_Speaker));
        nn::settings::system::GetAudioVolume(&m_AudioVolumes[AudioTarget_Headphone], ConvertToAudioVolumeTarget(AudioTarget_Headphone));
        nn::settings::system::GetAudioVolume(&m_AudioVolumes[AudioTarget_UsbOutputDevice], ConvertToAudioVolumeTarget(AudioTarget_UsbOutputDevice));

        // 出力先を決定
        if(nn::audio::GetNumUacSpeakers() > 0)
        {
            nn::audio::NotifyUacSpeakerSwitch(true);
        }
        else
        {
            nn::audio::NotifyHeadphoneOutSelect(nn::audio::IsJackPlugged());
        }

        // アナログマイク入力の有効・無効を設定
        nn::audio::NotifyHeadphoneInEnabled(nn::audio::IsJackPlugged());

        // 音量値・ミュート状態を設定
        {
            const AudioTarget Targets[] = { AudioTarget_Speaker, AudioTarget_Headphone, AudioTarget_UsbOutputDevice };

            for(const auto Target : Targets)
            {
                const auto& AudioVolume = m_AudioVolumes[Target];
                SetTargetVolume(Target, AudioVolume.volume);
                SetTargetMute(Target, AudioVolume.flags.Test<nn::settings::system::AudioVolumeFlag::Mute>());
            }
        }

        // ヘッドホン出力レベルの設定
        SetHeadphoneOutputLevelMode(ConvertToHeadphoneOutputLevelMode(m_AudioVolumes[AudioTarget_Headphone]));

        // ボリュームボタン・ヘッドホン挿抜ハンドリング機能を初期化・開始
        m_UserInputHandler.Initialize();
    }
    else
    {
        // コーデック IC が存在しないプラットフォームでは常にどちらも false
        m_IsTargetsConnected[AudioTarget_Speaker]   = false;
        m_IsTargetsConnected[AudioTarget_Headphone] = false;
    }

    // PlayReportController を初期化
    m_PlayReportController.Initialize(nn::TimeSpan::FromSeconds(5));

    // 読み込んだ設定値を PlayReportController に通知
    // (出力先は AudioHardware が通知する)
    {
        const AudioTarget targets[] = { AudioTarget_Speaker, AudioTarget_Headphone, AudioTarget_UsbOutputDevice };

        for(auto target : targets)
        {
            const PlayReportAudioVolume playReportAudioVolume = { GetTargetVolume(target), IsTargetMute(target) };
            m_PlayReportController.NotifyAudioVolumeUpdateForVolume(ConvertToPlayReportAudioOutputTarget(target), playReportAudioVolume);
        }

        m_PlayReportController.NotifyAudioVolumeUpdateForHeadphonePowerLimit(GetHeadphoneOutputLevelMode() == HeadphoneOutputLevelMode_Normal);
    }

    // AsyncTaskController を初期化
    m_AsyncTaskController.Initialize(m_HasCodecIc, m_HasHda, m_IsUacEnabled);
}

void AudioControllerServiceImpl::Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (m_HasCodecIc)
    {
        // ボリュームボタン・ヘッドホン挿抜ハンドリング機能を停止・終了
        m_UserInputHandler.Finalize();
    }

    // オーバレイ通知送信コントローラを終了
    m_OverlaySenderController.Finalize();

    // PlayReportController を終了
    m_PlayReportController.Finalize();

    // AsyncTaskController を終了
    m_AsyncTaskController.Finalize();

    // 初期化済みフラグを無効にする
    m_IsInitialized = false;
}

void AudioControllerServiceImpl::Sleep() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    m_AsyncTaskController.Sleep();

    if (m_HasCodecIc)
    {
        m_UserInputHandler.Sleep();
    }
}

void AudioControllerServiceImpl::Wake() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    if (m_HasCodecIc)
    {
        m_UserInputHandler.Wake();

        // ヘッドホン出力レベルモードを復帰させる
        SetHeadphoneOutputLevelMode(GetHeadphoneOutputLevelMode());
    }

    m_AsyncTaskController.Wake();
}

void AudioControllerServiceImpl::SetTargetVolume(AudioTarget target, int8_t volume) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    auto& audioVolume = m_AudioVolumes[target];
    audioVolume.volume = volume;


    const auto hardwareVolume = VolumeMapper::GetHardwareVolume(IsTargetMute(target) ? 0 : audioVolume.volume, target);

    // Codec IC にボリューム値を設定
    if (m_HasCodecIc && IsTargetCodecDevice(target))
    {
        if(target == AudioTarget_Headphone)
        {
            nn::audio::SetHeadphoneVolume(hardwareVolume);
        }
        else if(target == AudioTarget_Speaker)
        {
            nn::audio::SetSpeakerVolume(hardwareVolume);
        }
    }
    else if(target == AudioTarget_UsbOutputDevice)
    {
        const auto HardwareVolumeMax = VolumeMapper::GetHardwareVolume(GetTargetVolumeMax(), target);
        const auto GainStep = -0.375f; // [dB]
        const auto LogVolume = (HardwareVolumeMax - hardwareVolume) * GainStep; // [dB]
        const auto LinearVolume = pow(10.f, LogVolume / 20.f); // Log to Linear
        // NN_DETAIL_AUDIOCTRL_INFO("[%s:%s:%4d] HardwareVolumeMax(%d) hardwareVolume(%d) LogVolume(%.2f) LinearVolume(%.2f)\n",
        // __FILE__, NN_CURRENT_FUNCTION_NAME,  __LINE__, HardwareVolumeMax, hardwareVolume, LogVolume, LinearVolume);
        nn::audio::SetUacOutputDeviceVolume(LinearVolume);
    }

    const PlayReportAudioVolume playReportAudioVolume = { volume, IsTargetMute(target) };
    m_PlayReportController.NotifyAudioVolumeUpdateForVolume(ConvertToPlayReportAudioOutputTarget(target), playReportAudioVolume);
}

int8_t AudioControllerServiceImpl::GetTargetVolume(AudioTarget target) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    return m_AudioVolumes[target].volume;
}

void AudioControllerServiceImpl::SetTargetMute(AudioTarget target, bool isMute) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    auto& audioVolume = m_AudioVolumes[target];
    audioVolume.flags.Set<nn::settings::system::AudioVolumeFlag::Mute>(isMute);

    const auto hardwareVolume = VolumeMapper::GetHardwareVolume(isMute ? 0 : audioVolume.volume, target);
    // NN_DETAIL_AUDIOCTRL_INFO("[%s:%s:%4d] hardwareVolume(%d)\n", __FILE__, NN_CURRENT_FUNCTION_NAME,  __LINE__, hardwareVolume);

    // Codec IC にボリューム値を設定
    if (m_HasCodecIc && IsTargetCodecDevice(target))
    {
        if(target == AudioTarget_Headphone)
        {
            nn::audio::SetHeadphoneVolume(hardwareVolume);
        }
        else if(target == AudioTarget_Speaker)
        {
            nn::audio::SetSpeakerVolume(hardwareVolume);
        }
    }
    else if(target == AudioTarget_UsbOutputDevice)
    {
        const auto HardwareVolumeMax = VolumeMapper::GetHardwareVolume(GetTargetVolumeMax(), target);
        const auto GainStep = -0.375f; // [dB]
        const auto LogVolume = (HardwareVolumeMax - hardwareVolume) * GainStep; // [dB]
        const auto LinearVolume = pow(10.f, LogVolume / 20.f); // Log to Linear
        // NN_DETAIL_AUDIOCTRL_INFO("[%s:%s:%4d] HardwareVolumeMax(%d) hardwareVolume(%d) LogVolume(%.2f) LinearVolume(%.2f)\n",
        // __FILE__, NN_CURRENT_FUNCTION_NAME,  __LINE__, HardwareVolumeMax, hardwareVolume, LogVolume, LinearVolume);
        nn::audio::SetUacOutputDeviceVolume(LinearVolume);
    }

    const PlayReportAudioVolume playReportAudioVolume = { GetTargetVolume(target), isMute };
    m_PlayReportController.NotifyAudioVolumeUpdateForVolume(ConvertToPlayReportAudioOutputTarget(target), playReportAudioVolume);
}

bool AudioControllerServiceImpl::IsTargetMute(AudioTarget target) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    return m_AudioVolumes[target].flags.Test<nn::settings::system::AudioVolumeFlag::Mute>();
}

void AudioControllerServiceImpl::SaveTargetVolumeSetting(AudioTarget target) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    nn::settings::system::SetAudioVolume(m_AudioVolumes[target], ConvertToAudioVolumeTarget(target));
}

int AudioControllerServiceImpl::GetTargetVolumeMin() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    return VolumeMin;
}

int AudioControllerServiceImpl::GetTargetVolumeMax() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    return VolumeMax;
}

void AudioControllerServiceImpl::SetTargetConnected(AudioTarget target, bool isConnected) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    m_IsTargetsConnected[target] = isConnected;
}

bool AudioControllerServiceImpl::IsTargetConnected(AudioTarget target) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    return m_IsTargetsConnected[target];
}

bool AudioControllerServiceImpl::SetDefaultTarget(AudioTarget target, nn::TimeSpanType fadeOutTime, nn::TimeSpanType fadeInTime) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

#if defined(NN_BUILD_CONFIG_SPEC_NX)
    if (!m_HasCodecIc && m_HasHda && target != AudioTarget_Tv)
    {
        return false;
    }
    if (m_HasCodecIc && !m_HasHda && target == AudioTarget_Tv)
    {
        return false;
    }
#endif  // defined(NN_BUILD_CONFIG_SPEC_NX)
    NN_UNUSED(fadeOutTime);
    NN_UNUSED(fadeInTime);
    m_DefaultOutput = target;
    bool isHda = (target == AudioTarget_Tv) ? true : false;
    ::nn::audio::NotifyOutputDeviceEvent(isHda, fadeOutTime, fadeInTime);
    return true;
}

AudioTarget AudioControllerServiceImpl::GetDefaultTarget() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    return m_DefaultOutput;
}

AudioOutputMode AudioControllerServiceImpl::GetOutputModeSetting(AudioTarget target) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    if (!IsValidAudioTargetForOutputMode(target))
    {
        return AudioOutputMode_Invalid;
    }
    return ConvertToAudioOutputMode(nn::settings::system::GetAudioOutputMode(ConvertToAudioOutputModeTarget(target)));
}

void AudioControllerServiceImpl::SetOutputModeSetting(AudioTarget target, AudioOutputMode mode) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    if (!IsValidAudioTargetForOutputMode(target))
    {
        return;
    }

    if (!IsValidOutputMode(mode))
    {
        return;
    }

    return nn::settings::system::SetAudioOutputMode(ConvertToAudioOutputMode(mode), ConvertToAudioOutputModeTarget(target));
}

AudioOutputMode AudioControllerServiceImpl::GetAudioOutputMode(AudioTarget target) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    if (!IsValidAudioTargetForOutputMode(target))
    {
        return AudioOutputMode_Invalid;
    }
    return m_AudioOutputMode[target];
}

void AudioControllerServiceImpl::SetAudioOutputMode(AudioTarget target, AudioOutputMode mode) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    if (!IsValidAudioTargetForOutputMode(target))
    {
        return;
    }

    if (!IsValidOutputMode(mode))
    {
        return;
    }

    m_AudioOutputMode[target] = mode;
    nn::audio::SetOutputMode(target, mode);
    ::nn::audio::NotifyOutputDeviceEvent();
}

void AudioControllerServiceImpl::SetForceMutePolicy(ForceMutePolicy policy) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    if(policy == ForceMutePolicy_SpeakerMuteOnHeadphoneUnplugged)
    {
        nn::settings::system::SetForceMuteOnHeadphoneRemoved(true);
    }
    else if(policy == ForceMutePolicy_Disable)
    {
        nn::settings::system::SetForceMuteOnHeadphoneRemoved(false);
    }
    else
    {
        NN_ABORT("Unexpected policy");
    }
}

ForceMutePolicy AudioControllerServiceImpl::GetForceMutePolicy() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    if(nn::settings::system::IsForceMuteOnHeadphoneRemoved())
    {
        return ForceMutePolicy_SpeakerMuteOnHeadphoneUnplugged;
    }
    else
    {
        return ForceMutePolicy_Disable;
    }
}

void AudioControllerServiceImpl::SetOutputTarget(AudioTarget target) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    if (!IsValidTargetForOutputTarget(target))
    {
        return;
    }

    if(target == AudioTarget_Speaker || target == AudioTarget_Headphone)
    {
        SetTargetVolume(target, m_AudioVolumes[target].volume);
    }

    if (m_HasCodecIc)
    {
        m_UserInputHandler.SetHeadphoneJackDetectionEnabled(target == AudioTarget_Invalid);
        m_UserInputHandler.SetVolumeButtonEnabled(target == AudioTarget_Speaker || target == AudioTarget_Headphone);
        nn::audio::NotifyOutputDeviceForceSwitchEvent(target);
    }
}

void AudioControllerServiceImpl::SetInputTargetForceEnabled(bool isEnabled) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    nn::audio::NotifyInputDeviceForceEnable(isEnabled);
}

void AudioControllerServiceImpl::SetHeadphoneOutputLevelMode(HeadphoneOutputLevelMode mode) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    NN_SDK_REQUIRES(mode == HeadphoneOutputLevelMode_Normal || mode == HeadphoneOutputLevelMode_HighPower);
    auto& audioVolume = m_AudioVolumes[AudioTarget_Headphone];

    audioVolume.flags.Set<nn::settings::system::AudioVolumeFlag::HighPower>(mode == HeadphoneOutputLevelMode_HighPower);
    if(m_HasCodecIc)
    {
        nn::audio::SetHeadphoneAmplifierGain(GetHardwareHeadphoneAmplifierGain(mode));
    }

    m_PlayReportController.NotifyAudioVolumeUpdateForHeadphonePowerLimit(mode == HeadphoneOutputLevelMode_Normal);

    SaveTargetVolumeSetting(AudioTarget_Headphone);
}

HeadphoneOutputLevelMode AudioControllerServiceImpl::GetHeadphoneOutputLevelMode() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    if(m_AudioVolumes[AudioTarget_Headphone].flags.Test<nn::settings::system::AudioVolumeFlag::HighPower>())
    {
        return HeadphoneOutputLevelMode_HighPower;
    }
    else
    {
        return HeadphoneOutputLevelMode_Normal;
    }
}

void AudioControllerServiceImpl::AcquireAudioVolumeUpdateEventForPlayReport(nn::os::NativeHandle* outHandle) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    m_PlayReportController.AcquireAudioVolumeUpdateEvent(outHandle);
}

void AudioControllerServiceImpl::AcquireAudioOutputDeviceUpdateEventForPlayReport(nn::os::NativeHandle* outHandle) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    m_PlayReportController.AcquireAudioOutputDeviceUpdateEvent(outHandle);
}

void AudioControllerServiceImpl::NotifyAudioOutputTarget(AudioTarget target) NN_NOEXCEPT
{
    // TODO: Workaround for SIGLO-61827
    // std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    //
    // NN_SDK_ASSERT(m_IsInitialized);

    m_PlayReportController.NotifyOutputDeviceUpdate(ConvertToPlayReportAudioOutputTarget(target));
}

void AudioControllerServiceImpl::NotifyUnsupportedUsbOutputDeviceAttached() NN_NOEXCEPT
{
    // TODO: Workaround for SIGLO-61827
    // std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    //
    // NN_SDK_ASSERT(m_IsInitialized);

    m_PlayReportController.NotifyOutputDeviceUpdate(PlayReportAudioOutputTarget_UnsupportedUsbOutputDevice);
}

PlayReportAudioOutputTarget AudioControllerServiceImpl::GetAudioOutputTargetForPlayReport() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    return m_PlayReportController.GetAudioOutputTarget();
}

void AudioControllerServiceImpl::GetAudioVolumeDataForPlayReport(PlayReportAudioVolumeData* pOutData) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    return m_PlayReportController.GetAudioVolumeData(pOutData);
}

void AudioControllerServiceImpl::NotifyHeadphoneVolumeWarningDisplayedEvent() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    ++m_HeadphoneVolumeWarningDisplayedEventCount;
    nn::settings::system::SetHeadphoneVolumeWarningDisplayedEventCount(m_HeadphoneVolumeWarningDisplayedEventCount);
}

int AudioControllerServiceImpl::GetHeadphoneVolumeWarningDisplayedEventCount() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    return m_HeadphoneVolumeWarningDisplayedEventCount;
}

int AudioControllerServiceImpl::GetHeadphoneRs1VolumeMax(HeadphoneOutputLevelMode mode) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    switch (mode)
    {
    case HeadphoneOutputLevelMode_Normal:
        return VolumeMapper::Rs1VolumeMaxForHeadphoneNormalMode;
    case HeadphoneOutputLevelMode_HighPower:
        return VolumeMapper::Rs1VolumeMaxForHeadphoneHighPowerMode;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

OverlaySenderController& AudioControllerServiceImpl::GetOverlaySenderController() NN_NOEXCEPT
{
    return m_OverlaySenderController;
}

bool AudioControllerServiceImpl::IsHeadphoneVolumeWarningTransmissionNeeded() NN_NOEXCEPT
{
    // IAAD-1368 参照
    return GetHeadphoneOutputLevelMode() == HeadphoneOutputLevelMode_HighPower
        && GetTargetVolume(AudioTarget_Headphone) > GetHeadphoneRs1VolumeMax(nn::audioctrl::HeadphoneOutputLevelMode_HighPower)
        && GetHeadphoneVolumeWarningDisplayedEventCount() == 0;
}

void AudioControllerServiceImpl::SetSystemOutputMasterVolume(float volume) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    m_SystemOutputMasterVolume = volume;

    const auto curveDurationInMicroseconds = 0u;
    nn::audio::SetOutputMasterVolume(volume, curveDurationInMicroseconds);
}

float AudioControllerServiceImpl::GetSystemOutputMasterVolume() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    return m_SystemOutputMasterVolume;
}

bool AudioControllerServiceImpl::NeedsToUpdateHeadphoneVolume(bool* pOutIsHighPowerAvailable, int8_t* pOutVolume, bool isRestricted) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    return nn::settings::system::NeedsToUpdateHeadphoneVolume(pOutIsHighPowerAvailable, pOutVolume, isRestricted);
}

}}}}  // namespace nn::audioctrl::server::detail
