﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <atomic>
#include <mutex>

#include <nn/sf/sf_ObjectFactory.h>
#include <nn/applet/applet_Apis.h>
#include <nn/os/os_Mutex.h>

#include <nn/audio/audio_Result.h>
#include <nn/audio/audio_Device.h>
#include "../server/audio_AudioRendererManagerImpl.h"
#include "../common/audio_UpdateDataHeader.h"

#include "audio_CreateAudioRendererManager.h"

namespace nn { namespace audio {

namespace {
bool g_IsUsbAudioDeviceOutputDeviceVolumeEnabled;
#if defined(NN_BUILD_CONFIG_OS_WIN)
const bool IsDfc = true;
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
const bool IsDfc = false;
#else
#error unsupported platform
#endif

nn::sf::SharedPointer<detail::IAudioDevice> GetAudioDeviceService()
{
#if 1
    auto id = nn::applet::AppletResourceUserId::GetInvalidId();
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    id = applet::GetAppletResourceUserId();
#endif

    auto audioRendererManager = CreateAudioRendererManager(IsDfc);
    nn::sf::SharedPointer<detail::IAudioDevice> serviceRef;
    NN_ABORT_UNLESS_RESULT_SUCCESS(audioRendererManager->GetAudioDeviceServiceWithRevisionInfo(&serviceRef, id, common::GetCurrentRevision()));
    return serviceRef;
#else // DEATH_TEST_IF_SUPPORTED を通ると以下の処理では不正な挙動となる
    NN_FUNCTION_LOCAL_STATIC(std::atomic<bool>, initialized, (false));
    NN_FUNCTION_LOCAL_STATIC(nn::sf::SharedPointer<detail::IAudioDevice>, serviceRef);
    NN_FUNCTION_LOCAL_STATIC(nn::os::Mutex, mutex, (false));

    std::lock_guard<nn::os::Mutex> lock(mutex);

    if (initialized == false)
    {
        auto id = nn::applet::AppletResourceUserId::GetInvalidId();
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
        id = applet::GetAppletResourceUserId();
#endif

        auto audioRendererManager = CreateAudioRendererManager(IsDfc);
        NN_ABORT_UNLESS_RESULT_SUCCESS(audioRendererManager->GetAudioDeviceService(&serviceRef, id));
        initialized = true;
    }

    return serviceRef;
#endif
}

}

int ListAudioDeviceName(AudioDeviceName* outDeviceNames, int count) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outDeviceNames);
    NN_SDK_ASSERT_GREATER_EQUAL(count, 0);

    nn::sf::OutBuffer names(reinterpret_cast<char*>(outDeviceNames), sizeof(AudioDeviceName) * count);
    int32_t amountRet = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetAudioDeviceService()->ListAudioDeviceNameAuto(names, &amountRet));
    return amountRet;
}

void SetAudioDeviceOutputVolume(const AudioDeviceName* pDeviceName, float volume) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDeviceName);
    NN_SDK_REQUIRES_MINMAX(volume, AudioDeviceOutputVolumeMin, AudioDeviceOutputVolumeMax);

    if(strncmp(pDeviceName->name, "AudioUsbDeviceOutput", nn::audio::AudioDeviceName::NameLength) == 0)
    {
        g_IsUsbAudioDeviceOutputDeviceVolumeEnabled = true;
    }

    if(!g_IsUsbAudioDeviceOutputDeviceVolumeEnabled && strncmp(pDeviceName->name, "AudioStereoJackOutput", nn::audio::AudioDeviceName::NameLength) == 0)
    {
        nn::sf::InBuffer nameIn("AudioUsbDeviceOutput", AudioDeviceName::NameLength);
        NN_ABORT_UNLESS_RESULT_SUCCESS(GetAudioDeviceService()->SetAudioDeviceOutputVolumeAuto(nameIn, volume));
    }

    nn::sf::InBuffer nameIn(pDeviceName->name, AudioDeviceName::NameLength);
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetAudioDeviceService()->SetAudioDeviceOutputVolumeAuto(nameIn, volume));
}

float GetAudioDeviceOutputVolume(const AudioDeviceName* pDeviceName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDeviceName);

    nn::sf::InBuffer nameIn(pDeviceName->name, AudioDeviceName::NameLength);
    float volume;

    NN_ABORT_UNLESS_RESULT_SUCCESS(GetAudioDeviceService()->GetAudioDeviceOutputVolumeAuto(nameIn, &volume));
    return volume;
}

void GetActiveAudioDeviceName(AudioDeviceName* pOutDeviceName) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    nn::sf::OutBuffer name(reinterpret_cast<char*>(pOutDeviceName), sizeof(AudioDeviceName));
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetAudioDeviceService()->GetActiveAudioDeviceNameAuto(name));
#else
    NN_UNUSED(pOutDeviceName);
    return;
#endif
}

void AcquireAudioDeviceSwitchNotification(nn::os::SystemEvent* pOutSystemEvent) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    nn::sf::NativeHandle systemEventHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetAudioDeviceService()->QueryAudioDeviceSystemEvent(&systemEventHandle));
    pOutSystemEvent->AttachReadableHandle(systemEventHandle.GetOsHandle(), systemEventHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    systemEventHandle.Detach();
#else
    NN_UNUSED(pOutSystemEvent);
    return;
#endif
}

nn::Result AcquireAudioDeviceNotificationForInput(nn::os::SystemEvent* pOutSystemEvent) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_FUNCTION_LOCAL_STATIC(bool, called, = false);
    NN_RESULT_THROW_UNLESS(!called, nn::audio::ResultOperationFailed());

    nn::sf::NativeHandle systemEventHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetAudioDeviceService()->QueryAudioDeviceInputEvent(&systemEventHandle));
    pOutSystemEvent->AttachReadableHandle(systemEventHandle.GetOsHandle(), systemEventHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    systemEventHandle.Detach();

    called = true;
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(pOutSystemEvent);
    NN_RESULT_THROW(nn::audio::ResultNotSupported());
#endif
}

nn::Result AcquireAudioDeviceNotificationForOutput(nn::os::SystemEvent* pOutSystemEvent) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_FUNCTION_LOCAL_STATIC(bool, called, = false);
    NN_RESULT_THROW_UNLESS(!called, nn::audio::ResultOperationFailed());

    nn::sf::NativeHandle systemEventHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetAudioDeviceService()->QueryAudioDeviceOutputEvent(&systemEventHandle));
    pOutSystemEvent->AttachReadableHandle(systemEventHandle.GetOsHandle(), systemEventHandle.IsManaged(), nn::os::EventClearMode_AutoClear);
    systemEventHandle.Detach();

    called = true;
    NN_RESULT_SUCCESS;
#else
    NN_UNUSED(pOutSystemEvent);
    NN_RESULT_THROW(nn::audio::ResultNotSupported());
#endif
}

int GetActiveChannelCount() NN_NOEXCEPT
{
    int32_t channelCount = 0;
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetAudioDeviceService()->GetActiveChannelCount(&channelCount));
#endif
    return channelCount;
}

int GetActiveAudioDeviceChannelCountForOutput() NN_NOEXCEPT
{
    int32_t channelCount = 0;
#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetAudioDeviceService()->GetActiveChannelCount(&channelCount));
#endif
    return channelCount;
}

#if defined(NN_AUDIO_ENABLE_GET_AUDIO_SYSTEM_MASTER_VOLUME_SETTING_FOR_USERS)
// TODO: add the following declaration to an appropriate header file when exposing this API to users
//
// /**
//  * @brief           出力先デバイスの本体音量設定値を取得します。
//  * @param[in]       pDeviceName 本体音量設定値を取得する出力先デバイスの名前
//  * @return          pDeviceName に指定されたデバイスに設定されている本体音量設定値を返します。
//  * @pre
//  * - pDeviceName != nullptr
//  * - pDeviceName は nn::audio::ListAudioDeviceName() で取得されるデバイスに含まれる
//  *
//  * @details
//  * 本体音量設定は、本体音量ボタンや、HOME メニューなどで設定可能な音量です。
//  * 本体音量設定値を持たないデバイスの場合は、常に 1.0f を返します。
//  * 出力先デバイスに対してミュート設定が行われている場合は 0.0f を返します。
//  *
//  * @platformbegin{Windows}
//  * - Windows 環境では本 API はサポートされていません。 常に 1.0f を返します。
//  *
//  * @platformend
//  */
// float GetAudioSystemMasterVolumeSetting(const AudioDeviceName* pDeviceName) NN_NOEXCEPT;
//
float GetAudioSystemMasterVolumeSetting(const AudioDeviceName* pDeviceName) NN_NOEXCEPT
{
    float volume = 0.0f;
    nn::sf::InBuffer name(pDeviceName->name, AudioDeviceName::NameLength);
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetAudioDeviceService()->GetAudioSystemMasterVolumeSetting(&volume, name));
    return volume;
}
#endif  // defined(NN_AUDIO_ENABLE_GET_AUDIO_SYSTEM_MASTER_VOLUME_SETTING_FOR_USERS)

}} // namespace nn::audio
