﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_Mutex.h>
#include <nn/os/os_Semaphore.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_Thread.h>
#include <nn/applet/applet_Types.h>
#include <nn/audio/audio_AudioInTypes.h>
#include <nn/cduac/cduac_Spec.h>
#include <nn/cduac/cduac_Api.h>
#include <nne/audio/audio.h>
#include "audio_UacTypes.h"
#include "../audio_AudioSessionTypes.h"
#include "audio_UacConnectionManager.h"
#include "audio_UacInDevice.h"
#include "audio_UacOutDevice.h"

#define NN_AUDIO_ALIGNAS_UAC_BUFFER NN_OS_ALIGNAS_THREAD_STACK

namespace nn {
namespace audio {
namespace detail {

class UacManager
{
enum StateChangeReason
{
    StateChangeReason_Invalid = 0,
    StateChangeReason_SetSpeakerSource,
    StateChangeReason_StartMicrophone,
    StateChangeReason_StopMicrophone,
    StateChangeReason_OpenMicrophone,
    StateChangeReason_CloseMicrophone,
};

struct StateChangeArg
{
    StateChangeArg() : mutex(false), reason(StateChangeReason_Invalid), uacId(InvalidUacId) {}
    nn::os::Mutex mutex;
    StateChangeReason reason;
    UacSessionId uacId;
    union Parameters
    {
        struct OpenMicrophone
        {
            server::SessionFormat format;
            const char* name;
            const nn::applet::AppletResourceUserId* aruid;
            int sessionId;
            UacSessionId returnId;
        } openMicrophone;
        struct SetSource
        {
            nne::audio::gmix::Session* session;
            nn::os::Semaphore* semaphore;
        } setSource;
    } parameters;
};

public:
    static UacManager& GetInstance() NN_NOEXCEPT;

    void Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    void Sleep() NN_NOEXCEPT;
    void Wake()  NN_NOEXCEPT;

    //Called by AudioInSessionAbstractor Open/Close
    nn::Result OpenUacInSession(UacSessionId* outUacId, const char* pDeviceName, server::SessionFormat& format, int sessionId, const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;
    nn::Result CloseUacInSession(UacSessionId uacId) NN_NOEXCEPT;

    //Called by AudioInSessionAbstractor Start/Stop
    nn::Result StartUacInSession(UacSessionId uacId) NN_NOEXCEPT;
    nn::Result StopUacInSession(UacSessionId uacId) NN_NOEXCEPT;

    UacInDevice* GetUacInDevice(UacSessionId uacId) NN_NOEXCEPT;
    bool IsUacInDeviceSupported(nn::cduac::Interface* pInterface, nn::cduac::Parser* pParser) NN_NOEXCEPT;
    bool IsUacOutDeviceSupported(nn::cduac::Interface* pInterface, nn::cduac::Parser* pParser) NN_NOEXCEPT;
    int ListUacIns(AudioInInfo* outAudioIns, int count) NN_NOEXCEPT;
    bool IsUacInName(const char* name) NN_NOEXCEPT;

    void SetSpeakerSource(nne::audio::gmix::Session* session,nn::os::Semaphore* semaphore) NN_NOEXCEPT;

    void SetSpeakerVolume(float volume) NN_NOEXCEPT;
    nn::os::EventType* GetSpeakerAttachEvent() NN_NOEXCEPT;
    nn::os::EventType* GetSpeakerDetachEvent() NN_NOEXCEPT;
    nn::os::EventType* GetUnsupportedSpeakerAttachEvent() NN_NOEXCEPT;
    int GetNumSpeakers() NN_NOEXCEPT;

private:
    NN_DISALLOW_COPY(UacManager);
    NN_DISALLOW_MOVE(UacManager);
    UacManager();
    virtual ~UacManager();
    void StartThread() NN_NOEXCEPT;
    void StopThread() NN_NOEXCEPT;

    void InitializeImpl() NN_NOEXCEPT;
    void FinalizeImpl() NN_NOEXCEPT;

    static void DeviceManagerUpdateThreadFunc(void* arg);
    void DeviceManagerUpdateThreadFuncImpl() NN_NOEXCEPT;
    static void DeviceManagerAttachThreadFunc(void* arg);
    void DeviceManagerAttachThreadFuncImpl() NN_NOEXCEPT;
    void UpdateUnlinkedMicrophones(UacSessionId id) NN_NOEXCEPT;
    void UpdateMicrophone(UacSessionId id) NN_NOEXCEPT;
    void UpdateSpeaker(UacSessionId id) NN_NOEXCEPT;
    void LinkMicrophone(UacSessionId id) NN_NOEXCEPT;
    bool UnlinkMicrophone(UacSessionId id, bool defer) NN_NOEXCEPT;
    void UnlinkMicrophoneImpl(UacSessionId id) NN_NOEXCEPT;
    bool LinkSpeaker(UacSessionId id, bool defer) NN_NOEXCEPT;
    void LinkSpeakerImpl(UacSessionId id) NN_NOEXCEPT;
    bool UnlinkSpeaker(UacSessionId id, bool defer) NN_NOEXCEPT;
    void UnlinkSpeakerImpl(UacSessionId id) NN_NOEXCEPT;
    void LinkProcessSemaphore() NN_NOEXCEPT;
    void UnlinkProcessSemaphore() NN_NOEXCEPT;
    void UnlinkAllDevices() NN_NOEXCEPT;
    void PrintStatus() NN_NOEXCEPT;

    void HandleMicrophoneUpdate(int id) NN_NOEXCEPT;
    void HandleSpeakerUpdate(int id) NN_NOEXCEPT;
    bool HandleAttach() NN_NOEXCEPT;
    bool HandleDetach(int id) NN_NOEXCEPT;
    void HandleGmixProcess() NN_NOEXCEPT;
    void HandleStateChange() NN_NOEXCEPT;

    bool ConnectSpeaker(bool defer) NN_NOEXCEPT;
    bool DisconnectSpeaker(ConnectionId connectId, bool defer) NN_NOEXCEPT;
    void DisconnectAllSpeakers() NN_NOEXCEPT;

    void DoSynchronizedStateChange() NN_NOEXCEPT;
    void SetSpeakerSourceImpl(nne::audio::gmix::Session* session,nn::os::Semaphore* semaphore) NN_NOEXCEPT;
    UacSessionId OpenUacInSessionImpl(const char* pDviceName, server::SessionFormat& format, int sessionId, const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT;
    void CloseUacInSessionImpl(UacSessionId uacId) NN_NOEXCEPT;
    void StartUacInSessionImpl(UacSessionId uacId) NN_NOEXCEPT;
    void StopUacInSessionImpl(UacSessionId uacId) NN_NOEXCEPT;

    void HandleDeferredLinks() NN_NOEXCEPT;
private:
    static const int InSessionMax = 2;
    static const int OutSessionMax = 1;
    static const int StackSize = 4096 * 4;

    NN_OS_ALIGNAS_THREAD_STACK int8_t m_UpdateThreadStack[StackSize];
    NN_OS_ALIGNAS_THREAD_STACK int8_t m_AttachThreadStack[StackSize];
    nn::os::Semaphore* m_pProcessSemaphore;
    nne::audio::gmix::Session* m_pGmixSession;
    UacConnectionManager m_ConnectionManager;
    nn::os::ThreadType m_UpdateThread;
    nn::os::ThreadType m_AttachThread;
    nn::os::SystemEventType m_AttachEvent;
    nn::os::EventType m_StateChangeEvent;
    nn::os::EventType m_StateChangeCompleteEvent;
    nn::os::EventType m_CloseAttachThreadEvent;
    nn::os::EventType m_CheckDeferredLinksEvent;
    nn::os::EventType m_DeferredLinksCompleteEvent;
    nn::os::MultiWaitType m_UpdateMultiWait;
    nn::os::MultiWaitType m_AttachMultiWait;
    nn::os::MultiWaitHolderType m_MicrophoneUpdateHolder[InSessionMax];
    nn::os::MultiWaitHolderType m_SpeakerUpdateHolder[OutSessionMax];
    nn::os::MultiWaitHolderType m_AttachHolder;
    nn::os::MultiWaitHolderType m_GmixProcessHolder;
    nn::os::MultiWaitHolderType m_StateChangeHolder;
    nn::os::MultiWaitHolderType m_CloseAttachHolder;
    nn::os::MultiWaitHolderType m_CheckDeferredLinksHolder;
    nn::os::EventType m_SpeakerAttachEvent;
    nn::os::EventType m_SpeakerDetachEvent;
    nn::os::EventType m_UnsupportedSpeakerAttachEvent;
    UacInDevice m_InDevice[InSessionMax];
    UacOutDevice m_OutDevice[OutSessionMax];
    ConnectionId m_MicrophoneDeviceIdToConnectionId[InSessionMax];
    ConnectionId m_SpeakerDeviceIdToConnectId[OutSessionMax];
    nn::os::Mutex m_Mutex;
    int m_NumSpeakers;
    int m_InInitializedCount;
    int m_InStartedCount;
    int m_LastGmixPosition;
    StateChangeArg m_StateChangeArg;
    bool m_IsMicrophoneLinked[InSessionMax];
    bool m_IsMicrophoneUnlinkDeferred[InSessionMax];
    bool m_IsSpeakerLinked[OutSessionMax];
    bool m_IsSpeakerLinkDeferred[OutSessionMax];
    bool m_IsSpeakerUnlinkDeferred[OutSessionMax];
    bool m_IsInitialized;
    bool m_IsAwake;
    bool m_IsGmixLinked;
    bool m_CanConnectSpeaker;
    float m_SpeakerVolume;

};

}  // namespace detail
}  // namespace audio
}  // namespace nn
