﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_UacInDevice.h"
#include <nne/audio/audio.h>
#include <nn/audio/detail/audio_Log.h>
#include "../audio_AppletVolumeManager.h"

namespace nn {
namespace audio {
namespace detail {

namespace
{
    static const uint32_t g_UacSupportedSampleRates[] =
    {
        48000,
        44100,
        24000,
        22050,
        16000,
        12000,
        11025,
        8000
    };
}

UacInDevice::UacInDevice() NN_NOEXCEPT
    : m_Mutex(true)
    , m_pHost(nullptr)
    , m_pParser(nullptr)
    , m_pInterface(nullptr)
    , m_pStateChangeEvent(nullptr)
    , m_Name(nullptr)
    , m_AppletId(nn::applet::AppletResourceUserId::GetInvalidId())
    , m_NextUsbFrame(0)
    , m_UacBufferIndex(0)
    , m_UacBufferReadIndex(0)
    , m_OutCount(0)
    , m_AltSetting(1)
    , m_IsOpen(false)
    , m_IsStarted(false)
    , m_IsInitialized(false)
    , m_IsDeviceStarted(false)
    , m_NeedResync(false)
    , m_SkipSubmitBuffer(false)
{
    for(int i = 0; i < UacBufferMaxCount; ++i)
    {
        memset(m_UacBuffer[i], 0, UacBufferSize);
    }
    for (int i = 0; i < UacFrameDurationMs; ++i)
    {
        m_InputBytes[i] = UacMaxSamplesPerMs * UacMaxChannelCount * UacSampleSize;
    }

}

int UacInDevice::SetEndpointSamplingRate(nn::cduac::Interface* interface, nn::cduac::Parser* parser) NN_NOEXCEPT
{
    nn::Result result;

    auto profile = interface->GetInterfaceProfile();

    if(!profile)
    {
        return 0;
    }

    auto altSetting = FindAltSetting(interface, parser);
    nn::usb::UsbEndpointDescriptor *pEndpoint = parser->GetEndpoint(interface->GetInterfaceProfile(), altSetting, 0);

    if (pEndpoint)
    {
        auto format = parser->GetAudioStreamingFormatType(profile, altSetting);
        if(format && format->bSamFreqType == 1)
        {
            return GetEndpointSamplingRateFromFormatType(format);
        }
        else
        {
            nn::cduac::AudioSamplingFrequency audioSamplingFrequencyRequest;
            auto sampleRate = GetEndpointSamplingRateFromFormatType(format);
            SetSamplingFrequency(&audioSamplingFrequencyRequest, sampleRate);

            interface->SetEndpointControl(
                                            nn::cduac::Request_SetCur,
                                            nn::cduac::Endpoint_SamplingFrequencyControl,
                                            pEndpoint->bEndpointAddress,
                                            sizeof(nn::cduac::AudioSamplingFrequency),
                                            &audioSamplingFrequencyRequest
                                            );
            return sampleRate;
        }
    }
    return 0;
}

int UacInDevice::GetEndpointSamplingRateFromFormatType(nn::cduac::AudioStreamingFormatType* format) NN_NOEXCEPT
{
    for (int i = 0; i < sizeof(g_UacSupportedSampleRates) / sizeof(g_UacSupportedSampleRates[0]); i++)
    {
        for(uint8_t j = 0; j < format->bSamFreqType; ++j)
        {
            auto uacFreq = format->tSamFreq[j];
            auto sampleRate = uacFreq.n1 + (uacFreq.n256 * 256) + (uacFreq.n65536 * 65536);
            if(g_UacSupportedSampleRates[i] == sampleRate)
            {
                return sampleRate;
            }
        }
    }
    return 0;
}

bool UacInDevice::IsSupported(nn::cduac::Interface* interface, nn::cduac::Parser* parser) NN_NOEXCEPT
{
    auto altSetting = FindAltSetting(interface, parser);
    auto formatType = parser->GetAudioStreamingFormatType(interface->GetInterfaceProfile(), altSetting);
    if(formatType)
    {
        const auto sampleRate = GetEndpointSamplingRateFromFormatType(formatType);
        if(sampleRate != 0)
        {
            MixerFormat inFormat;
            inFormat.sampleRate = sampleRate;
            inFormat.channelCount = formatType->bNrChannles;
            inFormat.sampleFormat = UacFormatToAudioFormat(formatType->bFormatType, formatType->bBitResolution);
            return Mixer::IsInFormatSupported(inFormat);
        }
    }
    return false;
}

uint8_t UacInDevice::FindAltSetting(nn::cduac::Interface* interface, nn::cduac::Parser* parser) NN_NOEXCEPT
{
    int altSetting = -1;
    for (uint8_t i = 1; i < nn::usb::HsLimitMaxAltInterfaceSettingsCount; i++)
    {
        auto profile = interface->GetInterfaceProfile();
        nn::usb::UsbInterfaceDescriptor *pUsbInterfaceDescriptor = parser->GetInterface(profile, i);

        if (pUsbInterfaceDescriptor)
        {
            nn::cduac::AudioStreamingGeneral* general     = parser->GetAudioStreamingGeneral(profile, i);
            nn::cduac::AudioStreamingFormatType* format  = parser->GetAudioStreamingFormatType(profile, i);

            if (general && format)
            {
                if (general->wFormatTag == nn::cduac::AudioDataFormat::AudioDataFormat_Pcm && format->bBitResolution == (UacSampleSize * 8)) // PCM format
                {
                    if (format->bNrChannles == 1)
                    {
                        return i;
                    }
                    else if (format->bNrChannles == 2)
                    {
                        altSetting = i;
                    }
                }
            }
        }
    }
    return altSetting;
}

bool UacInDevice::Update() NN_NOEXCEPT
{
    UacReadResult result;

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if(!AppletVolumeManager::IsRunning(nn::audio::AppletVolumeManager::SessionType_AudioIn, GetSessionId()))
    {
        if(m_IsDeviceStarted && IsInitialized())
        {
            StopDeviceInternal(true);
        }
        return false;
    }
    if(IsInitialized())
    {
        if(!m_IsDeviceStarted)
        {
            StartDeviceInternal();
            SubmitUacBuffer();
            return !nn::os::TryWaitSystemEvent(m_pStateChangeEvent);
        }
        auto buffer = ReadFromDevice(&result);
        CopyToUserBuffer(buffer, result);
        if(m_NeedResync)
        {
            ResyncUacDevice(false);
        }
        if(!m_SkipSubmitBuffer)
        {
            SubmitUacBuffer();
        }
        m_SkipSubmitBuffer = false;
        return !nn::os::TryWaitSystemEvent(m_pStateChangeEvent);
    }
    else
    {
        //Device is disconnected. Write zeroes to user buffer.
        if(m_IsDeviceStarted)
        {
            StopDeviceInternal(false);
        }
        CopyToUserBuffer(nullptr, UacReadResult_Failure);
        return false;
    }
}

//Plug/Unplug
void UacInDevice::InitializeDevice(const char* deviceName, nn::cduac::Interface* interface, nn::cduac::Host* host, nn::cduac::Parser* parser, const nn::applet::AppletResourceUserId& aruid) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    m_pInterface = interface;
    m_pHost = host;
    m_pParser = parser;
    m_Name = deviceName;
    m_UacBufferIndex = 0;
    m_UacBufferReadIndex = 0;
    m_AppletId = aruid;
    m_Gain = 1.0f;
    for(int i = 0; i < UacBufferMaxCount; ++i)
    {
        memset(m_UacBuffer[i], 0, UacBufferSize);
    }
    m_IsInitialized = true;
    m_pStateChangeEvent = m_pInterface->GetStateChangeEvent();
}

void UacInDevice::FinalizeDevice() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    m_IsInitialized = false;
    m_Name = nullptr;
    m_pParser = nullptr;
    m_pHost = nullptr;
    m_pInterface = nullptr;
    m_AppletId = nn::applet::AppletResourceUserId::GetInvalidId();
}

bool UacInDevice::OpenSession(server::SessionFormat& format, int sessionId) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    m_OutFormat.sampleRate = format.sampleRate;
    m_OutFormat.channelCount = format.channelCount;
    m_OutFormat.sampleFormat = format.format;
    m_SessionId = sessionId;

    auto supported = Mixer::IsOutFormatSupported(m_OutFormat);
    m_IsOpen = supported;
    return supported;
}

void UacInDevice::CloseSession() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    m_IsOpen = false;
    StopDevice();
    ClearBuffers();
}

//Game Start/Stop
void UacInDevice::StartDevice() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if(!m_IsStarted)
    {
        m_IsStarted = true;
        StartDeviceInternal();
    }
}

void UacInDevice::StopDevice() NN_NOEXCEPT
{
    if(m_IsStarted)
    {
        {
            std::lock_guard<nn::os::Mutex> lock(m_Mutex);
            m_IsStarted = false;

            if(m_IsDeviceStarted)
            {
                StopDeviceInternal(m_IsDeviceInitialized);
                m_IsDeviceStarted = false;
            }
        }
    }
}

bool UacInDevice::AppendBuffer(void* buffer, size_t bufferSize, nn::dd::ProcessHandle clientProcessHandle) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_UNUSED(clientProcessHandle);
    if(m_IsStarted)
    {
        return m_Mixer.AppendBuffer(buffer, bufferSize);
    }
    else
    {
        return false;
    }
}

void* UacInDevice::GetBufferAddress() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    return m_Mixer.GetBufferAddress();
}

bool UacInDevice::IsStarted() NN_NOEXCEPT
{
    return m_IsStarted;
}

bool UacInDevice::IsInitialized() NN_NOEXCEPT
{
    return m_IsInitialized;
}

const char* UacInDevice::GetName() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);
    return m_Name;
}

int UacInDevice::GetSessionId() NN_NOEXCEPT
{
    return m_SessionId;
}

bool UacInDevice::IsDeviceInitialized() NN_NOEXCEPT
{
    return m_IsDeviceInitialized;
}

void UacInDevice::StartDeviceInternal() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if(!m_IsDeviceInitialized)
    {
        m_AltSetting = FindAltSetting(m_pInterface, m_pParser);
        auto profile = m_pInterface->GetInterfaceProfile();
        nn::usb::UsbEndpointDescriptor* pUsbEndpointDescriptor = m_pParser->GetEndpoint(profile, m_AltSetting, 0);
        m_pInputTerminal = m_pParser->GetInputTerminal(m_pInterface->GetInterfaceProfile());
        m_pFeatureUnit   = m_pParser->GetFeatureUnit(m_pInterface->GetInterfaceProfile());
        m_MaxPacketSize = pUsbEndpointDescriptor->wMaxPacketSize;

        SetUacControls();

        auto result = m_pInterface->SetAltSetting(
                                                m_AltSetting,                       // alternate setting to set
                                                pUsbEndpointDescriptor,                  // pointer to UsbEndpointDescriptor
                                                UacBufferMaxCount,  // maximum URB count
                                                (UacMaxSamplesPerFrame * UacSampleSize)        // maximum outstanding transfer size total
                                                );
        if (!result.IsSuccess())
        {
            if(result <= nn::usb::ResultInterfaceInvalid())
            {
                //Device was disconnected
                return;
            }
            else
            {
                NN_DETAIL_AUDIO_ERROR("SetAltSetting failed module %d description %d\n", result.GetModule(), result.GetDescription());
                return;
            }
        }

        result = m_pInterface->RegisterBuffer(m_UacBuffer, sizeof(m_UacBuffer), m_ReportBuffer, sizeof(m_ReportBuffer));
        if (!result.IsSuccess())
        {
            NN_DETAIL_AUDIO_ERROR("RegisterBuffer failed module %d description %d\n", result.GetModule(), result.GetDescription());
            m_pInterface->SetAltSetting(0, pUsbEndpointDescriptor, 0, 0);
        }

        SetUacEndpoint();
    }

    m_UacBufferIndex = 0;
    m_UacBufferReadIndex = 0;
    m_Mixer.ResetCurrentBuffer();
    SetNextUacFrameNumber();
    for(int i = 0; i < UacBufferMaxCount - 1; ++i)
    {
        SubmitUacBuffer();
    }
    m_IsDeviceInitialized = true;
    m_IsDeviceStarted = true;
    m_NeedResync = false;
    m_SkipSubmitBuffer = false;
    // Set default volume and mute state.
    SetCurrentGain(m_Gain);
    SetMute(false);
}

void UacInDevice::SetUacControls() NN_NOEXCEPT
{
    int16_t res = -1;
    if (m_pFeatureUnit && m_pInputTerminal)
    {
        uint8_t *pBmaControls = m_pFeatureUnit->bmaControls;

        auto numChannels = std::min(m_pInputTerminal->bNrChannels, static_cast<uint8_t>(UacMaxChannelCount));
        auto numControls = std::min(m_pFeatureUnit->bControlSize, static_cast<uint8_t>(sizeof(m_Controls[0])));
        auto overflowControls = m_pFeatureUnit->bControlSize - numControls;

        for (uint8_t i = 0; i < numChannels; i++)
        {
            m_Controls[i] = 0;
            for (uint8_t j = 0; j < numControls; j++)
            {
                m_Controls[i] <<= 8;
                m_Controls[i] |= *pBmaControls;

                pBmaControls++;
            }
            pBmaControls += overflowControls;
        }

        //Some devices support Volume/Mute control even if control bit is not set.
        //If control bit is not already set, try volume/mute apis once to see if control bits are valid.
        if(!(m_Controls[0] & nn::cduac::Control_Volume) || !(m_Controls[0] & nn::cduac::Control_Mute))
        {
            bool hasVolumeControl = (m_Controls[0] & nn::cduac::Control_Volume);
            bool hasMuteControl = m_Controls[0] & nn::cduac::Control_Mute;
            int16_t controlMask = 0;
            if(!hasVolumeControl)
            {
                int16_t volume;
                hasVolumeControl = m_pInterface->GetAudioControl(nn::cduac::Request_GetCur,
                                            (nn::cduac::Control_Volume << 8),
                                            m_pFeatureUnit->bUnitId,
                                            sizeof(volume),
                                            &volume
                                            ).IsSuccess();
                if(hasVolumeControl)
                {
                    controlMask |= nn::cduac::Control_Volume;
                }
            }

            if(!hasMuteControl)
            {
                bool mute;
                hasMuteControl = m_pInterface->GetAudioControl(nn::cduac::Request_GetCur,
                                            (nn::cduac::Control_Mute << 8),
                                            m_pFeatureUnit->bUnitId,
                                            sizeof(mute),
                                            &mute
                                            ).IsSuccess();
                if(hasMuteControl)
                {
                    controlMask |= nn::cduac::Control_Mute;
                }
            }

            if(controlMask)
            {
                for (uint8_t i = 0; i < numChannels; i++)
                {
                    m_Controls[i] |= controlMask;
                }
            }
        }

        if (m_Controls[0] & nn::cduac::Control_Volume)
        {
            m_pInterface->GetAudioControl(nn::cduac::Request_GetCur,
                                        (nn::cduac::Control_Volume << 8),
                                        m_pFeatureUnit->bUnitId,
                                        sizeof(m_CurrentVolume),
                                        &m_CurrentVolume
                                        );
            m_pInterface->GetAudioControl(nn::cduac::Request_GetMax,
                                        (nn::cduac::Control_Volume << 8),
                                        m_pFeatureUnit->bUnitId,
                                        sizeof(m_MaxVolume),
                                        &m_MaxVolume
                                        );
            m_pInterface->GetAudioControl(nn::cduac::Request_GetMin,
                                (nn::cduac::Control_Volume << 8),
                                m_pFeatureUnit->bUnitId,
                                sizeof(m_MinVolume),
                                &m_MinVolume
                                );
            m_pInterface->GetAudioControl(nn::cduac::Request_GetRes,
                                (nn::cduac::Control_Volume << 8),
                                m_pFeatureUnit->bUnitId,
                                sizeof(res),
                                &res
                                );
        }

        if (m_Controls[0] & nn::cduac::Control_Mute)
        {
            m_pInterface->GetAudioControl(nn::cduac::Request_GetCur,
                                        (nn::cduac::Control_Mute << 8),
                                        m_pFeatureUnit->bUnitId,
                                        sizeof(m_IsMute),
                                        &m_IsMute
                                        );
        }
    }
}

void UacInDevice::SetUacEndpoint() NN_NOEXCEPT
{
    auto uacSamplingRate = SetEndpointSamplingRate(m_pInterface, m_pParser);

    auto formatType = m_pParser->GetAudioStreamingFormatType(m_pInterface->GetInterfaceProfile(), m_AltSetting);
    MixerFormat inFormat;

    inFormat.sampleRate = uacSamplingRate;
    inFormat.channelCount = formatType->bNrChannles;
    inFormat.sampleFormat = UacFormatToAudioFormat(formatType->bFormatType, formatType->bBitResolution);

    auto sampleSize = GetSampleByteSize(inFormat.sampleFormat);
    auto inputBytes = (uacSamplingRate / 1000) * inFormat.channelCount * sampleSize;
    if(uacSamplingRate % 1000 != 0)
    {
        inputBytes += inFormat.channelCount * sampleSize;
    }
    if(inputBytes < m_MaxPacketSize)
    {
        inputBytes = m_MaxPacketSize;
    }
    for (int i = 0; i < UacFrameDurationMs; ++i)
    {
        m_InputBytes[i] = inputBytes;
    }

    m_Mixer.SetFormat(inFormat, m_OutFormat);
}

nn::audio::SampleFormat UacInDevice::UacFormatToAudioFormat(uint8_t uacFormat, uint8_t uacSampleSizeBits)
{
    if( uacFormat == nn::cduac::AudioDataFormat_Pcm8 || uacFormat == nn::cduac::AudioDataFormat_Pcm)
    {
        if(uacSampleSizeBits == 16)
        {
            return SampleFormat_PcmInt16;
        }
        if(uacSampleSizeBits == 32)
        {
            return SampleFormat_PcmInt32;
        }
        if(uacSampleSizeBits == 8)
        {
            return SampleFormat_PcmInt8;
        }
    }
    return SampleFormat_Invalid;
}

void UacInDevice::SetSamplingFrequency(nn::cduac::AudioSamplingFrequency* outFrequency, uint32_t frequency) NN_NOEXCEPT
{
    outFrequency->n1         = frequency % 256;
    outFrequency->n256       = (frequency >> 8) & 0xFF; // (frequency / 256) % 256
    outFrequency->n65536     = (frequency >> 16) & 0xFF; // (frequency / 65536) % 256
}


void UacInDevice::StopDeviceInternal(bool isConnected) NN_NOEXCEPT
{
    if(m_IsDeviceInitialized)
    {
        if(isConnected && m_IsInitialized)
        {
            m_AltSetting = 0;
            auto profile = m_pInterface->GetInterfaceProfile();
            nn::usb::UsbEndpointDescriptor* descriptor = m_pParser->GetEndpoint(profile, m_AltSetting, 0);
            m_pInterface->SetAltSetting(m_AltSetting, descriptor, 0, 0);
        }
        m_IsDeviceInitialized = false;
        m_IsDeviceStarted = false;
    }
}

int8_t* UacInDevice::ReadFromDevice(UacReadResult* outResult) NN_NOEXCEPT
{
    if(!m_IsInitialized)
    {
        *outResult = UacReadResult_Failure;
        return nullptr;
    }
    *outResult = CheckXferStatus();
    if(*outResult != UacReadResult_FailureAlreadyCopiedToUser)
    {
        return GetNextReadyBuffer();
    }
    return nullptr;
}

int8_t* UacInDevice::GetNextReadyBuffer() NN_NOEXCEPT
{
    auto uacBuffer = m_UacBuffer[m_UacBufferReadIndex];
    ++m_UacBufferReadIndex;
    if(m_UacBufferReadIndex >= UacBufferMaxCount)
    {
        m_UacBufferReadIndex = 0;
    }
    return uacBuffer;
}

void UacInDevice::SubmitUacBuffer() NN_NOEXCEPT
{
    auto index = m_UacBufferIndex;

    memset(m_UacBuffer[index], 0, sizeof(m_UacBuffer[index]));
    nn::Result result = m_pInterface->PostTransferAsync(
                                                            m_UacBuffer[index],
                                                            m_InputBytes,
                                                            UacFrameDurationMs,
                                                            m_NextUsbFrame,
                                                            index
                                                            );
    if (!result.IsSuccess())
    {
        NN_DETAIL_AUDIO_ERROR("microphone input error %d %d\n", result.GetModule(), result.GetDescription());
        m_NeedResync = true;
    }
    else
    {
        m_NextUsbFrame += UacFrameDurationMs;
        ++m_UacBufferIndex;
        if(m_UacBufferIndex >= UacBufferMaxCount)
        {
            m_UacBufferIndex = 0;
        }
    }
}

UacInDevice::UacReadResult UacInDevice::CheckXferStatus() NN_NOEXCEPT
{
    nn::Result result = m_pInterface->GetXferReport(&m_OutCount, m_XferReport, UacFrameDurationMs * UacBufferMaxCount);

    if (result.IsSuccess())
    {
        if (m_OutCount == UacFrameDurationMs)  // Most commone case, 1 buffer completed
        {
            uint32_t dataError = 0;
            for (uint32_t i = 0; i < m_OutCount; i++)
            {
                nn::usb::XferReport *pXferReport = &m_XferReport[i];

                if (!pXferReport->result.IsSuccess())
                {
                    // If we have a frame error we need to resync
                    if (pXferReport->result <= nn::usb::ResultNotInFrameWindow())
                    {
                        m_NeedResync = true;
                        return UacReadResult_Failure;
                    }
                    // For anything else treat it as a data error
                    else
                    {
                        dataError++;
                    }
                }
            }

            if (dataError == 0)
            {
                return UacReadResult_Success;
            }
            else // Error, depop input
            {
                return UacReadResult_Failure;
            }
        }
        else if (m_OutCount > UacFrameDurationMs)
        {
            if(m_OutCount == UacFrameDurationMs * UacBufferMaxCount)
            {
                SetNextUacFrameNumber();
                auto extraFrames = (m_OutCount / UacFrameDurationMs) - 1;
                for(int i = 0; i < extraFrames; ++i)
                {
                    CopyToUserBuffer(GetNextReadyBuffer(), UacReadResult_Success);
                    SubmitUacBuffer();
                }
                return UacReadResult_Success;
            }
            else
            {
                for(int i = 0; i < m_OutCount; ++i)
                {
                    if(!m_XferReport[i].result.IsSuccess())
                    {
                        m_NeedResync = true;
                        return UacReadResult_Failure;
                    }
                }

                auto extraFrames = (m_OutCount / UacFrameDurationMs) - 1;
                for(int i = 0; i < extraFrames; ++i)
                {
                    CopyToUserBuffer(GetNextReadyBuffer(), UacReadResult_Success);
                    SubmitUacBuffer();
                }
                return UacReadResult_Success;
            }
        }
        else // No buffers completed, this should never be the case :0
        {
            m_SkipSubmitBuffer = true;
            return UacReadResult_FailureAlreadyCopiedToUser;
        }
    }
    m_NeedResync = true;
    return UacReadResult_Failure;
}

void UacInDevice::ResyncUacDevice(bool waitCompletions) NN_NOEXCEPT
{
    if(m_IsDeviceInitialized)
    {
        StopDeviceInternal(true);
        StartDeviceInternal();
    }
    m_NeedResync = false;
}

void UacInDevice::SetNextUacFrameNumber() NN_NOEXCEPT
{
    auto result = m_pInterface->GetCurrentFrame(&m_NextUsbFrame);

    if (result.IsSuccess())
    {
        m_NextUsbFrame += (UacFrameDurationMs * 3) - (m_NextUsbFrame % UacFrameDurationMs);
    }
}

void UacInDevice::CopyToUserBuffer(int8_t* uacBuffer, UacReadResult readResult) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if(readResult == UacReadResult_FailureAlreadyCopiedToUser)
    {
        return;
    }

    bool shouldDepop = readResult == UacReadResult_Failure || uacBuffer == nullptr;

    if(shouldDepop)
    {
        m_Mixer.Depop();
    }
    else
    {
        auto writeIndex = m_XferReport[0].transferredSize;
        auto readIndex = m_XferReport[0].requestedSize;
        for(int i = 1; i < UacFrameDurationMs; ++i)
        {
            auto& xfer = m_XferReport[i];
            if(readIndex != writeIndex)
            {
                memmove(&uacBuffer[writeIndex], &uacBuffer[readIndex], xfer.transferredSize);
            }
            writeIndex += xfer.transferredSize;
            readIndex += xfer.requestedSize;
        }
        uint32_t inputBytes = writeIndex;
        float volume = 1.0f;
        auto result = AppletVolumeManager::GetAppletVolume(&volume, AppletVolumeManager::SessionType_AudioIn, m_AppletId);

        if(!result.IsSuccess())
        {
            NN_DETAIL_AUDIO_WARN("GetVolume failed (%d:%d)\n", result.GetModule(), result.GetDescription());
        }
        m_Mixer.Mix(uacBuffer, inputBytes, volume);
    }
}

int16_t UacInDevice::GetMinVolume() NN_NOEXCEPT
{
    return m_MinVolume;
}

int16_t UacInDevice::GetMaxVolume() NN_NOEXCEPT
{
    return m_MaxVolume;
}

int16_t UacInDevice::GetCurrentVolume() NN_NOEXCEPT
{
    return m_CurrentVolume;
}

Result UacInDevice::SetCurrentVolume(int16_t volume) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_pFeatureUnit && m_pInputTerminal, ResultNotSupported());
    NN_RESULT_THROW_UNLESS(volume <= m_MaxVolume && volume >= m_MinVolume, ResultNotSupported());
    NN_RESULT_THROW_UNLESS(m_IsDeviceStarted, ResultOperationFailed());

    auto numChannels = std::min(m_pInputTerminal->bNrChannels, static_cast<uint8_t>(UacMaxChannelCount));
    bool volumeSet = false;
    for (uint8_t i = 0; i < numChannels; i++)
    {
        if(m_Controls[i] & nn::cduac::Control_Volume)
        {
            auto result = m_pInterface->SetAudioControl(
                                                    nn::cduac::Request_SetCur,
                                                    (nn::cduac::Control_Volume << 8) | i,
                                                    m_pFeatureUnit->bUnitId,
                                                    sizeof(volume),
                                                    &volume
                                                    );

            if(!result.IsSuccess())
            {
                NN_DETAIL_AUDIO_ERROR("Could not set volume (%d:%d)\n", result.GetModule(), result.GetDescription());
            }
            else
            {
                volumeSet = true;
                m_CurrentVolume = volume;
            }
        }
    }
    NN_RESULT_THROW_UNLESS(volumeSet, ResultNotSupported());
    NN_RESULT_SUCCESS;
}

bool UacInDevice::IsMuted() NN_NOEXCEPT
{
    return m_IsMute;
}

void UacInDevice::SetMute(bool mute) NN_NOEXCEPT
{
    if(!m_pFeatureUnit || !m_pInputTerminal)
    {
        return;
    }
    if(mute == m_IsMute)
    {
        return;
    }

    uint8_t data = mute;
    auto numChannels = std::min(m_pInputTerminal->bNrChannels, static_cast<uint8_t>(UacMaxChannelCount));
    for (uint8_t i = 0; i < numChannels; i++)
    {
        if(m_Controls[i] & nn::cduac::Control_Mute)
        {
            auto result = m_pInterface->SetAudioControl(
                                            nn::cduac::Request_GetCur,
                                            (nn::cduac::Control_Mute << 8) | i,
                                            m_pFeatureUnit->bUnitId,
                                            sizeof(data),
                                            &data
                                            );
            if(!result.IsSuccess())
            {
                NN_DETAIL_AUDIO_ERROR("Could not set mute (%d:%d)\n", result.GetModule(), result.GetDescription());
            }
        }
    }
}

nn::os::SystemEventType* UacInDevice::GetBufferCompletionEvent() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if(m_pInterface)
    {
        return m_pInterface->GetCompletionEvent();
    }
    return nullptr;
}

void UacInDevice::ClearBufferCompletionEvent() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if(m_pInterface && m_IsDeviceStarted)
    {
        auto event = m_pInterface->GetCompletionEvent();
        if(event)
        {
            nn::os::ClearSystemEvent(event);
        }
    }
}

Result UacInDevice::SetCurrentGain(float gain) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_SDK_REQUIRES_MINMAX(gain, 0.0f, 1.0f);
    const auto VolumeRange = m_MaxVolume - m_MinVolume;
    NN_RESULT_DO(SetCurrentVolume(m_MinVolume + VolumeRange * gain));
    m_Gain = gain;
    NN_RESULT_SUCCESS;
}

void UacInDevice::ClearBuffers() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    m_Mixer.ClearBuffers();
}

bool UacInDevice::CanOpen() const NN_NOEXCEPT
{
    return !m_IsOpen;
}

}  // namespace detail
}  // namespace audio
}  // namespace nn


