﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/os/os_Mutex.h>

namespace nn {
namespace audio {
namespace detail {

template <class T, int ObjectInfoCount = 16>
class InterruptManager
{
private:
    void* m_ResetEvent;
    T* m_pObjectInfo[ObjectInfoCount];
    nn::os::Mutex m_Mutex;
    bool m_IsInitialized;

public:
    InterruptManager()
        : m_ResetEvent(nullptr)
        , m_Mutex(true)
        , m_IsInitialized(false)
    {
        memset(m_pObjectInfo, 0, sizeof(m_pObjectInfo));
    }


    ~InterruptManager()
    {
        m_IsInitialized = false;
    }

    void Initialize()
    {
        HANDLE hResetEvent_ = CreateEvent(NULL, FALSE, FALSE, NULL);
        NN_ABORT_UNLESS(hResetEvent_);
        m_ResetEvent = hResetEvent_;
        m_IsInitialized = true;
    }

    void Finalize()
    {
        SetEvent(m_ResetEvent);
        m_IsInitialized = false;
    }

    Result Register(T* pObject)
    {
        NN_RESULT_THROW_UNLESS(m_IsInitialized, ResultOperationFailed());
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        for (int i = 0; i < sizeof(m_pObjectInfo) / sizeof(*m_pObjectInfo); ++i)
        {
            if (m_pObjectInfo[i] == nullptr)
            {
                m_pObjectInfo[i] = pObject;
                SetEvent(m_ResetEvent);
                NN_RESULT_SUCCESS;
            }
        }
        NN_RESULT_THROW(ResultOperationFailed());
    }

    Result Unregister(T* pObject)
    {
        NN_RESULT_THROW_UNLESS(m_IsInitialized, ResultOperationFailed());
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        for (int i = 0; i < sizeof(m_pObjectInfo) / sizeof(*m_pObjectInfo); ++i)
        {
            if (m_pObjectInfo[i] == pObject)
            {
                m_pObjectInfo[i] = nullptr;
                SetEvent(m_ResetEvent);
                NN_RESULT_SUCCESS;
            }
        }
        NN_RESULT_THROW(ResultOperationFailed());
    }

    T* GetSignaled()
    {
        if(!m_IsInitialized)
        {
            return nullptr;
        }
        HANDLE waitArray[ObjectInfoCount + 1];
        int count = 0;
        {
            std::lock_guard<nn::os::Mutex> lock(m_Mutex);
            for (int i = 0; i < sizeof(m_pObjectInfo) / sizeof(*m_pObjectInfo); ++i)
            {
                if (m_pObjectInfo[i])
                {
                    if (count != i)
                    {
                        m_pObjectInfo[count] = m_pObjectInfo[i];
                        m_pObjectInfo[i] = nullptr;
                    }
                    ++count;
                }
            }
            for (int i = 0; i < sizeof(m_pObjectInfo) / sizeof(*m_pObjectInfo); ++i)
            {
                if (m_pObjectInfo[i])
                {
                    waitArray[i] = m_pObjectInfo[i]->GetWindowsEventHandle();
                }
            }
        }
        waitArray[count] = m_ResetEvent;

        DWORD waitResult = WaitForMultipleObjects(count + 1, waitArray, FALSE, INFINITE);

        if (waitResult >= WAIT_OBJECT_0 + 0 && waitResult < WAIT_OBJECT_0 + count)
        {
            return m_pObjectInfo[waitResult - WAIT_OBJECT_0];
        }
        return nullptr;
    }

};

}  // namespace detail
}  // namespace audio
}  // namespace nn
