﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/Result/result_HandlingUtility.h>
#include "audio_AudioUacDriver.h"

#include <nn/audio/detail/audio_Log.h>
using namespace nne::audio;

namespace nn {
namespace audio {
namespace detail {

AudioUacDriver::AudioUacDriver() NN_NOEXCEPT
    : m_DeviceGain(1.0f)
{
}

AudioUacDriver::~AudioUacDriver() NN_NOEXCEPT
{
}

Result AudioUacDriver::Open(server::SessionType sessionType, AppletVolumeManager::SessionType avmType, const char* name, server::SessionFormat& format, int sessionId, nn::dd::ProcessHandle processHandle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    m_ChannelCount = format.channelCount;
    m_SessionId = sessionId;
    m_AppletResourceUserId = aruid;
    auto& uacManager = UacManager::GetInstance();
    m_ClientProcessHandle = processHandle;
    m_AvmType = avmType;
    Result result = uacManager.OpenUacInSession(&m_UacId, name, format, m_SessionId, m_AppletResourceUserId);

    NN_ABORT_UNLESS_RESULT_SUCCESS(AppletVolumeManager::RegisterAppletResourceUserId(m_AppletResourceUserId));
    NN_ABORT_UNLESS_RESULT_SUCCESS(AppletVolumeManager::OpenSession(m_AvmType, m_SessionId, m_AppletResourceUserId, false));

    return result;
}
Result AudioUacDriver::Close() NN_NOEXCEPT
{

    NN_ABORT_UNLESS_RESULT_SUCCESS(AppletVolumeManager::ForceResume(m_AvmType, m_SessionId, true, m_AppletResourceUserId));
    NN_ABORT_UNLESS_RESULT_SUCCESS(AppletVolumeManager::CloseSession(m_AvmType, m_SessionId));
    NN_ABORT_UNLESS_RESULT_SUCCESS(AppletVolumeManager::UnregisterAppletResourceUserId(m_AppletResourceUserId));

    auto& uacManager = UacManager::GetInstance();
    uacManager.CloseUacInSession(m_UacId);
    m_TransferMemoryHelper.UnmapAll();
    m_DeviceGain = 1.0f;
    NN_RESULT_SUCCESS;
}

Result AudioUacDriver::Start() NN_NOEXCEPT
{
    auto& uacManager = UacManager::GetInstance();
    uacManager.StartUacInSession(m_UacId);
    NN_RESULT_SUCCESS;
}

Result AudioUacDriver::Stop() NN_NOEXCEPT
{
    auto& uacManager = UacManager::GetInstance();
    uacManager.StopUacInSession(m_UacId);
    NN_RESULT_SUCCESS;
}

Result AudioUacDriver::Sleep() NN_NOEXCEPT
{
    auto& uacManager = UacManager::GetInstance();
    uacManager.StopUacInSession(m_UacId);
    NN_RESULT_SUCCESS;
}

Result AudioUacDriver::Wake() NN_NOEXCEPT
{
    //SIGLO-64462: Uac microphones cannot restart automatically
    NN_RESULT_SUCCESS;
}

uint64_t AudioUacDriver::GetSamplesProcessed() const NN_NOEXCEPT
{
    return 0;
}

void* AudioUacDriver::MapBuffer(server::SessionBuffer& buffer, nn::dd::ProcessHandle processHandle) NN_NOEXCEPT
{
    //no-op. This if for mapping user buffer to adsp
    NN_UNUSED(buffer);
    NN_UNUSED(processHandle);
    buffer.mappedAddr = nullptr;
    return nullptr;
}

void* AudioUacDriver::MapBuffer(server::SessionBuffer& buffer, nn::sf::NativeHandle&& bufferHandle) NN_NOEXCEPT
{
    auto mappedAddr = m_TransferMemoryHelper.Map(buffer.addr, buffer.size, std::move(bufferHandle));
    buffer.mappedAddr = mappedAddr;
    return mappedAddr;
}

bool AudioUacDriver::UnmapBuffer(const server::SessionBuffer& buffer) NN_NOEXCEPT
{
    auto unmapped = m_TransferMemoryHelper.Unmap(buffer.mappedAddr);
    return unmapped;
}

bool AudioUacDriver::TryCompleteBuffer(const server::SessionBuffer& buffer) NN_NOEXCEPT
{
    auto bufferAddress = GetBufferAddress();

    if(buffer.mappedAddr != bufferAddress)
    {
        UnmapBuffer(buffer);
        return true;
    }
    return false;
}

Result AudioUacDriver::AppendBuffer(const server::SessionBuffer& buffer) NN_NOEXCEPT
{
    auto& uacManager = UacManager::GetInstance();
    uacManager.GetUacInDevice(m_UacId)->AppendBuffer(buffer.mappedAddr, buffer.size, m_ClientProcessHandle);
    NN_RESULT_SUCCESS;
}

void* AudioUacDriver::GetBufferAddress() const NN_NOEXCEPT
{
    auto& uacManager = UacManager::GetInstance();
    return uacManager.GetUacInDevice(m_UacId)->GetBufferAddress();
}

Result AudioUacDriver::SetDeviceGain(float gain) NN_NOEXCEPT
{
    auto& uacManager = UacManager::GetInstance();
    NN_RESULT_DO(uacManager.GetUacInDevice(m_UacId)->SetCurrentGain(gain));

    m_DeviceGain = gain;
    NN_RESULT_SUCCESS;
}

float AudioUacDriver::GetDeviceGain() const NN_NOEXCEPT
{
    return m_DeviceGain;
}

}  // namespace detail
}  // namespace audio
}  // namespace nn


