﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/Result/result_HandlingUtility.h>
#include "audio_AudioRecordSession-os.horizon.h"

namespace nn { namespace audio {

namespace server {

namespace {
    const char                   g_DeviceOutName[] = "DeviceRecord";
}

void InitializeFinalOutputRecorder(nn::os::SemaphoreType* pSemaphore, nn::os::EventType* pEventSemaphore) NN_NOEXCEPT
{
    NN_UNUSED(pEventSemaphore);
    nne::audio::gmix::SetProcessSemaphore(nne::audio::gmix::Session::Name::GameRecord, pSemaphore);
}

void FinalizeFinalOutputRecorder() NN_NOEXCEPT
{
}

} //namespace server

namespace detail {
AudioGmixDriver AudioRecordSession::s_GmixDriver[server::FinalOutputRecorderManagerImpl::NumberOfFinalOutputRecorderSessions];

AudioRecordSession::AudioRecordSession(int32_t sessionId, nn::os::SystemEvent& releaseClientBufferEvent) NN_NOEXCEPT
    : AudioSession(sessionId, GetDriver(sessionId), releaseClientBufferEvent, server::SessionType_Record)
{
}

AudioRecordSession::~AudioRecordSession() NN_NOEXCEPT
{
}

const char* AudioRecordSession::GetDefaultName() const NN_NOEXCEPT
{
    return server::g_DeviceOutName;
}

Result AudioRecordSession::IsSupported(const char* name, const server::SessionFormat& format) const NN_NOEXCEPT
{
    // Fail if the sampleRate is not 0 or 48000
    NN_RESULT_THROW_UNLESS((0 == format.sampleRate || 48000 == format.sampleRate), ResultInvalidSampleRate());
    // Fail if the channelCount is not 0 or 2 or 6
    NN_RESULT_THROW_UNLESS(0 == format.channelCount || 2 == format.channelCount || 6 == format.channelCount, ResultInvalidChannelCount());

    NN_RESULT_SUCCESS;
}

FinalOutputRecorderState AudioRecordSession::GetState() const NN_NOEXCEPT
{
    switch(m_State)
    {
        case server::SessionState_Started:
            return FinalOutputRecorderState_Started;
        case server::SessionState_Stopped:
            return FinalOutputRecorderState_Stopped;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
    return FinalOutputRecorderState_Stopped;
}

AudioDriver* AudioRecordSession::GetDriver(int sessionId) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_MINMAX(sessionId, 0, static_cast<int>(NN_ARRAY_SIZE(s_GmixDriver)) - 1);
    return &s_GmixDriver[sessionId];
}

}}} // namespace nn::audio::detail
