﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/Result/result_HandlingUtility.h>
#include "audio_AudioInSession-os.horizon.h"
#include "../common/audio_AudioInPrivate.h"
#include "../audio_AudioHardware.h"
#include "audio_UacManager.h"

namespace nn { namespace audio {

namespace server {

namespace {
    const char* g_DeviceInName = nn::audio::common::DeviceAudioIn;
}

int ListAudioIns(AudioInInfo* outAudioIns, int count, bool filtered) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outAudioIns);
    NN_SDK_REQUIRES(count >= 0);

    memset(outAudioIns, 0, count * sizeof(AudioInInfo));
    if( count > 0 )
    {
        int index = 0;
        if(!filtered || IsAnalogMicrophonePlugged())
        {
            strcpy(outAudioIns[0].name, nn::audio::common::DeviceAudioIn);
            index = 1;
        }
        auto& uacManager = ::nn::audio::detail::UacManager::GetInstance();
        index += uacManager.ListUacIns(&outAudioIns[index], count - index);
        count = index;
    }

    return count;
}

void AudioInInitialize(nn::os::SemaphoreType* semaphore, nn::os::EventType* pEventSemaphore) NN_NOEXCEPT
{
    NN_UNUSED(pEventSemaphore);
    nne::audio::gmix::SetProcessSemaphore(nne::audio::gmix::Session::Name::NearVoice, semaphore);
}

void AudioInFinalize() NN_NOEXCEPT
{
}

} //namespace server

namespace detail {
AudioGmixDriver AudioInSession::s_GmixDriver[server::AudioInManagerImpl::NumberOfDefaultInSessions];
AudioUacDriver  AudioInSession::s_UacDriver[server::AudioInManagerImpl::NumberOfAudioInSessions - server::AudioInManagerImpl::NumberOfDefaultInSessions];

AudioInSession::AudioInSession(int32_t sessionId, nn::os::SystemEvent& releaseClientBufferEvent) NN_NOEXCEPT
    : AudioSession(sessionId, GetDriver(sessionId), releaseClientBufferEvent, server::SessionType_Input)
{
}

AudioInSession::~AudioInSession() NN_NOEXCEPT
{
}

const char* AudioInSession::GetDefaultName() const NN_NOEXCEPT
{
    return server::g_DeviceInName;
}

Result AudioInSession::IsSupported(const char* name, const server::SessionFormat& format) const NN_NOEXCEPT
{
    // Fail if the sampleRate is not 0 or 48000
    NN_RESULT_THROW_UNLESS((0 == format.sampleRate || 48000 == format.sampleRate), ResultInvalidSampleRate());
    if(0 == strnlen(name, AudioIn::NameLength) || 0 == strncmp(name, GetDefaultName(), AudioIn::NameLength))
    {
        //Gmix device
        NN_RESULT_SUCCESS;
    }
    else
    {
        //uac device
        auto& uacManager = ::nn::audio::detail::UacManager::GetInstance();
        NN_RESULT_THROW_UNLESS(uacManager.IsUacInName(name), nn::audio::ResultNotFound());
    }

    NN_RESULT_SUCCESS;
}

AudioInState AudioInSession::GetState() const NN_NOEXCEPT
{
    switch(m_State)
    {
        case server::SessionState_Started:
            return AudioInState_Started;
        case server::SessionState_Stopped:
            return AudioInState_Stopped;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
    return AudioInState_Stopped;
}

void AudioInSession::SetName(const char* name, int length) NN_NOEXCEPT
{
    strncpy(m_Name, name, std::min(static_cast<size_t>(length), sizeof(m_Name)));
    m_Name[ sizeof(m_Name) - 1 ] = '\0';
}

AudioDriver* AudioInSession::GetDriver(int sessionId) const NN_NOEXCEPT
{
    const int NumberOfSessions = static_cast<int>(NN_ARRAY_SIZE(s_GmixDriver) + NN_ARRAY_SIZE(s_UacDriver));
    NN_ABORT_UNLESS_MINMAX(sessionId, 0, NumberOfSessions - 1);
    if(sessionId < server::AudioInManagerImpl::NumberOfDefaultInSessions)
    {
        return &s_GmixDriver[sessionId];
    }
    else
    {
        auto index = sessionId - server::AudioInManagerImpl::NumberOfDefaultInSessions;
        return &s_UacDriver[index];
    }
}

}}} // namespace nn::audio::detail
