﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>
#include <mutex>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Semaphore.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/audioctrl/audioctrl_AudioControllerTypes.h>
#include <nne/audio/audio.h>
#include <nn/cduac/cduac_Spec.h>
#include <nn/cduac/cduac_Api.h>
#include <nn/gpio/gpio.h>

#include "audio_UacManager.h"
#include "../audio_AudioHardware.h"

#if !defined(NN_BUILD_CONFIG_SOC_TEGRA_X1)
#error This SoC is not supported.
#endif

namespace nn { namespace audio { namespace server { namespace detail{

class SleepWakePolicy
{
public:
    enum class State : int32_t
    {
        Invalid = -1,
        Initialized = 0,
        Asleep = 1,
        Finalized = 2,
    };
    SleepWakePolicy() NN_NOEXCEPT;
    State GetSleepWakeState() NN_NOEXCEPT;
    void Initialize(bool hasCodecIc, bool hasHdmi) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    void Sleep() NN_NOEXCEPT;
    void Wake() NN_NOEXCEPT;

    virtual void InitializeImpl() NN_NOEXCEPT = 0;
    virtual void FinalizeImpl() NN_NOEXCEPT = 0;
    virtual void SleepImpl() NN_NOEXCEPT = 0;
    virtual void WakeImpl() NN_NOEXCEPT = 0;

    bool IsReadyToSleep() NN_NOEXCEPT;
    bool IsReadyToWake() NN_NOEXCEPT;
    bool IsReadyToFinalize() NN_NOEXCEPT;
    bool IsReadyToInitialize() NN_NOEXCEPT;
    inline bool CheckState(bool condition, State nextState) NN_NOEXCEPT;

private:
    std::atomic<SleepWakePolicy::State> m_SleepWakeState;

protected:
    nn::os::Mutex m_Mutex;
    bool m_HasCodecIc;
    bool m_HasHdmi;
};

class Gmix : public SleepWakePolicy
{
public:
    enum class DownmixMode
    {
        Invalid,
        Default,
        Mono,
    };
public:
    virtual void InitializeImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void FinalizeImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SleepImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void WakeImpl() NN_NOEXCEPT NN_OVERRIDE;
    void UpdateDownmixMode(DownmixMode mode) NN_NOEXCEPT;
    void UpdateDownmixModeForGameRecord(DownmixMode mode) NN_NOEXCEPT;
    void PrintPerformanceStatistics() NN_NOEXCEPT;

private:
    nn::os::SemaphoreType m_ProcessSemaphore;
};

class Ahub : public SleepWakePolicy
{
public:
    Ahub() NN_NOEXCEPT;
    virtual void InitializeImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void FinalizeImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SleepImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void WakeImpl() NN_NOEXCEPT NN_OVERRIDE;
    bool IsInitialized() NN_NOEXCEPT;
    void SetMute(bool isMute) NN_NOEXCEPT;
    void SetVolume(float volume, uint32_t curveDurationInMicroseconds) NN_NOEXCEPT;

    nne::audio::gmix::Session* GetGmixSession() NN_NOEXCEPT;
    nn::os::Semaphore* GetProcessSemaphore() NN_NOEXCEPT;

private:
    nne::audio::gmix::Session* m_pGmixAhub;
    nne::audio::device::Session* m_pAhub;
    nn::os::Semaphore m_ProcessSemaphore;
};

class AhubIn : public SleepWakePolicy
{
public:
    AhubIn() NN_NOEXCEPT;
    virtual void InitializeImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void FinalizeImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SleepImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void WakeImpl() NN_NOEXCEPT NN_OVERRIDE;

    void Prepare() NN_NOEXCEPT;
    void Start() NN_NOEXCEPT;
    void Stop() NN_NOEXCEPT;
    bool IsStarted() NN_NOEXCEPT;
    bool IsInitialized() NN_NOEXCEPT;

private:
    nne::audio::gmix::Session* m_pGmixAhubIn;
    nne::audio::device::Session* m_pAhubIn;
    nn::os::Mutex m_Mutex;
    bool m_IsStarted;
};

class Hda : public SleepWakePolicy
{
public:
    enum ChannelCount
    {
        ChannelCount_Invalid = 0,
        ChannelCount_2ch = 2,
        ChannelCount_6ch = 6,
    };

    Hda() NN_NOEXCEPT;
    virtual void InitializeImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void FinalizeImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SleepImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void WakeImpl() NN_NOEXCEPT NN_OVERRIDE;
    void UpdateHotplugState() NN_NOEXCEPT;
    bool IsConnected() const NN_NOEXCEPT;
    bool ShouldSurroundEnable() const NN_NOEXCEPT;
    void SetVolume(float volume, uint32_t curveDurationInMicroseconds) NN_NOEXCEPT;
    void SetOutputMode(nn::audioctrl::AudioOutputMode mode) NN_NOEXCEPT;
    void Start() NN_NOEXCEPT;
    void Stop() NN_NOEXCEPT;

    nne::audio::gmix::Session* GetGmixSession() const NN_NOEXCEPT;
    nn::os::Semaphore* GetProcessSemaphore() NN_NOEXCEPT;

    void StopDevice() NN_NOEXCEPT;

private:
    nne::audio::gmix::Session* m_pGmixHda;
    nne::audio::device::Session* m_pHda;
    nn::os::Semaphore m_ProcessSemaphore;
    bool m_IsConnected;
    ChannelCount m_ChannelCount;
    bool m_Initialized;
    bool m_IsDeviceStarted;
    bool m_IsGmixStarted;
    nn::audioctrl::AudioOutputMode m_OutputMode;
};

class Codec : public SleepWakePolicy
{
public:
    Codec() NN_NOEXCEPT;
    virtual void InitializeImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void FinalizeImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SleepImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void WakeImpl() NN_NOEXCEPT NN_OVERRIDE;
    bool IsInitialized() NN_NOEXCEPT;
    void Abort() NN_NOEXCEPT;

    nn::os::SystemEvent* RegisterJackEvent() NN_NOEXCEPT;
    void UnregisterJackEvent() NN_NOEXCEPT;
    void ClearJackEvent() NN_NOEXCEPT;
    bool IsJackPlugged() NN_NOEXCEPT;

    void SetSpeakerHeadphoneVolume(int32_t volume) NN_NOEXCEPT;
    void SetSpeakerMute(bool mute) NN_NOEXCEPT;
    bool IsSpeakerMute() NN_NOEXCEPT;
    void SetHeadphoneAmplifierGain(int32_t gain) NN_NOEXCEPT;

    bool IsSpeakerOverCurrent() NN_NOEXCEPT;
    bool IsOverTemperature() NN_NOEXCEPT;
    bool IsMicBiasOverCurrent() NN_NOEXCEPT;
    void DumpFactorySettings() NN_NOEXCEPT;

private:
    static const int VolumeLevelMin = 0x0;
    static const int VolumeLevelMax = 0xaf;

    // ALC5639 Datasheet Rev 0.944 8.4. MX-02h: Headphone Output Control 参照
    static const int HeadphoneAmplifierGainMin = 0x00;
    static const int HeadphoneAmplifierGainMax = 0x27;

    nn::gpio::GpioPadSession m_HeadphoneMicJackGpioSession;
    nn::os::SystemEvent m_HeadphoneMicJackGpioEvent;
    nn::os::Mutex m_StateCheckMutex;
    nn::os::Mutex m_NneAudioCodecMutex;
};

class CodecErrorDetector : public SleepWakePolicy
{
public:

    CodecErrorDetector(Codec& codec, AhubIn& ahubIn) NN_NOEXCEPT;
    virtual void InitializeImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void FinalizeImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void SleepImpl() NN_NOEXCEPT NN_OVERRIDE;
    virtual void WakeImpl() NN_NOEXCEPT NN_OVERRIDE;

private:
    Codec& m_Codec;
    AhubIn& m_AhubIn;

    void DetectionTreadFunc() NN_NOEXCEPT;
    void CodecAlertHandler() NN_NOEXCEPT;
    void SpeakerOverCurrentErrorHandler() NN_NOEXCEPT;
    void MicOverCurrentErrorHandler() NN_NOEXCEPT;
    NN_NORETURN
    void SystemFatalErrorHandler() NN_NOEXCEPT;
    void OpenCodecAlertGpioSession() NN_NOEXCEPT;
    void CloseCodecAlertGpioSession() NN_NOEXCEPT;

    // For CodecAlert
    nn::os::SystemEvent m_CodecAlertGpioEvent;
    nn::gpio::GpioPadSession codecAlertGpioSession;

    // For SpeakerOverCurrent (polling timer)
    nn::os::TimerEvent m_SpeakerOverCurrentPollingTimerEvent;

    // For Mic bias over current チャタリング防止タイマー (MicBiasOverCurrent 用)
    nn::os::TimerEvent m_MicBiasDebounceTimerEvent;
    bool m_IsInDebouncePeriod; // チャタリング防止タイマー(MicBiasOverCurrent 用)が動作中かどうかを表すフラグ

    bool m_IsJackStateChanged;
    bool m_IsMicBiasOverCurrentDetectionEnabled;

    // For detection thread sync
    nn::os::Event m_RequestSleepEvent;
    nn::os::Event m_RequestWakeEvent;
    nn::os::Event m_RequestExitEvent;
    nn::os::Event m_SleepWakeRequestCompleteEvent;
    nn::os::SystemEvent m_SystemFatalErrorEvent;

    nn::TimeSpan GetPollingTimerEventInterval() { return nn::TimeSpan::FromSeconds(1); };
    nn::TimeSpan GetMicBiasOverCurrentDetectionDelayTime() { return nn::TimeSpan::FromMilliSeconds(500); }; // Note: 500 ms はフルキーコントローラの実験結果を参考値にした値
    nn::os::ThreadType m_CodecAlertHandlerThread;

public:
    void EnableJackStateFlag() NN_NOEXCEPT;
    void TriggerMicCheck() NN_NOEXCEPT;
    void SetMicBiasOverCurrentDetectionEnabled(bool isEnabled) NN_NOEXCEPT;

    static void HandlerThreadFunc(void* arg) NN_NOEXCEPT
    {
        auto errorDetector = reinterpret_cast<CodecErrorDetector*>(arg);
        errorDetector->DetectionTreadFunc();
    }
};

class AudioDeviceController
{
public:
    enum class Device
    {
        Invalid,
        Hda,
        AhubHeadphone,
        AhubSpeaker,
        UacSpeaker,
    };
    enum EventIndex
    {
        EventIndex_RequestViaApi,
        EventIndex_Hotplug,
        EventIndex_HeadphoneSelect,
        EventIndex_UacSpeakerSelect,
        EventIndex_ExitRequest,

        EventIndex_Count
    };

    AudioDeviceController() NN_NOEXCEPT;

    void Initialize(bool hasCodecIc, bool hasHdmi) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    void Shutdown() NN_NOEXCEPT;
    void Sleep() NN_NOEXCEPT;
    void Wake() NN_NOEXCEPT;

    void SwitchOutputDevice(Device device) NN_NOEXCEPT;
    void SwitchDownmixMode(Device device) NN_NOEXCEPT;
    void NotifyDeviceSwitch(bool isHda, nn::TimeSpan fadeOutTime, nn::TimeSpan fadeInTime) NN_NOEXCEPT;
    void NotifyDeviceSwitch() NN_NOEXCEPT;
    void NotifyHeadphoneOutSwitch(bool isEnable) NN_NOEXCEPT;
    void NotifyHeadphoneInEnabled(bool isEnable) NN_NOEXCEPT;
    void NotifyUacSpeakerSwitch(bool isEnable) NN_NOEXCEPT;
    void SetInputDeviceForceEnabled(bool isEnabled) NN_NOEXCEPT;
    bool IsInputDeviceForceEnabled() const NN_NOEXCEPT;
    void NotifyForceDeviceSwitch(nn::audioctrl::AudioTarget target) NN_NOEXCEPT;
    void SetOutputMasterVolume(float volume, uint32_t curveDurationInMicroseconds) NN_NOEXCEPT;
    void SetSpeakerVolume(int32_t volume) NN_NOEXCEPT;
    void SetHeadphoneVolume(int32_t volume) NN_NOEXCEPT;

    void InitializeOutputControlThread() NN_NOEXCEPT;
    void FinalizeOutputControlThread() NN_NOEXCEPT;
    void OutputControlThreadFunc() NN_NOEXCEPT;

    Device GetRequestedDevice() const NN_NOEXCEPT;
    bool IsHdaRequested() const NN_NOEXCEPT;
    bool IsFadeRequired() const NN_NOEXCEPT;

    static void HotPlugFunc(void* arg)
    {
        auto controller = reinterpret_cast<AudioDeviceController*>(arg);
        controller->OutputControlThreadFunc();
    };

    Codec* GetCodecDevice() NN_NOEXCEPT;
    Ahub* GetAhubDevice() NN_NOEXCEPT;
    AhubIn* GetAhubInDevice() NN_NOEXCEPT;

    void SetOutputMode(nn::audioctrl::AudioTarget target, nn::audioctrl::AudioOutputMode mode) NN_NOEXCEPT;
    void SetCodecSpeakerVolume(int32_t volume) NN_NOEXCEPT;
    void SetCodecHeadphoneVolume(int32_t volume) NN_NOEXCEPT;
private:
    nn::os::Event m_HotplugEvent;
    nn::os::Event m_OutputDeviceControlEvent;
    nn::os::Event m_DeviceFadeEvent;
    nn::os::Event m_HeadphoneSelectEvent;
    nn::os::Event m_UacSpeakerSelectEvent;
    nn::os::Event m_ExitRequestEvent;
    std::atomic_bool m_IsHdaConnectionRequested = { true };
    std::atomic_bool m_IsHeadphoneOutSelected = { false };
    std::atomic_bool m_IsUacSpeakerSelected = { false };
    std::atomic<nn::audioctrl::AudioTarget> m_ForcedTarget = { nn::audioctrl::AudioTarget_Invalid };
    std::atomic_bool m_IsAwake = { true };
    std::atomic<float> m_OutputMasterVolume = { 1.0f };
    std::atomic<int32_t> m_CodecSpeakerVolume;
    std::atomic<int32_t> m_CodecHeadphoneVolume;

    Gmix m_GmixDevice;
    Ahub m_AhubDevice;
    AhubIn m_AhubInDevice;
    Hda m_HdaDevice;
    Codec m_CodecDevice;
    CodecErrorDetector m_CodecErrorDetector;

    nn::os::ThreadType m_HdaHotplugDetectThread;
    std::atomic_bool m_IsHdaHotplugDetectThreadActive;

    nn::TimeSpan m_DeviceFadeInTime;
    nn::TimeSpan m_DeviceFadeOutTime;
    std::atomic<Device> m_RequestedDevice;
    std::atomic<Device> m_SelectedDevice;

    bool m_HasCodecIc;
    bool m_HasHdmi;

    bool m_IsInputDeviceForceEnabled;

    nn::audioctrl::AudioOutputMode m_OutputModes[nn::audioctrl::AudioTarget_Count];
    AudioDeviceController::Device ConvertToDevice(nn::audioctrl::AudioTarget target) const NN_NOEXCEPT;
};

}}}} // namespace nn::audio::server::detail
