﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/Result/result_HandlingUtility.h>
#include "audio_AudioGmixDriver.h"

#include <nn/audio/detail/audio_Log.h>
using namespace nne::audio;

namespace nn {
namespace audio {
namespace detail {

AudioGmixDriver::AudioGmixDriver() NN_NOEXCEPT
{
}

AudioGmixDriver::~AudioGmixDriver() NN_NOEXCEPT
{
}

Result AudioGmixDriver::Open(server::SessionType sessionType, AppletVolumeManager::SessionType avmType, const char* name, server::SessionFormat& format, int sessionId, nn::dd::ProcessHandle processHandle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    NN_UNUSED(name);
    m_pGmixSession = nullptr;
    m_ChannelCount = format.channelCount;
    m_Name = SessionTypeToGmixType(sessionType);
    m_SessionId = sessionId;
    m_AppletResourceUserId = aruid;
    m_AvmType = avmType;
    const auto gmixFormat = m_ChannelCount <= 2 ? gmix::Session::Format::Stereo : gmix::Session::Format::Surround;
    gmix::OpenSession(&m_pGmixSession, m_Name, m_SessionId, gmix::Session::Name::Default, gmixFormat, gmix::Session::Mode::Default, reinterpret_cast<uint64_t>(m_AppletResourceUserId.lower));
    m_ClientProcessHandle = processHandle;

    NN_ABORT_UNLESS_RESULT_SUCCESS(AppletVolumeManager::RegisterAppletResourceUserId(m_AppletResourceUserId));
    NN_ABORT_UNLESS_RESULT_SUCCESS(AppletVolumeManager::OpenSession(m_AvmType, m_SessionId, m_AppletResourceUserId, true));

    NN_RESULT_THROW_UNLESS(m_pGmixSession != nullptr, ResultOperationFailed());
    NN_RESULT_SUCCESS;
}
Result AudioGmixDriver::Close() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(AppletVolumeManager::ForceResume(m_AvmType, m_SessionId, true, m_AppletResourceUserId));

    NN_ABORT_UNLESS_RESULT_SUCCESS(AppletVolumeManager::CloseSession(m_AvmType, m_SessionId));
    NN_ABORT_UNLESS_RESULT_SUCCESS(AppletVolumeManager::UnregisterAppletResourceUserId(m_AppletResourceUserId));

    gmix::CloseSession(m_pGmixSession);
    m_pGmixSession = nullptr;
    NN_RESULT_SUCCESS;
}

Result AudioGmixDriver::Start() NN_NOEXCEPT
{
    m_pGmixSession->Start();
    NN_RESULT_SUCCESS;
}

Result AudioGmixDriver::Stop() NN_NOEXCEPT
{
    m_pGmixSession->Stop();
    NN_RESULT_SUCCESS;
}

Result AudioGmixDriver::Sleep() NN_NOEXCEPT
{
    m_GmixSamplesProcessed += m_pGmixSession->GetSamplesProcessed();
    m_pGmixSession->Stop();
    gmix::CloseSession(m_pGmixSession);
    NN_RESULT_SUCCESS;
}

Result AudioGmixDriver::Wake() NN_NOEXCEPT
{
    const auto gmixFormat = m_ChannelCount <= 2 ? gmix::Session::Format::Stereo : gmix::Session::Format::Surround;
    gmix::OpenSession(&m_pGmixSession, m_Name, m_SessionId, gmix::Session::Name::Default, gmixFormat, gmix::Session::Mode::Default, reinterpret_cast<uint64_t>(m_AppletResourceUserId.lower));
    m_pGmixSession->Start();
    NN_RESULT_SUCCESS;
}

uint64_t AudioGmixDriver::GetSamplesProcessed() const NN_NOEXCEPT
{
    if(!m_pGmixSession)
    {
        return 0;
    }
    return m_pGmixSession->GetSamplesProcessed() + m_GmixSamplesProcessed;
}

void* AudioGmixDriver::MapBuffer(server::SessionBuffer& buffer, nn::dd::ProcessHandle processHandle) NN_NOEXCEPT
{
    auto iovaAddr = nne::audio::iova::Map(processHandle, buffer.addr, buffer.size);
    buffer.mappedAddr = reinterpret_cast<void*>(iovaAddr);
    return reinterpret_cast<void*>(iovaAddr);
}

void* AudioGmixDriver::MapBuffer(server::SessionBuffer& buffer, nn::sf::NativeHandle&& bufferHandle) NN_NOEXCEPT
{
    //no-op: This is for mapping user buffer to audio process
    NN_UNUSED(buffer);
    NN_UNUSED(bufferHandle);
    buffer.mappedAddr = nullptr;
    return nullptr;
}

bool AudioGmixDriver::UnmapBuffer(const server::SessionBuffer& buffer) NN_NOEXCEPT
{
    uintptr_t iova = reinterpret_cast<uint64_t>(buffer.mappedAddr);
    nne::audio::iova::Unmap(buffer.addr, buffer.size, static_cast<nne::audio::iova_t>(iova));
    return true;
}

bool AudioGmixDriver::TryCompleteBuffer(const server::SessionBuffer& buffer) NN_NOEXCEPT
{
    auto bufferAddress = GetBufferAddress();

    if(bufferAddress != buffer.addr)
    {
        UnmapBuffer(buffer);
        return true;
    }
    return false;
}

Result AudioGmixDriver::AppendBuffer(const server::SessionBuffer& buffer) NN_NOEXCEPT
{
    uintptr_t iova = reinterpret_cast<uint64_t>(buffer.mappedAddr);
    m_pGmixSession->AppendBuffer(m_ClientProcessHandle, buffer.addr, buffer.size, static_cast<nne::audio::iova_t>(iova));
    NN_RESULT_SUCCESS;
}

void* AudioGmixDriver::GetBufferAddress() const NN_NOEXCEPT
{
    return m_pGmixSession->GetBufferAddress();
}

nne::audio::gmix::Session::Name AudioGmixDriver::SessionTypeToGmixType(server::SessionType sessionType) const NN_NOEXCEPT
{
    switch (sessionType)
    {
        case server::SessionType_Output:
            return gmix::Session::Name::AudioOut;
        case server::SessionType_Input:
            return gmix::Session::Name::NearVoice;
        case server::SessionType_Record:
            return gmix::Session::Name::GameRecord;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    return gmix::Session::Name::Undefined;
}

Result AudioGmixDriver::SetDeviceGain(float gain) NN_NOEXCEPT
{
    NN_UNUSED(gain);
    NN_RESULT_THROW(ResultNotSupported());
}

float AudioGmixDriver::GetDeviceGain() const NN_NOEXCEPT
{
    return 1.0f;
}

}  // namespace detail
}  // namespace audio
}  // namespace nn


