﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/audio/detail/audio_AudioRendererTypesInternal.h>
#include "../audio_SplitterInfoManager.h"
#include "../common/audio_WorkBufferAllocator.h"
#include "audio_ServiceBehaviorInfo.h"

namespace nn {
namespace audio {
namespace server {

class SplitterDestinationData;
class SplitterInfo;
class SplitterContext;

class NN_AUDIO_INFOTYPE_FILED_ALIGN SplitterDestinationData
{
    friend SplitterContext;
    friend SplitterInfo;

private:
    common::SplitterDestinationId m_Id;
    MixId m_DestinationId;
    float m_MixVolume[MixBufferCountMax];
    float m_MixVolumePrev[MixBufferCountMax];
    SplitterDestinationData* m_Next;
    bool m_InUse;
    bool m_NeedToUpdateInternalState;
    int8_t _reserved[NN_AUDIO_ADDRESS_SELECT(12, 8)];

    void ClearMixVolume() NN_NOEXCEPT;

public:
    explicit SplitterDestinationData(int index) NN_NOEXCEPT;
    common::SplitterDestinationId GetId() const NN_NOEXCEPT;
    MixId GetMixId() const NN_NOEXCEPT;
    bool  IsConfigured() const NN_NOEXCEPT;
    float GetMixVolume(int index) const NN_NOEXCEPT;
    float GetMixVolumePrev(int index) const NN_NOEXCEPT;
    const float* GetMixVolume() const NN_NOEXCEPT;
    const float* GetMixVolumePrev() const NN_NOEXCEPT;

    size_t Update(const nn::audio::SplitterDestinationData::InParameter* inParam) NN_NOEXCEPT;
    void MarkAsNeedToUpdateInternalState() NN_NOEXCEPT;
    void UpdateInternalState() NN_NOEXCEPT;
};
#if defined(NN_BUILD_CONFIG_SPEC_NX) && !defined(NN_BUILD_CONFIG_OS_WIN)
NN_AUDIO_INFOTYPE_CHECK(SplitterDestinationData, 224);
#endif

class NN_AUDIO_INFOTYPE_FILED_ALIGN SplitterInfo
{
    friend SplitterContext;

private:
    int32_t m_Id;
    int32_t m_SampleRate;
    int32_t m_SendDataIdLength;
    bool m_HasNewConnection;
    int8_t _reserved1[3];
    SplitterDestinationData* m_SendDataHead;
    int8_t _reserved2[NN_AUDIO_ADDRESS_SELECT(12, 8)];

public:
    static void InitializeInfos(SplitterInfo* splitterInfos, int splitterInfoCount) NN_NOEXCEPT;

    explicit SplitterInfo(int32_t id) NN_NOEXCEPT;
    size_t Update(const nn::audio::SplitterInfo::InParameter* pInParameter) NN_NOEXCEPT;
    SplitterDestinationData* GetData(int index) const NN_NOEXCEPT;
    int GetDestinationCount() const NN_NOEXCEPT;

    bool HasNewConnection() const NN_NOEXCEPT;
    void ClearNewConnectionFlag() NN_NOEXCEPT;
};
#if defined(NN_BUILD_CONFIG_SPEC_NX) && !defined(NN_BUILD_CONFIG_OS_WIN)
NN_AUDIO_INFOTYPE_CHECK(SplitterInfo, 32);
#endif

class SplitterContext
{
    server::SplitterInfo* m_Infos;
    int m_InfoCount;
    server::SplitterDestinationData* m_Datas;
    int m_DataCount;
    bool m_SplitterBugFixed;

public:
    static size_t CalcWorkBufferSize(
        const server::BehaviorInfo& behavior,
        const ::nn::audio::detail::AudioRendererParameterInternal& parameter) NN_NOEXCEPT;

public:
    SplitterContext() NN_NOEXCEPT = default;

    bool Initialize(
        const server::BehaviorInfo& behavior,
        const ::nn::audio::detail::AudioRendererParameterInternal &parameter,
        common::WorkBufferAllocator& allocator) NN_NOEXCEPT;

    server::SplitterInfo& GetInfo(int id) const NN_NOEXCEPT;
    server::SplitterDestinationData& GetData(int id) const NN_NOEXCEPT;
    int GetInfoCount() const NN_NOEXCEPT;
    int GetDataCount() const NN_NOEXCEPT;

    SplitterDestinationData* GetDestinationData(int infoId, int dataIndex) const NN_NOEXCEPT;
    bool UsingSplitter() const NN_NOEXCEPT;
    void ClearAllNewConnectionFlag() NN_NOEXCEPT;

    void Setup(
        server::SplitterInfo* infos,
        int infoCount,
        server::SplitterDestinationData* datas,
        int dataCount,
        bool splitterBugFixed) NN_NOEXCEPT;

    bool Update(uintptr_t inParameter, size_t& consumedSize) NN_NOEXCEPT;
    void UpdateInternalState() NN_NOEXCEPT;

private:
    int GetDestCountPerInfoForCompat() const NN_NOEXCEPT;
    void RecomposeDestination(SplitterInfo& info, const nn::audio::SplitterInfo::InParameter* pInParameter) NN_NOEXCEPT;
    size_t UpdateInfo(nn::util::ConstBytePtr& base, size_t startOffset, int infoCount) NN_NOEXCEPT;
    size_t UpdateData(nn::util::ConstBytePtr& base, size_t startOffset, int dataCount) NN_NOEXCEPT;
};

}  // namespace server
}  // namespace audio
}  // namespace nn
