﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/audio/detail/audio_Log.h>
#include "audio_ServiceSplitterInfo.h"
#include "../common/audio_CommonWaveBuffer.h"
#include "../common/audio_Util.h"
#include "../common/audio_SplitterParameters.h"

namespace nn {
namespace audio {
namespace server {

SplitterDestinationData::SplitterDestinationData(int index) NN_NOEXCEPT
    : m_Id(index)
    , m_DestinationId(Invalid_MixId)
    , m_Next(nullptr)
    , m_InUse(false)
    , m_NeedToUpdateInternalState(false)
{
    ClearMixVolume();
    memset(_reserved, 0, sizeof(_reserved));
}

void SplitterDestinationData::ClearMixVolume() NN_NOEXCEPT
{
    for (auto& vol : m_MixVolume)
    {
        vol = 0.0f;
    }
    for (auto& vol : m_MixVolumePrev)
    {
        vol = 0.0f;
    }
}

nn::audio::common::SplitterDestinationId SplitterDestinationData::GetId() const NN_NOEXCEPT
{
    return m_Id;
}

bool SplitterDestinationData::IsConfigured() const NN_NOEXCEPT
{
    return m_InUse && m_DestinationId != Invalid_MixId;
}

nn::audio::MixId SplitterDestinationData::GetMixId() const NN_NOEXCEPT
{
    return m_DestinationId;
}

float SplitterDestinationData::GetMixVolume(int index) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(index, 0, MixBufferCountMax);
    if (0 <= index && index < MixBufferCountMax)
    {
        return m_MixVolume[index];
    }
    return 0.0f;
}

const float* SplitterDestinationData::GetMixVolume() const NN_NOEXCEPT
{
    return m_MixVolume;
}

float SplitterDestinationData::GetMixVolumePrev(int index) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(index, 0, MixBufferCountMax);
    if (0 <= index && index < MixBufferCountMax)
    {
        return m_MixVolumePrev[index];
    }
    return 0.0f;
}

const float* SplitterDestinationData::GetMixVolumePrev() const NN_NOEXCEPT
{
    return m_MixVolumePrev;
}

size_t SplitterDestinationData::Update(const nn::audio::SplitterDestinationData::InParameter* inParam) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(inParam);
    NN_SDK_ASSERT_EQUAL(m_Id, inParam->_id);
    NN_SDK_ASSERT_EQUAL(inParam->magic, common::GetSplitterSendDataMagic());

    if (inParam != nullptr &&
        inParam->magic == common::GetSplitterSendDataMagic() &&
        inParam->_id == m_Id)
    {
        m_DestinationId = inParam->_destinationId;
        for (auto i = 0; i < MixBufferCountMax; ++i)
        {
            m_MixVolume[i] = inParam->_mixVolume[i];
        }
        if (!m_InUse && inParam->_inUse)
        {
            std::memcpy(m_MixVolumePrev, m_MixVolume, sizeof(m_MixVolume));
            m_NeedToUpdateInternalState = false;
        }
        m_InUse = inParam->_inUse;
//        NN_DETAIL_AUDIO_TRACE("[SendData] Update[%d]: dest:%d (%d)\n", m_Id, m_DestinationId, m_InUse);
    }
    return sizeof(nn::audio::SplitterDestinationData::InParameter);
}

void SplitterDestinationData::MarkAsNeedToUpdateInternalState() NN_NOEXCEPT
{
    m_NeedToUpdateInternalState = true;
}

void SplitterDestinationData::UpdateInternalState() NN_NOEXCEPT
{
    if(m_InUse && m_NeedToUpdateInternalState)
    {
        std::memcpy(m_MixVolumePrev, m_MixVolume, sizeof(m_MixVolume));
    }

    m_NeedToUpdateInternalState = false;
}

//////////////////////////////////////////////////////////////////////////

SplitterInfo::SplitterInfo(int32_t id) NN_NOEXCEPT
    : m_Id(id)
    , m_SampleRate(0)
    , m_SendDataIdLength(0)
    , m_HasNewConnection(false)
    , m_SendDataHead(nullptr)
{
    memset(_reserved1, 0, sizeof(_reserved1));
    memset(_reserved2, 0, sizeof(_reserved2));
}

void SplitterInfo::InitializeInfos(SplitterInfo* splitterInfos, int splitterInfoCount) NN_NOEXCEPT
{
    if (splitterInfos == nullptr)
    {
        return;
    }

    for (auto i = 0; i < splitterInfoCount; ++i)
    {
        splitterInfos[i].m_SendDataIdLength = 0;
        splitterInfos[i].m_SendDataHead = nullptr;
        splitterInfos[i].m_HasNewConnection = true;
    }
}

size_t SplitterInfo::Update(const nn::audio::SplitterInfo::InParameter* pInParameter) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(pInParameter->sendId, m_Id);
    if (pInParameter->sendId != m_Id)
    {
        return 0;
    }

    m_SampleRate = pInParameter->sampleRate;
    m_HasNewConnection = true; // Node connection is updated.

    auto headersize = sizeof(nn::audio::SplitterInfo::InParameter);
    auto dataSize = sizeof(nn::audio::common::SplitterDestinationId) * (pInParameter->length - 1);
    return headersize + dataSize;
}

SplitterDestinationData* SplitterInfo::GetData(int index) const NN_NOEXCEPT
{
    auto cur = m_SendDataHead;
    for (auto i = 0; i < index; ++i)
    {
        if (cur == nullptr || cur->m_Next == nullptr)
        {
            return nullptr;
        }
        cur = cur->m_Next;
    }
    return cur;
}

int SplitterInfo::GetDestinationCount() const NN_NOEXCEPT
{
    return m_SendDataIdLength;
}

bool SplitterInfo::HasNewConnection() const NN_NOEXCEPT
{
    return m_HasNewConnection;
}

void SplitterInfo::ClearNewConnectionFlag() NN_NOEXCEPT
{
    m_HasNewConnection = false;
}

// -----------------------------------------------------------------------------

nn::audio::server::SplitterDestinationData* SplitterContext::GetDestinationData(int infoId, int dataIndex) const NN_NOEXCEPT
{
    return GetInfo(infoId).GetData(dataIndex);
}

int SplitterContext::GetDataCount() const NN_NOEXCEPT
{
    return m_DataCount;
}

int SplitterContext::GetInfoCount() const NN_NOEXCEPT
{
    return m_InfoCount;
}

nn::audio::server::SplitterDestinationData& SplitterContext::GetData(int id) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(id, 0, m_DataCount);
    return m_Datas[id];
}

nn::audio::server::SplitterInfo& SplitterContext::GetInfo(int id) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(id, 0, m_InfoCount);
    return m_Infos[id];
}

void SplitterContext::Setup(server::SplitterInfo* infos, int infoCount, server::SplitterDestinationData* datas, int dataCount, bool splitterBugFixed) NN_NOEXCEPT
{
    m_Infos = infos;
    m_InfoCount = infoCount;
    m_Datas = datas;
    m_DataCount = dataCount;
    m_SplitterBugFixed = splitterBugFixed;
}

bool SplitterContext::UsingSplitter() const NN_NOEXCEPT
{
    return (m_Infos != nullptr) &&
           (m_InfoCount > 0) &&
           (m_Datas != nullptr) &&
           (m_DataCount > 0);
}

void SplitterContext::ClearAllNewConnectionFlag() NN_NOEXCEPT
{
    for (auto i = 0; i < m_InfoCount; ++i)
    {
        m_Infos[i].ClearNewConnectionFlag();
    }
}

bool SplitterContext::Initialize(const server::BehaviorInfo& behavior, const ::nn::audio::detail::AudioRendererParameterInternal &parameter, common::WorkBufferAllocator& allocator) NN_NOEXCEPT
{
    if (!(behavior.IsSplitterSupported() && parameter.splitterCount > 0 && parameter.splitterSendChannelCount > 0))
    {
        Setup(nullptr, 0, nullptr, 0, false);
        return true;
    }

    // info
    auto splitterInfoCount = parameter.splitterCount;
    auto pSplitterInfos = common::PrepareArrayWithIndex<server::SplitterInfo>(allocator.Allocate(sizeof(server::SplitterInfo) * splitterInfoCount), splitterInfoCount);
    if (pSplitterInfos == nullptr)
    {
        return false;
    }
    // send
    auto sendChCnt = parameter.splitterSendChannelCount;
    auto pSendDatas = common::PrepareArrayWithIndex<server::SplitterDestinationData>(allocator.Allocate(sizeof(server::SplitterDestinationData) * sendChCnt), sendChCnt);
    if (pSendDatas == nullptr)
    {
        return false;
    }

    server::SplitterInfo::InitializeInfos(pSplitterInfos, splitterInfoCount);
    Setup(pSplitterInfos, splitterInfoCount, pSendDatas, sendChCnt, behavior.IsSplitterBugFixed());

    return true;
}

bool SplitterContext::Update(const uintptr_t inParameter, size_t& outSize) NN_NOEXCEPT
{
    if (GetDataCount() <= 0 || GetInfoCount() <= 0)
    {
        // no send info passed
        outSize = 0;
        return true;
    }
    auto base = nn::util::ConstBytePtr(reinterpret_cast<const void*>(inParameter));
    auto paramHeader = reinterpret_cast<const nn::audio::SplitterInfo::InParameterHeader*>(base.Get());
    if (paramHeader->magic != common::GetSplitterInParamHeaderMagic())
    {
        return false;
    }

    // SplitterInfo
    ClearAllNewConnectionFlag();
    size_t consumedSize = sizeof(nn::audio::SplitterInfo::InParameterHeader);
    consumedSize = UpdateInfo(base, consumedSize, paramHeader->infoCount);
    consumedSize = UpdateData(base, consumedSize, paramHeader->dataCount);
    outSize = nn::util::align_up(consumedSize, nn::audio::common::InfoTypeFieldAlignSize);
    return true;
}

void SplitterContext::UpdateInternalState() NN_NOEXCEPT
{
    SplitterDestinationData* cur = m_Datas;
    while (cur != nullptr)
    {
        cur->UpdateInternalState();
        cur = cur->m_Next;
    }
}

void SplitterContext::RecomposeDestination(SplitterInfo& info, const nn::audio::SplitterInfo::InParameter* pInParameter) NN_NOEXCEPT
{
    // return current dest data instances
    SplitterDestinationData* cur = info.m_SendDataHead;
    while (cur != nullptr)
    {
        auto next = cur->m_Next;
        // DestinationData の更新は、このあと UpdateData() 内で行われる。
        // よってここで、接続情報以外は触ってはいけない。
        cur->m_Next = nullptr;
        cur = next;
    }
    info.m_SendDataHead = nullptr;

    // recompose dest data instances
    int newLength = m_SplitterBugFixed ?
        pInParameter->length :
        std::min(pInParameter->length, GetDestCountPerInfoForCompat());
    if (newLength <= 0)
    {
        return;
    }

    auto* head = &GetData(pInParameter->resourceIds[0]);
    cur = head;
    for (auto i = 1; i < newLength; i++)
    {
        auto* ins = &GetData(pInParameter->resourceIds[i]);
        cur->m_Next = ins;
        cur = ins;
    }

    info.m_SendDataHead = head;
    info.m_SendDataIdLength = newLength;
}

int SplitterContext::GetDestCountPerInfoForCompat() const NN_NOEXCEPT
{
    return (m_InfoCount > 0) ? m_DataCount / m_InfoCount : 0;
}

size_t SplitterContext::UpdateInfo(nn::util::ConstBytePtr& base, size_t startOffset, int splitterInfoCount) NN_NOEXCEPT
{
    auto consumedSize = startOffset;
    auto infoCount = 0;

    while (infoCount++ < splitterInfoCount)
    {
        auto param = (base + consumedSize).Get<const nn::audio::SplitterInfo::InParameter>();
        if (param->magic != common::GetSplitterInfoMagic() ||
            param->sendId < 0 ||
            GetInfoCount() <= param->sendId)
        {
            continue;
        }

        auto& info = GetInfo(param->sendId);
        RecomposeDestination(info, param);
        consumedSize += info.Update(param);
    }

    return consumedSize;
}

size_t SplitterContext::UpdateData(nn::util::ConstBytePtr& base, size_t startOffset, int destinationDataCount) NN_NOEXCEPT
{
    auto consumedSize = startOffset;
    auto dataCount = 0;

    while (dataCount++ < destinationDataCount)
    {
        auto param = (base + consumedSize).Get<const nn::audio::SplitterDestinationData::InParameter>();
        if (param->magic != common::GetSplitterSendDataMagic() ||
            param->_id < 0 ||
            GetDataCount() <= param->_id)
        {
            continue;
        }

        consumedSize += GetData(param->_id).Update(param);
    }
    return consumedSize;
}

size_t SplitterContext::CalcWorkBufferSize(const server::BehaviorInfo& behavior, const ::nn::audio::detail::AudioRendererParameterInternal& parameter) NN_NOEXCEPT
{
    size_t size = 0;

    if (behavior.IsSplitterSupported() == false)
    {
        return size;
    }

    auto splitterSendChannelCount = parameter.splitterSendChannelCount;
    size += nn::util::align_up(sizeof(server::SplitterInfo) * parameter.splitterCount, ::nn::audio::common::InfoTypeFieldAlignSize);
    size += nn::util::align_up(sizeof(server::SplitterDestinationData) * splitterSendChannelCount, ::nn::audio::common::InfoTypeFieldAlignSize);
    if (behavior.IsSplitterBugFixed() == false)
    {
        size += nn::util::align_up(sizeof(common::SplitterDestinationId) * splitterSendChannelCount, ::nn::audio::common::InfoTypeFieldAlignSize);
    }

    return size;
}

}  // namespace server
}  // namespace audio
}  // namespace nn
