﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring> // memset, memcpy
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include "audio_ServiceSinkInfo.h"


namespace nn { namespace audio { namespace server {

SinkInfoBase::SinkInfoBase() NN_NOEXCEPT
    : m_Type(common::SinkType_Invalid)
    , m_IsInUse(false)
    , m_ShouldSkip(false)
{
    memset(&m_ParameterBuffer, 0, sizeof(m_ParameterBuffer));
    memset(&m_StateBuffer, 0, sizeof(m_StateBuffer));
}

void SinkInfoBase::Update(common::BehaviorParameter::ErrorInfo* pOutErrorInfo, common::SinkOutStatus* pOutStatus, const common::SinkInParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(pUserInParam->_type, common::SinkType_Invalid);
    NN_SDK_ASSERT_EQUAL(m_Type, common::SinkType_Invalid);
    NN_UNUSED(pUserInParam);
    NN_UNUSED(poolMapper);
    memset(pOutStatus, 0, sizeof(common::SinkOutStatus));

    pOutErrorInfo->SetResult(ResultSuccess(), 0);
}

void SinkInfoBase::UpdateForCommandGeneration() NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(m_Type, common::SinkType_Invalid);
}

nn::audio::server::DeviceSinkState* SinkInfoBase::GetDeviceState() NN_NOEXCEPT
{
    return reinterpret_cast<DeviceSinkState*>(m_StateBuffer);
}

nn::audio::common::SinkType SinkInfoBase::GetType() const NN_NOEXCEPT
{
    return m_Type;
}

bool SinkInfoBase::IsUsed() const NN_NOEXCEPT
{
    return m_IsInUse;
}

bool SinkInfoBase::ShouldSkip() const NN_NOEXCEPT
{
    return m_ShouldSkip;
}

NodeId SinkInfoBase::GetNodeId() const NN_NOEXCEPT
{
    return m_NodeId;
}

void SinkInfoBase::CleanUp() NN_NOEXCEPT
{
    m_Type = common::SinkType_Invalid;
    m_IsInUse = false;
}

const void* SinkInfoBase::GetState() const NN_NOEXCEPT
{
    return reinterpret_cast<const void*>(m_StateBuffer);
}

const void* SinkInfoBase::GetParameter() const NN_NOEXCEPT
{
    return reinterpret_cast<const void*>(m_ParameterBuffer);
}

DeviceSinkInfo::DeviceSinkInfo() NN_NOEXCEPT
{
    m_Type = common::SinkType_Device;
    memset(GetParameter(), 0, sizeof(DeviceSinkState));
    m_IsInUse = false;
    m_ShouldSkip = false;
}

void DeviceSinkInfo::Update(common::BehaviorParameter::ErrorInfo* pOutErrorInfo, common::SinkOutStatus* pOutStatus, const common::SinkInParameter* fromUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(fromUserInParam->_type, common::SinkType_Device);
    NN_SDK_ASSERT_EQUAL(m_Type, common::SinkType_Device);
    NN_UNUSED(poolMapper);

    // update server side common::SinkInParameter
    if (m_IsInUse != fromUserInParam->_isInUse)
    {
        m_Type = fromUserInParam->_type;
        m_IsInUse = fromUserInParam->_isInUse;
        m_NodeId = fromUserInParam->_nodeId;
        *GetParameter() = fromUserInParam->_device;
    }
    else
    {
        // DownMixMatrix 関連だけは定常的に更新する
        GetParameter()->useDownMixMatrix = fromUserInParam->_device.useDownMixMatrix;
        memcpy(GetParameter()->downMixMatrixCoeff, fromUserInParam->_device.downMixMatrixCoeff, sizeof(float) * common::DeviceParameter::CoeffCount);
    }
    for (auto i = 0; i < common::DeviceParameter::CoeffCount; ++i)
    {
        GetState()->_downMixMatrixCoeff[i] = static_cast<int32_t>(GetParameter()->downMixMatrixCoeff[i] * (1 << 16));
    }

    // DeviceSink has no field to return to user side.
    memset(pOutStatus, 0 , sizeof(common::SinkOutStatus));

    pOutErrorInfo->SetResult(ResultSuccess(), 0);
}

void DeviceSinkInfo::CleanUp() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_EQUAL(m_Type, common::SinkType_Device);
    auto& samplerInfo = GetState()->pUpsamplerInfo;
    if (samplerInfo)
    {
        auto samplerManager = reinterpret_cast<server::UpsamplerManager*>(samplerInfo->pUpsamplerManager.GetPointer());
        samplerManager->Free(samplerInfo);
        samplerInfo = 0;
    }
    memset(GetParameter(), 0, sizeof(DeviceSinkState));
    SinkInfoBase::CleanUp();
}

void DeviceSinkInfo::UpdateForCommandGeneration() NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(m_Type, common::SinkType_Device);
}

void CircularBufferSinkState::CleanUp() NN_NOEXCEPT
{
    _currentPos = 0;
    _commandPos = 0;
    _previousCommandPos = 0;
    //_circularBuffer = 0;
    _circularBuffer.Setup(nullptr, 0);
}

CircularBufferSinkInfo::CircularBufferSinkInfo() NN_NOEXCEPT
{
    m_Type = common::SinkType_CircularBuffer;
    m_IsInUse = false;
    m_ShouldSkip = false;
    memset(GetParameter(), 0, sizeof(CircularBufferSinkState));
    GetState()->CleanUp();
}

bool CircularBufferSinkInfo::ShouldUpdateWorkBufferInfo(const common::SinkInParameter* pUserInParam) const NN_NOEXCEPT
{
    auto parameterChanged = (IsUsed() != pUserInParam->_isInUse);
    auto shouldRetry = m_ShouldSkip;
    return parameterChanged || shouldRetry;
}

void CircularBufferSinkInfo::Update(common::BehaviorParameter::ErrorInfo* pOutErrorInfo, common::SinkOutStatus* pOutStatus, const common::SinkInParameter* pUserInParam, PoolMapper& poolMapper) NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(pUserInParam->_type, common::SinkType_CircularBuffer);
    NN_SDK_ASSERT_EQUAL(m_Type, common::SinkType_CircularBuffer);

    if (ShouldUpdateWorkBufferInfo(pUserInParam))
    {
        m_NodeId = pUserInParam->_nodeId;
        if ((m_IsInUse = pUserInParam->_isInUse) == true)// in use.
        {
            NN_SDK_ASSERT(GetState()->_circularBuffer.GetCpuAddr() == nullptr);
            NN_SDK_ASSERT(GetState()->_circularBuffer.GetReference(false) == NULL);
            m_ShouldSkip =
                poolMapper.TryAttachBuffer(
                    pOutErrorInfo,
                    &GetState()->_circularBuffer,
                    pUserInParam->_circular._bufferAddress,
                    pUserInParam->_circular._size) ? false : true;
            *GetParameter() = pUserInParam->_circular;
        }
        else // not in use
        {
            NN_SDK_ASSERT(GetState()->_circularBuffer.GetCpuAddr() != nullptr);
            NN_SDK_ASSERT(GetState()->_circularBuffer.GetReference(false) != NULL);
            // TODO: unmap user pointer
            *GetParameter() = pUserInParam->_circular;
        }
    }
    else
    {
        pOutErrorInfo->SetResult(ResultSuccess(), 0);
    }

    pOutStatus->circular._currentPosition = GetState()->_currentPos;
}


void CircularBufferSinkInfo::CleanUp() NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(m_Type, common::SinkType_CircularBuffer);
    memset(GetState(), 0, sizeof(CircularBufferSinkState));
    SinkInfoBase::CleanUp();
}

void CircularBufferSinkInfo::UpdateForCommandGeneration() NN_NOEXCEPT
{
    NN_SDK_ASSERT_EQUAL(m_Type, common::SinkType_CircularBuffer);
    auto param = GetParameter();
    auto state = GetState();
    if (IsUsed())
    {
        state->_currentPos = state->_previousCommandPos;
        state->_previousCommandPos = state->_commandPos;
        state->_commandPos += param->_sampleCount * param->_inputCount * sizeof(int16_t);
        if (param->_size > 0)
        {
            state->_commandPos %= param->_size;
        }
    }
}

SinkContext::SinkContext() NN_NOEXCEPT
    : m_Infos(nullptr)
    , m_InfoCount(0)
{}

void SinkContext::Initialize(server::SinkInfoBase* sinkInfos, int infoCount) NN_NOEXCEPT
{
    m_Infos = sinkInfos;
    m_InfoCount = infoCount;
}

nn::audio::server::SinkInfoBase& SinkContext::GetInfo(int id) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(id, 0, m_InfoCount);
    return m_Infos[id];
}

int SinkContext::GetCount() const NN_NOEXCEPT
{
    return m_InfoCount;
}

}}}  // namespace nn::audio::server
