﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring> // memset, memcpy
#include <algorithm> // min
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include "audio_ServiceBehaviorInfo.h"


namespace nn { namespace audio { namespace server {

BehaviorInfo::BehaviorInfo() NN_NOEXCEPT
    : m_ProcessRevision(common::GetCurrentRevision())
    , m_UserLibRevision(0)
    , m_ErrorCount(0)
{
    m_Flags.Reset();
    for (auto i = 0; i < common::BehaviorParameter::ErrorInfoCountMax; ++i)
    {
        m_ErrorInfo[i].SetResult(ResultSuccess(), 0);
    }
}

nn::audio::common::RevisionInfo BehaviorInfo::GetProcessRevision() const NN_NOEXCEPT
{
    return m_ProcessRevision;
}

nn::audio::common::RevisionInfo BehaviorInfo::GetUserLibRevision() const NN_NOEXCEPT
{
    return m_UserLibRevision;
}

void BehaviorInfo::ClearError() NN_NOEXCEPT
{
    m_ErrorCount = 0;
}

void BehaviorInfo::AppendError(common::BehaviorParameter::ErrorInfo& errorInfo) NN_NOEXCEPT
{
    // save only first "ErrorInfoCountMax" errors.
    if (m_ErrorCount < common::BehaviorParameter::ErrorInfoCountMax)
    {
        m_ErrorInfo[m_ErrorCount] = errorInfo;
        ++m_ErrorCount;
    }
}

void BehaviorInfo::SetUserLibRevision(common::RevisionInfo revision) NN_NOEXCEPT
{
    m_UserLibRevision = revision;
}

bool BehaviorInfo::IsMemoryPoolForceMappingEnabled() const NN_NOEXCEPT
{
    return m_Flags[common::BehaviorParameter::FlagIndex_MemoryPoolAssert];
}

void BehaviorInfo::UpdateFlags(const common::BehaviorParameter::Flags& flag) NN_NOEXCEPT
{
    m_Flags = flag;
}

void BehaviorInfo::CopyErrorInfo(common::BehaviorParameter::ErrorInfo* pDest, int* errorInfoCount) const NN_NOEXCEPT
{
    auto infoCount = std::min(common::BehaviorParameter::ErrorInfoCountMax, m_ErrorCount);
    *errorInfoCount = infoCount;
    memset(pDest, 0, sizeof(common::BehaviorParameter::ErrorInfo) * common::BehaviorParameter::ErrorInfoCountMax);

    auto count = std::min(m_ErrorCount, infoCount);
    for (auto i = 0; i < count; ++i)
    {
        pDest[i] = m_ErrorInfo[i];
    }
}

bool BehaviorInfo::IsAdpcmLoopContextBugFixed() const NN_NOEXCEPT
{
    return common::CheckFeatureSupported(common::SupportTags::AdpcmLoopContextBugFix, GetUserLibRevision());
}

bool BehaviorInfo::IsSplitterSupported() const NN_NOEXCEPT
{
    return common::CheckFeatureSupported(common::SupportTags::Splitter, GetUserLibRevision());
}

bool BehaviorInfo::IsLongSizePreDelaySupported() const NN_NOEXCEPT
{
    return common::CheckFeatureSupported(common::SupportTags::LongSizePreDelay, GetUserLibRevision());
}

bool BehaviorInfo::IsAudioRenererProcessingTimeLimit70PercentSupported() const NN_NOEXCEPT
{
    return common::CheckFeatureSupported(common::SupportTags::AudioRendererProcessingTimeLimit70Percent, GetUserLibRevision());
}

bool BehaviorInfo::IsAudioRenererProcessingTimeLimit75PercentSupported() const NN_NOEXCEPT
{
    return common::CheckFeatureSupported(common::SupportTags::AudioRendererProcessingTimeLimit75Percent, GetUserLibRevision());
}

bool BehaviorInfo::IsAudioRenererProcessingTimeLimit80PercentSupported() const NN_NOEXCEPT
{
    return common::CheckFeatureSupported(common::SupportTags::AudioRendererProcessingTimeLimit80Percent, GetUserLibRevision());
}

bool BehaviorInfo::IsCommandProcessingTimeEstimatorVersion2Supported() const NN_NOEXCEPT
{
    return common::CheckFeatureSupported(common::SupportTags::CommandProcessingTimeEstimatorVersion2, GetUserLibRevision());
}

bool BehaviorInfo::IsVariadicCommandBufferSizeSupported() const NN_NOEXCEPT
{
    return common::CheckFeatureSupported(common::SupportTags::AudioRendererVariadicCommandBufferSize, GetUserLibRevision());
}

bool BehaviorInfo::IsFlushVoiceWaveBuffersSupported() const NN_NOEXCEPT
{
    return common::CheckFeatureSupported(common::SupportTags::FlushVoiceWaveBuffers, GetUserLibRevision());
}

bool BehaviorInfo::IsSplitterBugFixed() const NN_NOEXCEPT
{
    return common::CheckFeatureSupported(common::SupportTags::SplitterBugFix, GetUserLibRevision());
}

bool BehaviorInfo::IsElapsedFrameCountSupported() const NN_NOEXCEPT
{
    return common::CheckFeatureSupported(common::SupportTags::ElapsedFrameCount, GetUserLibRevision());
}

bool BehaviorInfo::IsPerformanceMetricsDataFormatVersion2Supported() const NN_NOEXCEPT
{
    return common::CheckFeatureSupported(common::SupportTags::PerformanceMetricsDataFormatVersion2, GetUserLibRevision());
}

PerformanceMetricsDataFormat BehaviorInfo::GetPerformanceMetricsDataFormat() const NN_NOEXCEPT
{
    if(IsPerformanceMetricsDataFormatVersion2Supported())
    {
        return PerformanceMetricsDataFormat::Version2;
    }
    else
    {
        return PerformanceMetricsDataFormat::Version1;
    }
}

bool BehaviorInfo::IsAudioCodecSupported() const NN_NOEXCEPT
{
    return false;
}

bool BehaviorInfo::IsVoicePitchAndSrcSkippedSupported() const NN_NOEXCEPT
{
    return common::CheckFeatureSupported(common::SupportTags::VoicePitchAndSrcSkipped, GetUserLibRevision());
}

bool BehaviorInfo::IsVoicePlayedSampleCountResetAtLoopPointSupported() const NN_NOEXCEPT
{
    return common::CheckFeatureSupported(common::SupportTags::VoicePlayedSampleCountResetAtLoopPoint, GetUserLibRevision());
}

}}}  // namespace nn::audio::server
