﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkLog.h>
#include "audio_PerformanceMeasuredCommandBuffer.h"

#include <nn/audio/audio_PerformanceMetricsTypes.private.h>

namespace nn { namespace audio { namespace server {

PerfMeasureAspect::PerfMeasureAspect(ICommandBuffer* publisher, PerformanceManager* pPerformanceManager, NodeId parentId, PerformanceSysDetailType target) NN_NOEXCEPT
    : BaseAspect(publisher)
    , m_ParentId(parentId)
    , m_Published(false)
    , m_PerformanceManager(pPerformanceManager)
{
    if (m_PerformanceManager != nullptr &&
        m_PerformanceManager->IsInitialized() &&
        m_PerformanceManager->IsDetailTarget(m_ParentId))
    {

        if(m_PerformanceManager->GetNextEntry(&m_Addresses, &m_pEstimatedProcessingTime, target, m_ParentId))
        {
            m_Published = true;
            m_Publisher->GeneratePerformanceCommand(&m_Addresses, PerformanceCommandType_Begin, m_ParentId);
        }
    }
}

PerfMeasureAspect::~PerfMeasureAspect() NN_NOEXCEPT
{
    if (m_Published)
    {
        m_Publisher->GeneratePerformanceCommand(&m_Addresses, PerformanceCommandType_End, m_ParentId);
    }
}

PerformanceMeasuredCommandBuffer::PerformanceMeasuredCommandBuffer(ICommandBuffer* pCommandBuffer, PerformanceManager* pPerformanceManager) NN_NOEXCEPT
    : m_pCommandBuffer(pCommandBuffer),
      m_PerformanceManager(pPerformanceManager)
{
    NN_SDK_ASSERT_NOT_NULL(pCommandBuffer);
}

PerfMeasureAspect PerformanceMeasuredCommandBuffer::CreateAspect(NodeId parentId, PerformanceSysDetailType target) NN_NOEXCEPT
{
    return PerfMeasureAspect(m_pCommandBuffer, m_PerformanceManager, parentId, target);
}

PcmInt16DataSourceCommand* PerformanceMeasuredCommandBuffer::GeneratePcmInt16DataSourceCommand(const server::VoiceInfo* pVoice, VoiceState* state, int mixBufferCount, int32_t channel, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_PcmInt16DataSource)
        .Invoke(&ICommandBuffer::GeneratePcmInt16DataSourceCommand, pVoice, state, mixBufferCount, channel, nodeId);
}

AdpcmDataSourceCommand* PerformanceMeasuredCommandBuffer::GenerateAdpcmDataSourceCommand(const server::VoiceInfo* pVoice, VoiceState* state, int mixBufferCount, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_AdpcmDataSource)
        .Invoke(&ICommandBuffer::GenerateAdpcmDataSourceCommand, pVoice,  state, mixBufferCount, nodeId);
}

AuxCommand* PerformanceMeasuredCommandBuffer::GenerateAuxCommand(int mixBufferOffset, int8_t input, int8_t output, const AuxBufferAddresses* pAddresses, bool enabled, int32_t sampleCount, DspAddr sendBuffer, DspAddr returnBuffer, int updateCount, int offset, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_Aux)
        .Invoke(&ICommandBuffer::GenerateAuxCommand, mixBufferOffset, input, output, pAddresses, enabled, sampleCount, sendBuffer, returnBuffer, updateCount, offset, nodeId);
}

DelayCommand* PerformanceMeasuredCommandBuffer::GenerateDelayEffectCommand(int mixBufferOffset, const DelayParameter* pParameter, void* pState, bool enabled, DspAddr workBuffer, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_Delay)
        .Invoke(&ICommandBuffer::GenerateDelayEffectCommand, mixBufferOffset, pParameter, pState, enabled, workBuffer, nodeId);
}

ReverbCommand* PerformanceMeasuredCommandBuffer::GenerateReverbEffectCommand(int mixBufferOffset, const ReverbParameter* pParameter, void* pState, bool enabled, bool longSizePreDelaySupported, DspAddr workBuffer, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_Reverb)
        .Invoke(&ICommandBuffer::GenerateReverbEffectCommand, mixBufferOffset, pParameter, pState, enabled, longSizePreDelaySupported, workBuffer, nodeId);
}

I3dl2ReverbCommand* PerformanceMeasuredCommandBuffer::GenerateI3dl2ReverbEffectCommand(int mixBufferOffset, const I3dl2ReverbParameter* pParameter, void* pState, bool enabled, DspAddr workBuffer, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_I3dl2Reverb)
        .Invoke(&ICommandBuffer::GenerateI3dl2ReverbEffectCommand, mixBufferOffset, pParameter, pState, enabled, workBuffer, nodeId);
}

CopyMixBufferCommand* PerformanceMeasuredCommandBuffer::GenerateCopyMixBufferCommand(int inputBufferIndex, int outputBufferIndex, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_CopyMixBuffer)
        .Invoke(&ICommandBuffer::GenerateCopyMixBufferCommand, inputBufferIndex, outputBufferIndex, nodeId);
}

BiquadFilterCommand* PerformanceMeasuredCommandBuffer::GenerateBiquadFilterCommand(int mixBufferOffset, const BiquadFilterParameter* pFilterParameter, BiquadFilterState* pState, int8_t input, int8_t output, bool needInitialization, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_BiquadFilter)
        .Invoke(&ICommandBuffer::GenerateBiquadFilterCommand, mixBufferOffset, pFilterParameter, pState, input, output, needInitialization, nodeId);
}

MixCommand* PerformanceMeasuredCommandBuffer::GenerateMixCommand(int inputBufferIndex, int outputBufferIndex, float mixVolume, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_Mix)
        .Invoke(&ICommandBuffer::GenerateMixCommand, inputBufferIndex, outputBufferIndex, mixVolume, nodeId);
}

MixRampCommand* PerformanceMeasuredCommandBuffer::GenerateMixRampCommand(int inputBufferIndex, int outputBufferIndex, float mixVolume0, float mixVolume1, DspAddr voiceStateDspAddr, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_MixRamp)
        .Invoke(&ICommandBuffer::GenerateMixRampCommand, inputBufferIndex, outputBufferIndex, mixVolume0, mixVolume1, voiceStateDspAddr, nodeId);
}

MixRampGroupedCommand* PerformanceMeasuredCommandBuffer::GenerateMixRampGroupedCommand(int mixBufferCount, int inputBufferIndex, int outputBufferIndexStart, const float* pMixVolume, const float* pMixVolumePrev, DspAddr lastSampleAddress, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_MixRampGrouped)
        .Invoke(&ICommandBuffer::GenerateMixRampGroupedCommand, mixBufferCount, inputBufferIndex, outputBufferIndexStart, pMixVolume, pMixVolumePrev, lastSampleAddress, nodeId);
}

DepopPrepareCommand* PerformanceMeasuredCommandBuffer::GenerateDepopPrepareCommand(VoiceState* state, int32_t* depopBuffer, int mixBufferCount, int mixBufferOffset, NodeId nodeId, bool commandEnabled) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_DepopPrepare)
        .Invoke(&ICommandBuffer::GenerateDepopPrepareCommand, state, depopBuffer, mixBufferCount, mixBufferOffset, nodeId, commandEnabled);
}

DepopForMixBuffersCommand* PerformanceMeasuredCommandBuffer::GenerateDepopForMixBuffersCommand(int32_t* depopBuffer, int mixBufferOffset, int mixBufferCount, NodeId nodeId, int32_t sampleRate) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_DepopForMixBuffers)
        .Invoke(&ICommandBuffer::GenerateDepopForMixBuffersCommand, depopBuffer, mixBufferOffset, mixBufferCount, nodeId, sampleRate);
}

ClearMixBufferCommand* PerformanceMeasuredCommandBuffer::GenerateClearMixBufferCommand(NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_ClearMixBuffer)
        .Invoke(&ICommandBuffer::GenerateClearMixBufferCommand, nodeId);
}

VolumeCommand* PerformanceMeasuredCommandBuffer::GenerateVolumeCommand(float volume, int mixBufferCount, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_Volume)
        .Invoke(&ICommandBuffer::GenerateVolumeCommand, volume, mixBufferCount, nodeId);
}

VolumeRampCommand* PerformanceMeasuredCommandBuffer::GenerateVolumeRampCommand(float volume0, float volume1, int mixBufferCount, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_VolumeRamp)
        .Invoke(&ICommandBuffer::GenerateVolumeRampCommand, volume0, volume1, mixBufferCount, nodeId);
}

DeviceSinkCommand* PerformanceMeasuredCommandBuffer::GenerateDeviceSinkCommand(int mixBufferOffset, const server::SinkInfoBase* pSinkInfo, uint32_t sessionId, DspAddr mixBufferAddr, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_DeviceSink)
        .Invoke(&ICommandBuffer::GenerateDeviceSinkCommand, mixBufferOffset, pSinkInfo, sessionId, mixBufferAddr, nodeId);
}

CircularBufferSinkCommand* PerformanceMeasuredCommandBuffer::GenerateCircularBufferSinkCommand(int mixBufferOffset, const server::SinkInfoBase* pSinkInfo, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_CircularBufferSink)
        .Invoke(&ICommandBuffer::GenerateCircularBufferSinkCommand, mixBufferOffset, pSinkInfo, nodeId);
}

DownMix6chTo2chCommand* PerformanceMeasuredCommandBuffer::GenerateDownMixCommand(int mixBufferOffset, const int8_t* input, const int8_t* output, const int32_t* downMixParam, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_DownMix6chTo2ch)
        .Invoke(&ICommandBuffer::GenerateDownMixCommand, mixBufferOffset, input, output, downMixParam, nodeId);
}

UpsampleCommand* PerformanceMeasuredCommandBuffer::GenerateUpsampleCommand(int mixBufferOffset, UpsamplerInfo* pInfo, int inputCount, const int8_t* input, int32_t buffeCount, int32_t sampleCount, int32_t sampleRate, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_Upsample)
        .Invoke(&ICommandBuffer::GenerateUpsampleCommand, mixBufferOffset, pInfo, inputCount, input, buffeCount, sampleCount, sampleRate, nodeId);
}

PerformanceCommand* PerformanceMeasuredCommandBuffer::GeneratePerformanceCommand(const PerformanceEntryAddresses* entry, const PerformanceCommandType type, NodeId nodeId) NN_NOEXCEPT
{
    return CreateAspect(nodeId, PerformanceSysDetailType_Performance)
        .Invoke(&ICommandBuffer::GeneratePerformanceCommand, entry, type, nodeId);
}

}}}  // namespace nn::audio::server
