﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <algorithm>
#include <nn/nn_Abort.h>
#include "audio_Mixer.h"

namespace nn {
namespace audio {

Mixer::Mixer()
    : m_ResampleState()
    , m_InFormat()
    , m_OutFormat()
    , m_DepopState(MixerDepopState_NeedRamp)
    , m_FrameRate(200)
    , m_NumAppendBuffers(0)
    , m_MaxInputBytes(0)
    , m_CanAppendBuffers(false)
    , m_IsFormatValid(false)
{
    memset(m_LastGoodSample, 0, sizeof(m_LastGoodSample));
    memset(m_DepopDelta, 0, sizeof(m_DepopDelta));
}

void Mixer::ResetCurrentBuffer() NN_NOEXCEPT
{
    if(m_NumAppendBuffers != 0)
    {
        m_UserBuffer[0].writePosition = 0;
    }
}

bool Mixer::IsInFormatSupported(const MixerFormat& inFormat) NN_NOEXCEPT
{
    if(inFormat.channelCount != 1 && inFormat.channelCount != 2)
    {
        return false;
    }
    if(inFormat.sampleFormat != SampleFormat_PcmInt16)
    {
        return false;
    }
    if(inFormat.sampleRate > MaxInputSampleRate)
    {
        return false;
    }
    return true;
}

bool Mixer::IsOutFormatSupported(const MixerFormat& outFormat) NN_NOEXCEPT
{
    if(outFormat.channelCount != 2)
    {
        return false;
    }
    if(outFormat.sampleFormat != SampleFormat_PcmInt16)
    {
        return false;
    }
    return true;
}

bool Mixer::SetFormat(const MixerFormat& inFormat, const MixerFormat& outFormat) NN_NOEXCEPT
{
    if(!IsInFormatSupported(inFormat) || !IsOutFormatSupported(outFormat))
    {
        return false;
    }

    m_InFormat = inFormat;
    m_OutFormat = outFormat;

    auto ratio = static_cast<uint32_t>((static_cast<uint64_t>(1 << nn::audio::NQRATIO) * outFormat.sampleRate) / inFormat.sampleRate);
    nn::audio::InitializeResampler(&m_ResampleState, ratio);
    m_IsFormatValid = true;
    m_CanAppendBuffers = true;

    auto inBytesPerSample = m_InFormat.channelCount * GetSampleByteSize(m_InFormat.sampleFormat);
    auto maxInputSamples = (m_InFormat.sampleRate / (1000 / MaxOutputFrameDurationMs)) + SrcPaddingSamples;
    m_MaxInputBytes = maxInputSamples * inBytesPerSample;
    return true;
}

void Mixer::SetFrameRate(int samplesPerSecond) NN_NOEXCEPT
{
    m_FrameRate = samplesPerSecond;
}

bool Mixer::AppendBuffer(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    if(!m_CanAppendBuffers || m_NumAppendBuffers == UserBufferMaxCount)
    {
        return false;
    }
    memset(buffer, 0, bufferSize);
    m_UserBuffer[m_NumAppendBuffers] = UserBuffer(buffer, bufferSize);
    ++m_NumAppendBuffers;
    return true;
}

void* Mixer::GetBufferAddress() NN_NOEXCEPT
{
    if(m_NumAppendBuffers == 0)
    {
        return nullptr;
    }
    auto& userBuffer = m_UserBuffer[0];
    return userBuffer.buffer;
}

void Mixer::ClearBuffers() NN_NOEXCEPT
{
    for(int i = 0; i < m_NumAppendBuffers; ++i)
    {
        memset(m_UserBuffer[i].buffer, 0, m_UserBuffer[i].size);
        m_UserBuffer[i] = UserBuffer(0, 0);
    }
    m_NumAppendBuffers = 0;
    m_CanAppendBuffers = false;
}

void* Mixer::GetUserBufferAddress() NN_NOEXCEPT
{
    auto& userBufferInfo = m_UserBuffer[0];
    auto userBuffer = reinterpret_cast<int16_t*>(userBufferInfo.buffer);
    if(userBuffer == nullptr || m_NumAppendBuffers == 0)
    {
        return nullptr;
    }
    userBuffer += userBufferInfo.writePosition / sizeof(int16_t);
    return reinterpret_cast<void*>(userBuffer);
}

int32_t Mixer::GetUserBufferSamplesRemaining() NN_NOEXCEPT
{
    auto& userBufferInfo = m_UserBuffer[0];
    auto bytesPerSample = nn::audio::GetSampleByteSize(m_OutFormat.sampleFormat) * m_OutFormat.channelCount;
    return (userBufferInfo.size - userBufferInfo.writePosition) / bytesPerSample;
}

void Mixer::UserBufferInfoUpdatePosition(int32_t samplesToCopy) NN_NOEXCEPT
{
    auto& userBufferInfo = m_UserBuffer[0];
    auto bytesPerSample = nn::audio::GetSampleByteSize(m_OutFormat.sampleFormat) * m_OutFormat.channelCount;
    userBufferInfo.writePosition += samplesToCopy * bytesPerSample;

    if(userBufferInfo.writePosition >= userBufferInfo.size)
    {
        for(int i = 1; i < m_NumAppendBuffers; ++i)
        {
            m_UserBuffer[i - 1] = m_UserBuffer[i];
            m_UserBuffer[i] = UserBuffer(0, 0);
        }
        --m_NumAppendBuffers;
    }
}

void Mixer::ReleaseBuffer() NN_NOEXCEPT
{
    if(m_NumAppendBuffers)
    {
        for(int i = 1; i < m_NumAppendBuffers; ++i)
        {
            m_UserBuffer[i - 1] = m_UserBuffer[i];
        }
        --m_NumAppendBuffers;
    }
}

void Mixer::Depop() NN_NOEXCEPT
{
    auto numOutSamples = m_OutFormat.sampleRate / m_FrameRate;
    auto samplesNeededForDepop = 0;

    m_DepopState = MixerDepopState_Depopping;
    for(int ch = 0; ch < m_OutFormat.channelCount; ++ch)
    {
        m_DepopDelta[ch] = m_LastGoodSample[ch] / numOutSamples;
        samplesNeededForDepop = std::max(numOutSamples, samplesNeededForDepop);
    }

    while(numOutSamples && m_NumAppendBuffers > 0)
    {
        auto userBuffer = GetUserBufferAddress();
        auto userSamplesRemaining = GetUserBufferSamplesRemaining();
        auto samplesToCopy = std::min(userSamplesRemaining, numOutSamples);

        int16_t* buffer = reinterpret_cast<int16_t*>(userBuffer);
        if(m_DepopState == MixerDepopState_Depopping)
        {

            auto depopSamples = std::min(samplesToCopy, samplesNeededForDepop);
            for(int i = 0; i < depopSamples; ++i)
            {
                for(int ch = 0; ch < m_OutFormat.channelCount; ++ch)
                {
                    m_LastGoodSample[ch] -= m_DepopDelta[ch];
                    *buffer++ = m_LastGoodSample[ch];
                }
            }
            for(int i = 0; i < samplesToCopy - depopSamples; ++i)
            {
                for(int ch = 0; ch < m_OutFormat.channelCount; ++ch)
                {
                    *buffer++ = 0;
                }
            }

            if(depopSamples >= samplesNeededForDepop)
            {
                for(int ch = 0; ch < m_OutFormat.channelCount; ++ch)
                {
                    m_LastGoodSample[ch] = 0;
                }
                m_DepopState = MixerDepopState_NeedRamp;
            }
        }
        else
        {
            //Depop finished, but still getting bad data. Write 0s
            for(int i = 0; i < samplesToCopy; ++i)
            {
                for(int ch = 0; ch < m_OutFormat.channelCount; ++ch)
                {
                    *buffer++ = 0;
                }
            }
        }

        numOutSamples -= samplesToCopy;
        UserBufferInfoUpdatePosition(samplesToCopy);
    }
}

int Mixer::MixToOutBuffer(void* pOutBuffer, void* pInBuffer, int numOutBytes, int numInBytes, float volume) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_NumAppendBuffers == 0);
    AppendBuffer(pOutBuffer, numOutBytes);
    Mix(pInBuffer, numInBytes, volume);
    ReleaseBuffer();
    return m_UserBuffer[0].writePosition;
}

void Mixer::Mix(const void* pInBuffer, int32_t numInBytes, float volume) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_MaxInputBytes >= numInBytes, "Mix input bytes too large (Did UacMaxSamplesPerMs change?): MaxBytes: %d, inBytes: %d\n", m_MaxInputBytes, numInBytes);
    if(volume == 0.0f)
    {
        Depop();
        return;
    }

    auto inBytesPerSample = m_InFormat.channelCount * nn::audio::GetSampleByteSize(m_InFormat.sampleFormat);
    int32_t numInSamples = numInBytes / inBytesPerSample;

    const int16_t* inBuffer = reinterpret_cast<const int16_t*>(pInBuffer);
    int16_t srcBuffer [SrcMaxSampleCount];
    if(m_InFormat.sampleRate != m_OutFormat.sampleRate || volume != 1.0f)
    {
        memset(srcBuffer, 0, sizeof(srcBuffer));
        auto numSrcSamples = ResampleOutputGetNextFrameSize(&m_ResampleState, 1, numInSamples);
        NN_ABORT_UNLESS(SrcMaxSampleCount >= numSrcSamples, "Src samples too large. Max samples: %d, src samples: %d\n", SrcMaxSampleCount, numSrcSamples);
        nn::audio::ResampleOutputFrame(&m_ResampleState, 1, 0, inBuffer, srcBuffer, numInSamples, numSrcSamples, volume);

        inBuffer = srcBuffer;
        numInSamples = numSrcSamples;
    }

    while(numInSamples && m_NumAppendBuffers)
    {
        auto userBuffer = GetUserBufferAddress();
        auto userSamplesRemaining = GetUserBufferSamplesRemaining();
        int32_t samplesToCopy = std::min(userSamplesRemaining, numInSamples);

        if(m_InFormat.channelCount == 1)
        {
            switch(m_OutFormat.channelCount)
            {
                case 2:
                    MixMonoToStereo(inBuffer, userBuffer, samplesToCopy);
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
            }
        }
        else if(m_InFormat.channelCount == 2)
        {
            switch(m_OutFormat.channelCount)
            {
                case 2:
                    MixStereoToStereo(inBuffer, userBuffer, samplesToCopy);
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
            }
        }
        else if(m_InFormat.channelCount == 6)
        {
            switch(m_OutFormat.channelCount)
            {
                case 2:
                    MixSurroundToStereo(inBuffer, userBuffer, samplesToCopy);
                default:
                    NN_UNEXPECTED_DEFAULT;
            }
        }

        inBuffer += samplesToCopy;
        numInSamples -= samplesToCopy;
        UserBufferInfoUpdatePosition(samplesToCopy);
    }
}

void Mixer::MixMonoToStereo(const void* pInBuffer, void* pOutBuffer, int32_t numSamples) NN_NOEXCEPT
{
    int16_t* outBuffer = reinterpret_cast<int16_t*>(pOutBuffer);
    const int16_t* inBuffer = reinterpret_cast<const int16_t*>(pInBuffer);
    if(m_DepopState == MixerDepopState_Normal)
    {
        for(int i = 0; i < numSamples; ++i)
        {
            *outBuffer++ = *inBuffer;
            *outBuffer++ = *inBuffer++;
        }
        m_LastGoodSample[0] = outBuffer[-2];
        m_LastGoodSample[1] = outBuffer[-1];
    }
    else
    {
        auto volume = 0;
        auto volumeDelta = 0xFFFF / numSamples;
        for(int i = 0; i < numSamples; ++i)
        {
            int32_t sample = static_cast<int16_t>(*inBuffer++) * volume;
            int16_t sample16 = static_cast<int16_t>(sample / 0xFFFF);
            *outBuffer++ = sample16;
            *outBuffer++ = sample16;
            volume += volumeDelta;
        }
        m_LastGoodSample[0] = outBuffer[-2];
        m_LastGoodSample[1] = outBuffer[-1];
        m_DepopState = MixerDepopState_Normal;
    }
}

void Mixer::MixStereoToStereo(const void* pInBuffer, void* pOutBuffer, int32_t numSamples) NN_NOEXCEPT
{
    int16_t* outBuffer = reinterpret_cast<int16_t*>(pOutBuffer);
    const int16_t* inBuffer = reinterpret_cast<const int16_t*>(pInBuffer);
    if(m_DepopState == MixerDepopState_Normal)
    {
        for(int i = 0; i < numSamples; ++i)
        {
            *outBuffer++ = *inBuffer++;
            *outBuffer++ = *inBuffer++;
        }
        m_LastGoodSample[0] = outBuffer[-2];
        m_LastGoodSample[1] = outBuffer[-1];
    }
    else
    {
        auto volume = 0;
        auto volumeDelta = 0xFFFF / numSamples;
        for(int i = 0; i < numSamples; ++i)
        {
            for(int ch = 0; ch < 2; ++ch)
            {
                int32_t sample = static_cast<int16_t>(*inBuffer++) * volume;
                int16_t sample16 = static_cast<int16_t>(sample / 0xFFFF);
                *outBuffer++ = sample16;
            }
            volume += volumeDelta;
        }
        m_LastGoodSample[0] = outBuffer[-2];
        m_LastGoodSample[1] = outBuffer[-1];
        m_DepopState = MixerDepopState_Normal;
    }
}

void Mixer::MixSurroundToStereo(const void* pInBuffer, void* pOutBuffer, int32_t numSamples) NN_NOEXCEPT
{
    int16_t* outBuffer = reinterpret_cast<int16_t*>(pOutBuffer);
    const int16_t* inBuffer = reinterpret_cast<const int16_t*>(pInBuffer);
    if(m_DepopState == MixerDepopState_Normal)
    {
        for(int i = 0; i < numSamples; ++i)
        {
            *outBuffer++ = *inBuffer++;
            *outBuffer++ = *inBuffer++;
            inBuffer++;
            inBuffer++;
            inBuffer++;
            inBuffer++;
        }
        m_LastGoodSample[0] = outBuffer[-2];
        m_LastGoodSample[1] = outBuffer[-1];
    }
    else
    {
        auto volume = 0;
        auto volumeDelta = 0xFFFF / numSamples;
        for(int i = 0; i < numSamples; ++i)
        {
            for(int ch = 0; ch < 2; ++ch)
            {
                int32_t sample = static_cast<int16_t>(*inBuffer++) * volume;
                int16_t sample16 = static_cast<int16_t>(sample / 0xFFFF);
                *outBuffer++ = sample16;
            }
            inBuffer++;
            inBuffer++;
            inBuffer++;
            inBuffer++;
            volume += volumeDelta;
        }
        m_LastGoodSample[0] = outBuffer[-2];
        m_LastGoodSample[1] = outBuffer[-1];
        m_DepopState = MixerDepopState_Normal;
    }
}

}  // namespace audio
}  // namespace nn
