﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/audio/detail/audio_Log.h>
#include <nn/nn_SdkAssert.h>
#include <nn/audio/server/audio_FirmwareDebugSettings.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

namespace nn { namespace audio { namespace server {

namespace {

bool g_IsSettingsLoaded;
bool g_IsAdditionalMicCheckTriggerSuppressed;
bool g_IsAdspLogEnabled;
bool g_IsSupendForDebuggerEnabled;
bool g_IsUacEnabled;
bool g_HasCodecIc;
bool g_HasHda;

bool LoadBoolSetting(const char* name, const char* key, bool defaultValue) NN_NOEXCEPT
{
    bool value;

    if(nn::settings::fwdbg::GetSettingsItemValue(&value, sizeof(value), name, key) == sizeof(value))
    {
        return value;
    }
    else
    {
        NN_DETAIL_AUDIO_ERROR("[audio] Fail to read %s's value in firmware debug settings so returns default value:%d.\n", key, defaultValue);
        return defaultValue;
    }
}

bool LoadAudioBoolSetting(const char* key, bool defaultValue) NN_NOEXCEPT
{
    return LoadBoolSetting("audio", key, defaultValue);
}

bool LoadPlatformConfigBoolSetting(const char* key, bool defaultValue) NN_NOEXCEPT
{
    return LoadBoolSetting("platformconfig", key, defaultValue);
}

} // anonymous namespace

void LoadFirmwareDebugSettings() NN_NOEXCEPT
{
    g_IsAdditionalMicCheckTriggerSuppressed = LoadAudioBoolSetting("suppress_additional_mic_check", false);
    g_IsAdspLogEnabled = LoadAudioBoolSetting("adsp_log_enabled", false);
    g_IsSupendForDebuggerEnabled = LoadAudioBoolSetting("suspend_for_debugger_enabled", true);
    g_IsUacEnabled = LoadAudioBoolSetting("uac_enabled", true);
    g_HasCodecIc = LoadPlatformConfigBoolSetting("has_codec_ic", false);
    g_HasHda = LoadPlatformConfigBoolSetting("has_hda", false);

    g_IsSettingsLoaded = true;
}

void DumpFirmwareDebugSettings() NN_NOEXCEPT
{
    NN_DETAIL_AUDIO_INFO("SettingLoaded: %d AdditionalMixCheckSuppressed: %d AdspLogEnabled: %d SuspendForDebuggerEnabled:%d UacEnabled:%d HasCodecIc:%d HasHda:%d\n",
        g_IsSettingsLoaded,
        g_IsAdditionalMicCheckTriggerSuppressed,
        g_IsAdspLogEnabled,
        g_IsSupendForDebuggerEnabled,
        g_IsUacEnabled,
        g_HasCodecIc,
        g_HasHda
    );
}

bool IsAdditionalMicCheckTriggerSuppressed() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsSettingsLoaded);

    return g_IsAdditionalMicCheckTriggerSuppressed;
}

bool IsAdspLogEnabled() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsSettingsLoaded);

    return g_IsAdspLogEnabled;
}

bool IsSuspendForDebuggerEnabled() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsSettingsLoaded);

    return g_IsSupendForDebuggerEnabled;
}

// TODO: This feature does not work for now. (SIGLO-80785)
bool IsUacEnabled() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsSettingsLoaded);

    return g_IsUacEnabled;
}

bool HasCodecIc() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsSettingsLoaded);

    return g_HasCodecIc;
}

bool HasHda() NN_NOEXCEPT
{
    NN_SDK_ASSERT(g_IsSettingsLoaded);

    return g_HasHda;
}

}}} // namespace nn::audio::server
