﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/dd/dd_DeviceAddressSpace.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/audio/audio_AudioInTypes.h>
#include <nn/audio/audio_FinalOutputRecorderTypes.h>
#include <nn/audio/audio_ResultForPrivate.h>

#include "audio_FinalOutputRecorderManagerImpl.h"
#include "audio_SystemEventHolderPool.h"
#include "audio_ServiceDiagnostics.h"
#include "audio_AudioRecordSession.h"

namespace nn{
namespace audio{
namespace server{

namespace
{
    static const int g_NumberOfFinalOutputRecorderSessions = 2;
    nn::audio::server::SystemEventHolderPool<g_NumberOfFinalOutputRecorderSessions> g_SystemEventHolderPool;
}

class FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl
{
public:
    explicit FinalOutputRecorderImpl(FinalOutputRecorderManagerImpl* pParent, uint32_t sessionId, nn::os::EventType* pEvent) NN_NOEXCEPT
        : m_SystemEventHolder(g_SystemEventHolderPool.Acquire())
        , m_FinalOutputRecorderClass(sessionId, m_SystemEventHolder.GetSystemEvent())
        , m_Parent(pParent, true)
    {
        NN_UNUSED(pEvent);
        m_pParentMutex = m_Parent->GetMutex();
    }

    ~FinalOutputRecorderImpl() NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> lock(*m_pParentMutex);
        m_FinalOutputRecorderClass.Close();
        m_Parent->ReportCloseSession(m_FinalOutputRecorderClass.GetSessionId());
        g_SystemEventHolderPool.Release(m_SystemEventHolder);

        NN_AUDIO_SERVICE_LOG("[FinalOutputRecorder] Close sessionId(%d)\n", m_FinalOutputRecorderClass.GetSessionId());
    }

    nn::Result GetFinalOutputRecorderState(nn::sf::Out<uint32_t> outFinalOutputRecorder) NN_NOEXCEPT;
    nn::Result Start() NN_NOEXCEPT;
    nn::Result Stop() NN_NOEXCEPT;
    nn::Result Sleep() NN_NOEXCEPT;
    nn::Result Wake() NN_NOEXCEPT;
    nn::Result AppendFinalOutputRecorderBuffer(nn::sf::InBuffer inAudioBuffer, uint64_t outBufferClientPtr) NN_NOEXCEPT;
    nn::Result AppendFinalOutputRecorderBufferAuto(nn::sf::InBuffer inAudioBuffer, uint64_t outBufferClientPtr) NN_NOEXCEPT;
    nn::Result RegisterBufferEvent(nn::sf::Out<nn::sf::NativeHandle> bufferEvent) NN_NOEXCEPT;
    nn::Result GetReleasedFinalOutputRecorderBuffers(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count, nn::sf::Out<int64_t> released) NN_NOEXCEPT;
    nn::Result GetReleasedFinalOutputRecorderBuffersAuto(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count, nn::sf::Out<int64_t> released) NN_NOEXCEPT;
    nn::Result ContainsFinalOutputRecorderBuffer(uint64_t audioBufferPointer, nn::sf::Out<bool> contains) NN_NOEXCEPT;
    nn::Result GetFinalOutputRecorderBufferEndTime(uint64_t audioBufferPointer, nn::sf::Out<int64_t> released) NN_NOEXCEPT;
    nn::Result Initialize(nn::audio::FinalOutputRecorderParameter param, nn::sf::NativeHandle& processHandle, nn::sf::Out<nn::audio::detail::FinalOutputRecorderParameterInternal> finalOutputRecorderInter, const nn::applet::AppletResourceUserId& appletId) NN_NOEXCEPT;
    nn::Result Update() NN_NOEXCEPT;

private:
    nn::Result AppendFinalOutputRecorderBufferInternal(nn::sf::InBuffer inAudioBuffer, uint64_t outBufferClientPtr) NN_NOEXCEPT;
    nn::Result GetReleasedFinalOutputRecorderBuffersInternal(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count, nn::sf::Out<int64_t> released) NN_NOEXCEPT;

private:
    SystemEventHolder& m_SystemEventHolder;
    nn::audio::detail::AudioRecordSession m_FinalOutputRecorderClass;
    nn::sf::SharedPointer<FinalOutputRecorderManagerImpl> m_Parent;
    nn::os::Mutex* m_pParentMutex;
};

FinalOutputRecorderManagerImpl::FinalOutputRecorderManagerImpl() NN_NOEXCEPT
    : m_IsInitialized(false)
    , m_Mutex(true)
    , m_pSemaphore(nullptr)
    , m_IsAwake(true)
    , m_IsThreadActive(false)
    , m_Thread()
{
    m_IsAwake.store(true);
    m_HeapHandle = nn::lmem::CreateExpHeap(&m_HeapBuffer, sizeof(m_HeapBuffer), nn::lmem::CreationOption_ThreadSafe);
    m_Allocator.Attach(m_HeapHandle);

    m_pSemaphore = static_cast<os::SemaphoreType*>(m_Allocator.Allocate( sizeof (*m_pSemaphore) ));
    os::InitializeSemaphore(m_pSemaphore, 0, 1);
    os::InitializeEvent(&m_EventSemaphoreSignal, false, nn::os::EventClearMode_AutoClear);
    for(int i = 0; i < NumberOfFinalOutputRecorderSessions; ++i)
    {
        m_Session[i] = nullptr;
        m_AppletId[i] = nn::applet::AppletResourceUserId::GetInvalidId();
    }
}

FinalOutputRecorderManagerImpl::~FinalOutputRecorderManagerImpl() NN_NOEXCEPT
{
    if(m_IsInitialized)
    {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        m_IsThreadActive = false;
        nn::os::WaitThread( &m_Thread );
        nn::os::DestroyThread( &m_Thread );

        m_Allocator.Deallocate(m_Stack, SessionStackSize);
#endif
        nn::audio::server::FinalizeFinalOutputRecorder();
        m_IsInitialized = false;
    }

    os::FinalizeEvent(&m_EventSemaphoreSignal);
    os::FinalizeSemaphore(m_pSemaphore);
    m_Allocator.Deallocate(m_pSemaphore, sizeof(*m_pSemaphore));
    nn::lmem::DestroyExpHeap(m_HeapHandle);
}

Result FinalOutputRecorderManagerImpl::OpenFinalOutputRecorder(nn::sf::Out<nn::sf::SharedPointer<detail::IFinalOutputRecorder>> outFinalOutputRecorder, nn::audio::FinalOutputRecorderParameter param, nn::sf::NativeHandle processHandle, nn::sf::Out<nn::audio::detail::FinalOutputRecorderParameterInternal> finalOutputRecorderInter, nn::applet::AppletResourceUserId appletId) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    FirstTimeInitialize();

    int session=-1;
    nn::Result res = m_Sessions.GetNextSessions(&session);

    if( res.IsFailure() )
    {
        return res;
    }

    typedef nn::sf::ObjectFactory<MyAllocator::Policy> Factory;
    auto p = Factory::CreateSharedEmplaced<detail::IFinalOutputRecorder, FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl>(&m_Allocator, this, static_cast<uint32_t>(session), &(m_Event[session]));
    os::InitializeEvent( &(m_Event[session]), false, nn::os::EventClearMode_AutoClear );
    m_Session[session] = Factory::GetEmplacedImplPointer<FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl>(p);
    m_AppletId[session] = appletId;
    res = m_Session[session]->Initialize(param, processHandle, finalOutputRecorderInter, appletId);
    *outFinalOutputRecorder = std::move(p);

    NN_AUDIO_SERVICE_LOG(
            "[FinalOutputRecorder] Open sessionId(%d) aruid.lower(%llu) workBufferSize(%llu) parameter(%d %d)\n",
            session,
            param.sampleRate,
            param.channelCount);

    return res;
}

nn::Result FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl::GetFinalOutputRecorderState(nn::sf::Out<uint32_t> outFinalOutputRecorder) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    //TODO: Session state -> AudioOut state
    outFinalOutputRecorder.Set(m_FinalOutputRecorderClass.GetState());
    NN_RESULT_SUCCESS;
}

nn::Result FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl::Start() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    return m_FinalOutputRecorderClass.Start();
}

nn::Result FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl::Stop() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    m_FinalOutputRecorderClass.Stop();
    NN_RESULT_SUCCESS;
}

nn::Result FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl::Sleep() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    m_FinalOutputRecorderClass.Sleep();
    NN_RESULT_SUCCESS;
}

nn::Result FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl::Wake() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    m_FinalOutputRecorderClass.Wake();
    NN_RESULT_SUCCESS;
}

void FinalOutputRecorderManagerImpl::ReportCloseSession(int sessionId) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    m_Sessions.InsertSession(sessionId);
    os::FinalizeEvent(&(m_Event[sessionId]));
    m_Session[sessionId] = nullptr;
    m_AppletId[sessionId] = nn::applet::AppletResourceUserId::GetInvalidId();
}

void FinalOutputRecorderManagerImpl::FirstTimeInitialize()
{
    if(!m_IsInitialized)
    {
        nn::audio::server::InitializeFinalOutputRecorder(m_pSemaphore, &m_EventSemaphoreSignal);
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        m_Stack = static_cast<char*>(nn::lmem::AllocateFromExpHeap(m_HeapHandle, SessionStackSize, nn::os::ThreadStackAlignment));
        m_IsThreadActive = true;

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Thread, ThreadFunc, this, m_Stack, SessionStackSize, NN_SYSTEM_THREAD_PRIORITY(audio, FinalOutputRecorderManager)));
        nn::os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(audio, FinalOutputRecorderManager));
        nn::os::StartThread(&m_Thread);
#endif
        m_IsInitialized = true;
    }
}

nn::Result FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl::Initialize(nn::audio::FinalOutputRecorderParameter param, nn::sf::NativeHandle& processHandle, nn::sf::Out<nn::audio::detail::FinalOutputRecorderParameterInternal> outFinalOutputRecorderInter, const nn::applet::AppletResourceUserId& appletId) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    SessionFormat format;
    format.sampleRate = param.sampleRate;
    format.channelCount = static_cast<uint16_t>(param.channelCount);
    format.format = SampleFormat_PcmInt16;
    NN_RESULT_DO(m_FinalOutputRecorderClass.Open("", format, processHandle.GetOsHandle(), appletId));
    processHandle.Detach();
    outFinalOutputRecorderInter->sampleRate = m_FinalOutputRecorderClass.GetSampleRate();
    outFinalOutputRecorderInter->channelCount = m_FinalOutputRecorderClass.GetChannelCount();
    outFinalOutputRecorderInter->state = m_FinalOutputRecorderClass.GetState();
    outFinalOutputRecorderInter->sampleFormat = m_FinalOutputRecorderClass.GetSampleFormat();
    NN_RESULT_SUCCESS;
}

nn::Result FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl::Update() NN_NOEXCEPT
{
    return m_FinalOutputRecorderClass.Update();
}

nn::Result FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl::RegisterBufferEvent(nn::sf::Out<nn::sf::NativeHandle> bufferEvent) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    nn::os::NativeHandle handle;
    m_FinalOutputRecorderClass.RegisterBufferEvent(&handle);
    bufferEvent.Set(nn::sf::NativeHandle(handle, false));
    NN_RESULT_SUCCESS;
}

nn::Result FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl::AppendFinalOutputRecorderBufferInternal(nn::sf::InBuffer inAudioBuffer, uint64_t inBufferClientPtr) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    const FinalOutputRecorderBuffer* pTempBuffer = reinterpret_cast<const FinalOutputRecorderBuffer*>(inAudioBuffer.GetPointerUnsafe());
    auto couldAppend = m_FinalOutputRecorderClass.AppendBuffer(pTempBuffer->buffer, pTempBuffer->size, reinterpret_cast<const void*>(inBufferClientPtr));
    NN_RESULT_THROW_UNLESS(couldAppend, nn::audio::ResultBufferCountMax());
    NN_RESULT_SUCCESS;
}

nn::Result FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl::AppendFinalOutputRecorderBuffer(nn::sf::InBuffer inAudioBuffer, uint64_t inBufferClientPtr) NN_NOEXCEPT
{
    return AppendFinalOutputRecorderBufferInternal(inAudioBuffer, inBufferClientPtr);
}

nn::Result FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl::AppendFinalOutputRecorderBufferAuto(nn::sf::InBuffer inAudioBuffer, uint64_t inBufferClientPtr) NN_NOEXCEPT
{
    return AppendFinalOutputRecorderBufferInternal(inAudioBuffer, inBufferClientPtr);
}

nn::Result FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl::GetReleasedFinalOutputRecorderBuffersInternal(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count, nn::sf::Out<int64_t> released) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    uintptr_t* pAudioArray = reinterpret_cast<uintptr_t*>(outAudioBuffer.GetPointerUnsafe());
    pAudioArray[0] = 0;
    size_t length = outAudioBuffer.GetSize() / sizeof(*pAudioArray);

    int loops = 0;
    for(unsigned int i = 0; i < length; ++i)
    {
        nn::audio::FinalOutputRecorderBuffer* pAudioBuffer;
        auto sessionBuffer = m_FinalOutputRecorderClass.GetReleasedBuffer();

        if(!sessionBuffer)
        {
            break;
        }

        void* userAddr = const_cast<void*>(sessionBuffer->userAddr);
        pAudioBuffer = reinterpret_cast<nn::audio::FinalOutputRecorderBuffer*>(userAddr);

        if(pAudioBuffer != nullptr)
        {
            released.Set(sessionBuffer->completionTime.GetNanoSeconds());
            pAudioArray[i] = reinterpret_cast<uintptr_t>(pAudioBuffer);
            ++loops;
        }
        else
        {
            break;
        }
    }
    count.Set(loops);
    NN_RESULT_SUCCESS;
}

nn::Result FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl::GetReleasedFinalOutputRecorderBuffers(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count, nn::sf::Out<int64_t> released) NN_NOEXCEPT
{
    return GetReleasedFinalOutputRecorderBuffersInternal(outAudioBuffer, count, released);
}

nn::Result FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl::GetReleasedFinalOutputRecorderBuffersAuto(nn::sf::OutBuffer outAudioBuffer, nn::sf::Out<int> count, nn::sf::Out<int64_t> released) NN_NOEXCEPT
{
    return GetReleasedFinalOutputRecorderBuffersInternal(outAudioBuffer, count, released);
}

nn::Result FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl::ContainsFinalOutputRecorderBuffer(uint64_t audioBufferPointer, nn::sf::Out<bool> contains) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(*m_pParentMutex);
    bool found = m_FinalOutputRecorderClass.ContainsBuffer(reinterpret_cast<void*>(audioBufferPointer));
    contains.Set(found);
    NN_RESULT_SUCCESS;
}

nn::Result FinalOutputRecorderManagerImpl::FinalOutputRecorderImpl::GetFinalOutputRecorderBufferEndTime(uint64_t audioBufferPointer, nn::sf::Out<int64_t> released) NN_NOEXCEPT
{
    //Deprecated.  released time comes from shim
    NN_UNUSED(audioBufferPointer);
    released.Set(0);
    NN_RESULT_SUCCESS;
}

void FinalOutputRecorderManagerImpl::ThreadFunc(void* arg) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(arg);
    reinterpret_cast<FinalOutputRecorderManagerImpl*>(arg)->ThreadFuncImpl();
}

void FinalOutputRecorderManagerImpl::ThreadFuncImpl() NN_NOEXCEPT
{
    while(m_IsThreadActive)
    {
        if(nn::os::TimedAcquireSemaphore(m_pSemaphore, nn::TimeSpan::FromMilliSeconds(5)))
        {
            std::lock_guard<os::Mutex> lock(m_Mutex);
            for(int i = 0; i < NumberOfFinalOutputRecorderSessions; ++i)
            {
                if(m_Session[i] != nullptr)
                {
                    m_Session[i]->Update();
                }
            }
            nn::os::SignalEvent( &m_EventSemaphoreSignal );
        }
    }
}

nn::Result FinalOutputRecorderManagerImpl::Wake() NN_NOEXCEPT
{
    if( !m_IsAwake )
    {
        std::lock_guard<os::Mutex> lock(m_Mutex);
        nn::audio::server::InitializeFinalOutputRecorder(m_pSemaphore, &m_EventSemaphoreSignal);
        for( int i = 0; i < NumberOfFinalOutputRecorderSessions; ++i )
        {
            if( m_Session[i] != nullptr )
            {
                m_Session[i]->Wake();

                //SIGLO-77616:
                //GameRecord sessions do not signal semaphore if no buffers are appended
                //Force an update, to make sure buffer release event is signaled so apps can re-append buffers.
                m_Session[i]->Update();
            }
        }
        m_IsAwake = true;
    }
    NN_RESULT_SUCCESS;
}

nn::Result FinalOutputRecorderManagerImpl::Sleep() NN_NOEXCEPT
{
    if( m_IsAwake )
    {
        std::lock_guard<os::Mutex> lock(m_Mutex);
        for( int i = 0; i < NumberOfFinalOutputRecorderSessions; ++i )
        {
            if( m_Session[i] != nullptr )
            {
                m_Session[i]->Sleep();
            }
        }
        m_IsAwake = false;
    }
    NN_RESULT_SUCCESS;
}

nn::os::Mutex* FinalOutputRecorderManagerImpl::GetMutex() NN_NOEXCEPT
{
    return &m_Mutex;
}

nn::Result FinalOutputRecorderManagerImpl::GetAdspLoad(nn::sf::Out<int32_t> value) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_HARDWARE_NX)
    *value = static_cast<int32_t>(nne::audio::adsp::getAdspUsage());
#else  // defined(NN_BUILD_CONFIG_HARDWARE_NX)
    *value = 0;
#endif  // defined(NN_BUILD_CONFIG_HARDWARE_NX)
    NN_RESULT_SUCCESS;
}

}}}//namespace nn, audio, server
